<?php

if ( ! function_exists( 'talemy_get_llms_course_price' ) ) {
	/**
	 * Get course price
	 *
	 * @param object $product
	 * @return string
	 */
	function talemy_get_llms_course_price( &$product = null ) {
		if ( is_null( $product ) ) {
			$product = new LLMS_Product( get_the_ID() );
		}

		$purchaseable = $product->is_purchasable();
		$has_free     = $product->has_free_access_plan();
		$free_only    = ( $has_free && ! $purchaseable );
		$price_html   = '';

		if ( $free_only ) {
			$price_html .= esc_html__( 'FREE', 'talemy' );
		} else if ( $purchaseable ) {
			$plans = $product->get_access_plans( $free_only );

			if ( $plans ) {
				$index   = 0;
				$is_free = false;
				foreach ( $plans as $plan ) {
					$price_key = $plan->is_on_sale() ? 'sale_price' : 'price';
					$price     = $plan->get_price( $price_key, [], 'float' );

					if ( 0.0 === $price ) {
						$min_price = $plan->get_free_pricing_text( 'html' );
						$min_key   = $index;
						$is_free   = true;
						break;
					}
					if ( ! isset( $min_price ) ) {
						$min_price = $price;
						$min_key   = $index;
					}
					if ( $price < $min_price ) {
						$min_price = $price;
						$min_key   = $index;
					}
					$index ++;
				}
				if ( isset( $min_price ) ) {
					if ( $is_free ) {
						$price_html .= $min_price;
					} else {
						if ( count( $plans ) > 1 ) {
							$price_html .= '<span class="price-prefix">' . esc_html_x( 'Stars from', 'starter price', 'talemy' ) . '</span>';
						}
						$price_html .= llms_price( $min_price, [] );
						$schedule = $plans[ $min_key ]->get_schedule_details();
						if ( ! empty( $schedule ) ) {
							$price_html .= sprintf( '<span class="price-schedule">%s</span>', $schedule );
						}
					}
				}
			}
		}

		return $price_html;
	}
}

if ( ! function_exists( 'talemy_get_llms_loop_course_meta' ) ) {
	/**
	 * Get course meta
	 *
	 * @param object $course
	 * @param array $meta
	 * @return string
	 */
	function talemy_get_llms_loop_course_meta( &$course = null, $meta = [] ) {
		global $post;

		if ( empty( $meta ) ) {
			return '';
		}

		if ( empty( $course ) ) {
			$course = new LLMS_Course( $post );
		}

		$out = '';

		if ( in_array( 'duration', $meta ) ) {
			$duration = $course->get( 'length' );
			if ( ! empty( $duration ) ) {
				$out .= '<li><i class="far fa-clock"></i>' . esc_html( $duration ) . '</li>';
			}
		}

		if ( in_array( 'level', $meta ) ) {
			$difficulty = $course->get_difficulty();
			if ( ! empty( $difficulty ) ) {
				$out .= '<li><i class="fas fa-signal"></i>' . esc_html( $difficulty ) . '</li>';
			}
		}

		if ( in_array( 'enrolled', $meta ) ) {
			$student_count = $course->get_student_count();
			if ( $student_count > 0 ) {
				$out .= '<li><i class="fas fa-users"></i>' . esc_html( $student_count ) . '</li>';
			}
		}

		if ( ! empty( $out ) ) {
			$out = '<ul class="post-meta">' . $out . '</ul>';
		}

		return $out;
	}
}

if ( ! function_exists( 'talemy_get_llms_loop_thumb' ) ) {
	/**
	 * Get loop thumbnail
	 *
	 * @param  string size thumbnail size
	 * @param  string html
	 * @return string html
	 */
	function talemy_get_llms_loop_thumb( &$course, $size = 'thumbnail', $html = '' ) {
		global $post, $talemy_enqueued_fancybox;

		if ( empty( $course ) ) {
			$course = new LLMS_Course( $post );
		}

		$has_thumbnail = ( has_post_thumbnail() );
		$out           = '';
		$has_video     = false;

		if ( $has_thumbnail || talemy_get_option( 'loop_thumb_placeholder' ) ) {
			$out .= '<div class="post-thumb">';

			if ( talemy_get_option( 'ld_courses_video_preview' ) ) {
				$embed_code = $course->get( 'video_embed' );
				$has_video  = ! empty( $embed_code ) && preg_match( '/http(s?)\:\/\//i', $embed_code ) ? true : false;
			}

			if ( $has_video ) {
				$out .= '<a data-fancybox data-width="1280" data-height="720" href="' . esc_url( $embed_code ) . '">';
				$out .= '<span class="video-play-button"><i class="icon-triangle"></i></span>';

				if ( ! $talemy_enqueued_fancybox ) {
					if ( ! wp_script_is( 'queue', 'fancybox' ) ) {
						wp_enqueue_script( 'fancybox' );
					}

					if ( ! wp_style_is( 'queue', 'fancybox' ) ) {
						wp_enqueue_style( 'fancybox' );
					}
					$talemy_enqueued_fancybox = true;
				}

			} else {
				$out .= '<a href="' . esc_url( get_permalink() ) . '">';
				$out .= '<span class="post-thumb-btn">' . talemy_get_option( 'ld_thumb_hover_text' ) . '</span>';
			}

			if ( $has_thumbnail ) {
				$out .= get_the_post_thumbnail( get_the_ID(), $size );
			} elseif ( talemy_get_option( 'loop_thumb_placeholder' ) ) {
				$out .= '<img src="' . esc_url( TALEMY_THEME_URI . 'assets/images/thumbs/talemy_thumb_small.png' ) . '" alt="thumbnail">';
			}

			$out .= '</a>';
			$out .= $html;
			$out .= '</div>';
		}

		return $out;
	}
}

if ( ! function_exists( 'talemy_get_llms_loop_badge' ) ) {
	/**
	 * Get loop badge
	 *
	 * @return void
	 */
	function talemy_get_llms_loop_badge() {
		global $post;
		$badge_class = $badge_text = $out = '';

		if ( is_user_logged_in() ) {
			$student = llms_get_student( get_current_user_id() );
			if ( $student ) {
				$enrollment_status = $student->get_enrollment_status( $post->ID );
				$completed         = $student->get_progress( $post->ID, 'course' );

				if ( 100 == $completed ) {
					$badge_class = 'post-badge completed';
					$badge_text  = esc_html__( 'Completed', 'talemy' );
				} else if ( 0 < $completed ) {
					$badge_class = 'post-badge inprogress';
					$badge_text  = esc_html__( 'In Progress', 'talemy' );
				} else if ( 'enrolled' === $enrollment_status ) {
					$badge_class = 'post-badge enrolled';
					$badge_text  = llms_get_enrollment_status_name( $enrollment_status );
				}
			}
		}

		if ( ! empty( $badge_text ) ) {
			$out .= '<div class="' . esc_attr( $badge_class ) . '">' . esc_html( $badge_text ) . '</div>';
		}

		return $out;
	}
}

if ( ! function_exists( 'talemy_get_llms_access_plan_button' ) ) {
	/**
	 * Get loop progress
	 *
	 * @return void
	 */
	function talemy_get_llms_access_plan_button( &$product = null ) {
		if ( is_null( $product ) ) {
			$product = new LLMS_Product( get_the_ID() );
		}

		$purchaseable = $product->is_purchasable();
		$has_free     = $product->has_free_access_plan();
		$free_only    = ( $has_free && ! $purchaseable );
		$out   		  = '';
	
		if ( $free_only ) {

		} else if ( $purchaseable ) {
			$plans = $product->get_access_plans( $free_only );
			$plan_count = count( $plans );
			
			if ( $plan_count === 1 ) {
				$plan = $plans[0];
				$out .= '<a class="btn btn-primary btn-lg btn-block to-llms-access-plans" href="' . esc_url( $plan->get_checkout_url() ) . '">' . $plan->get_enroll_text() . '</a>';
			} else if ( $plan_count > 1 ) {
				$out .= '<a class="btn btn-primary btn-lg btn-block to-llms-access-plans" href="' . esc_url( get_the_permalink() ) . '">' . esc_html__( 'See Plans', 'talemy' ) . '</a>';
			}
		}
		return $out;
	}
}

if ( ! function_exists( 'talemy_get_llms_loop_course_progress' ) ) {
	/**
	 * Get loop progress
	 *
	 * @return void
	 */
	function talemy_get_llms_loop_course_progress() {
		global $post;
		$out = '';

		if ( is_user_logged_in() ) {
			$student = llms_get_student( get_current_user_id() );
			if ( $student ) {
				$enrollment_status = $student->get_enrollment_status( $post->ID );
				$progress          = $student->get_progress( $post->ID, 'course' );
			}
		}

		if ( ! empty( $progress ) ) {
			$out .= '<div class="course-progress">' . esc_html( $progress ) . '</div>';
		}

		return $out;
	}
}

if ( ! function_exists( 'talemy_get_llms_course_categories' ) ) {
    /**
     * Get course category links
     * Use it within the loop
     *
     * @return html category links
     */
    function talemy_get_llms_course_categories( $separator = ', ' ) {
		$cats  = get_the_terms( get_the_ID(), 'course_cat' );
		$links = [];
		if ( ! is_wp_error( $cats ) && ! empty( $cats ) ) {
			foreach ( $cats as $cat ) {
				$links[] = sprintf( '<a href="%s">%s</a>', get_term_link( $cat->term_id, 'course_cat' ), $cat->name );
			}
		}
		return ! empty( $links ) ? '<span class="course-categories">' . implode( $separator, $links ) . '</span>' : '';
    }
}

if ( ! function_exists( 'talemy_get_llms_author_box' ) ) {
    /**
     * Get course author
     *
     * @return html category links
     */
    function talemy_get_llms_author_box( $args ) {
		$defaults = [
			'avatar'      => true,
			'avatar_size' => 100,
			'label'       => '',
			'user_id'     => get_the_author_meta( 'ID' ),
		];
		$args = wp_parse_args( $args, $defaults );
		$author_id    = $args['user_id'];
		$author_email = get_the_author_meta( 'email', $author_id );
		$author_url   = get_the_author_meta( 'url', $author_id );
		$author_desc  = get_the_author_meta( 'description', $author_id );
		?>
		<div class="author-box">
			<a class="author-portray" itemprop="author" href="<?php echo get_author_posts_url( $author_id ); ?>"><?php echo get_avatar( $author_email, $args['avatar_size'] ); ?></a>
			<div class="author-description">
				<a class="author-name" itemprop="author" href="<?php echo get_author_posts_url( $author_id ); ?>"><?php echo get_the_author_meta( 'display_name', $author_id ); ?></a>

				<?php if ( ! empty( $args['label'] ) ) : ?>
					<div class="author-title"><?php echo esc_html( $args['label'] ); ?></div>
				<?php endif; ?>

				<?php if ( ! empty( $author_desc ) ) : ?>
					<div class="author-bio"><?php echo esc_html( $author_desc ); ?></div>
				<?php endif; ?>

				<?php if ( ! empty( $author_url ) ) : ?>
					<div class="author-url"><a href="<?php echo esc_url( $author_url ); ?>" target="_blank"><?php echo esc_url( $author_url ); ?></a></div>
				<?php endif; ?>

				<?php do_action( 'sf_author_social_links', $author_id ); ?>
			</div>
		</div>
		<?php
    }
}