<?php

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

if ( ! function_exists( 'talemy_get_ld_course_meta' ) ) {
    /**
     * Get course meta
     *
     * @param  array $key meta key
     * @return string
     */
    function talemy_get_ld_course_meta( $key = '', $course_id = null ) {
        if ( empty( $course_id ) ) {
            $course_id = get_the_ID();
        }

        if ( empty( $key ) ) {
            return get_post_meta( $course_id, '_ld_custom_meta', true );
        }

        $meta_data = get_post_meta( $course_id, '_ld_custom_meta', true );

        return isset( $meta_data[$key] ) ? $meta_data[$key] : '';
    }
}

if ( ! function_exists( 'talemy_get_ld_loop_course_meta' ) ) {
    /**
     * Get course meta html
     *
     * @param  array $meta            meta to show
     * @param  array  $data           meta data
     * @return string
     */
    function talemy_get_ld_loop_course_meta( $meta, $data = array() ) {
        if ( empty( $meta ) ) {
            return '';
        }

        $out = '';

        if ( empty( $data ) ) {
            $data = get_post_meta( get_the_ID(), '_ld_custom_meta', true );
        }

        $meta = apply_filters( 'talemy_loop_course_meta', $meta );
        $data = apply_filters( 'talemy_loop_course_meta_data', $data );

        if ( in_array( 'duration', $meta ) && ! empty( $data['duration'] ) ) {
            $out .= '<li><i class="far fa-clock"></i>' . esc_html( $data['duration'] ) . '</li>';
        }

        if ( in_array( 'level', $meta ) && ! empty( $data['level'] ) ) {
            if ( function_exists( 'ldcm_get_course_level_label' ) ) {
                $level = ldcm_get_course_level_label( $data['level'] );
            } else {
                $level = $data['level'];
            }
            $out .= '<li><i class="fas fa-signal"></i>' . esc_html( $level ) . '</li>';
        }

        if ( in_array( 'language', $meta ) && ! empty( $data['language'] ) ) {
            $out .= '<li><i class="fas fa-globe"></i>' . esc_html( $data['language'] ) . '</li>';
        }

        if ( in_array( 'enrolled', $meta ) ) {
            $enrolled = ! empty( $data['enrolled'] ) ? $data['enrolled'] : 0;
            switch ( get_post_type( get_the_ID() ) ) {
            case 'sfwd-courses':
                $count = talemy_get_ld_course_student_count( get_the_ID(), $enrolled );
                break;
            case 'groups':
                $count = talemy_get_ld_group_student_count( get_the_ID(), $enrolled );
                break;
            }
            $out .= '<li><i class="fas fa-users"></i>' . $count . '</li>';
        }

        /** Add custom meta items here */
        $html = apply_filters( 'talemy_loop_course_meta_items', $out );

        if ( ! empty( $html ) ) {
            $html = '<ul class="post-meta">' . $html . '</ul>';
        }

        return $html;
    }
}

if ( ! function_exists( 'talemy_get_ld_loop_author' ) ) {
    /**
     * Get loop author
     *
     * @param  bool show_avatar true/false
     * @return string html
     */
    function talemy_get_ld_loop_author( $show_avatar = false ) {
        $author_id = get_the_author_meta( 'ID' );
        $out       = '';
        $out .= '<span class="ld-course-author">';
        if ( $show_avatar ) {
            $out .= '<a class="author-avatar" href="' . esc_url( get_author_posts_url( $author_id ) ) . '">' . get_avatar( $author_id, 30 ) . '</a>';
            $out .= '<a href="' . esc_url( get_author_posts_url( $author_id, get_the_author_meta( 'user_nicename' ) ) ) . '">' . get_the_author() . '</a>';
        } else {
            $out .= sprintf( esc_html__( 'by %s', 'talemy' ), '<a href="' . esc_url( get_author_posts_url( $author_id, get_the_author_meta( 'user_nicename' ) ) ) . '">' . get_the_author() . '</a>' );
        }
        $out .= '</span>';
        return $out;
    }
}

if ( ! function_exists( 'talemy_get_ld_loop_thumb' ) ) {
    /**
     * Get loop thumbnail
     *
     * @param  string size thumbnail size
     * @param  string html
     * @return string html
     */
    function talemy_get_ld_loop_thumb( $size = 'thumbnail', $html = '' ) {
        global $talemy_enqueued_fancybox;
        $has_thumbnail = ( has_post_thumbnail() );
        $out           = '';
        $has_video     = false;

        if ( $has_thumbnail || talemy_get_option( 'loop_thumb_placeholder' ) ) {
            $out .= '<div class="post-thumb">';

            if ( talemy_get_option( 'ld_courses_video_preview' ) ) {
                $embed_code = talemy_get_ld_course_meta( 'embed_code' );
                $has_video  = ! empty( $embed_code ) && preg_match( '/http(s?)\:\/\//i', $embed_code ) ? true : false;
            }

            if ( $has_video ) {
                $out .= '<a data-fancybox data-width="1280" data-height="720" href="' . esc_url( $embed_code ) . '">';
                $out .= '<span class="video-play-button"><i class="icon-triangle"></i></span>';

                if ( ! $talemy_enqueued_fancybox ) {
                    if ( ! wp_script_is( 'queue', 'fancybox' ) ) {
                        wp_enqueue_script( 'fancybox' );
                    }

                    if ( ! wp_style_is( 'queue', 'fancybox' ) ) {
                        wp_enqueue_style( 'fancybox' );
                    }
                    $talemy_enqueued_fancybox = true;
                }

            } else {
                $out .= '<a href="' . esc_url( get_permalink() ) . '">';
                $out .= '<span class="post-thumb-btn">' . talemy_get_option( 'ld_thumb_hover_text' ) . '</span>';
            }

            if ( $has_thumbnail ) {
                $out .= get_the_post_thumbnail( get_the_ID(), $size );
            } elseif ( talemy_get_option( 'loop_thumb_placeholder' ) ) {
                $out .= '<img src="' . esc_url( TALEMY_THEME_URI . 'assets/images/thumbs/talemy_thumb_small.png' ) . '" alt="thumbnail">';
            }

            $out .= '</a>';
            $out .= $html;
            $out .= '</div>';
        }

        return $out;
    }
}

/**
 * Get course/group loop price
 *
 * @param int $post_id
 * @return void
 */
function talemy_ld_loop_price( $post_id = null ) {

    if ( empty( $post_id ) ) {
		$post_id = get_the_ID();
	}

	if ( empty( $post_id ) ) {
		return;
	}

	$post_type  = get_post_type( $post_id );
	$user_id    = get_current_user_id();
	$has_access = false;
	$price_type = '';

	if ( 'sfwd-courses' === $post_type ) {
		$has_access = sfwd_lms_has_access( $post_id, $user_id );
		$price_type = strtolower( learndash_get_course_meta_setting( $post_id, 'course_price_type' ) );
	} else if ( 'groups' === $post_type ) {
		if ( learndash_is_user_in_group( $user_id, $post_id ) ) {
			$has_access = true;
		} else {
			$group_pricing = learndash_get_group_price( $post_id );
			$price_type    = strtolower( $group_pricing['type'] );
		}
	}

	if ( ! $has_access ) {
		$related_product = talemy_get_ld_course_meta( 'related_product' );
		$is_wc_product   = 'closed' === $price_type && defined( 'WC_PLUGIN_FILE' ) && defined( 'LEARNDASH_WOOCOMMERCE_FILE' ) && ! empty( $related_product );

		if ( $is_wc_product ) {
			echo talemy_get_wc_product_price( $related_product );
		} else {
			if ( 'sfwd-courses' === $post_type ) {
				echo talemy_get_ld_course_price( $post_id, true );
			} else if ( 'groups' === $post_type ) {
				echo talemy_get_ld_group_price( $post_id, true );
			}
		}
	}
}

if ( ! function_exists( 'talemy_get_ld_course_categories' ) ) {
    /**
     * Get course category links
     * Use it within the loop
     *
     * @return html category links
     */
    function talemy_get_ld_course_categories( $separator = ', ' ) {
        $cats  = get_the_terms( get_the_ID(), 'ld_course_category' );
        $links = array();
        if ( ! is_wp_error( $cats ) && ! empty( $cats ) ) {
            foreach ( $cats as $cat ) {
                $links[] = sprintf( '<a href="%s">%s</a>', get_term_link( $cat->term_id, 'ld_course_category' ), $cat->name );
            }
        }
        return ! empty( $links ) ? '<span class="course-categories">' . implode( $separator, $links ) . '</span>' : '';
    }
}

if ( ! function_exists( 'talemy_get_ld_group_categories' ) ) {
    /**
     * Get group category links
     * Use it within the loop
     *
     * @return html category links
     */
    function talemy_get_ld_group_categories( $separator = ', ' ) {
        $cats  = get_the_terms( get_the_ID(), 'ld_group_category' );
        $links = array();
        if ( ! is_wp_error( $cats ) && ! empty( $cats ) ) {
            foreach ( $cats as $cat ) {
                $links[] = sprintf( '<a href="%s">%s</a>', get_term_link( $cat->term_id, 'ld_group_category' ), $cat->name );
            }
        }
        return ! empty( $links ) ? '<span class="course-categories">' . implode( $separator, $links ) . '</span>' : '';
    }
}

/**
 * Gets the price formatted based on the LearnDash global currency configuration.
 * 
 * Modifies learndash_get_price_formatted function.
 *
 * @param mixed $price price to format
 * @return string Returns price formatted.
 */
function talemy_get_ld_price_formatted( $price ) {
	// Empty prices should not be displayed.
	if ( '' === $price ) {
		return '';
	}

	$currency_code = ! empty( $currency_code ) ? $currency_code : learndash_get_currency_code();

	// Price is shown as is if no currency set.
	if ( empty( $currency_code ) ) {
		return $price;
	}

	// Price is shown as is if non-numeric.
	if ( ! is_numeric( $price ) ) {
		return $price;
	}

	if ( class_exists( 'NumberFormatter' ) ) {
		$number_format = new NumberFormatter(
			get_locale() . '@currency=' . $currency_code,
			NumberFormatter::CURRENCY
		);
		$decimal_separator = $number_format->getSymbol( NumberFormatter::DECIMAL_SEPARATOR_SYMBOL );
		$price             = $number_format->format( floatval( $price ) );
		$price             = rtrim( $price, '0' );
		$price             = rtrim( $price, $decimal_separator );
		return $price;
	}

	$currency_symbol = learndash_get_currency_symbol( $currency_code );

	return strlen( $currency_symbol ) > 1
		? "$price $currency_symbol" // it's currency code: we should display at the end of the price.
		: "$currency_symbol{$price}"; // show the currency symbol at the beginning of the price (en_US style).
}

/**
 * Get course price
 *
 * @param  int    $course_id
 * @param  bool   $short_form
 * @return string
 */
function talemy_get_ld_course_price( $course_id = false, $short_form = false ) {
	if ( empty( $course_id ) ) {
		$course_id = get_the_ID();
	}

	if ( empty( $course_id ) ) {
		return '';
	}

	$course_pricing = learndash_get_course_price( $course_id );
	// Some simple price settings validation logic. Not 100%.
	$course_pricing = wp_parse_args(
		$course_pricing,
		array(
			'type'             => LEARNDASH_DEFAULT_COURSE_PRICE_TYPE,
			'price'            => '',
			'interval'         => '',
			'frequency'        => '',
			'trial_price'      => '',
			'trial_interval'   => '',
			'trial_frequency'  => '',
			'repeats'          => '',
			'repeat_frequency' => '',
		)
	);

	$price_type = strtolower( $course_pricing['type'] );
	$price_html = '';

	if ( 'subscribe' === $price_type ) {
		if ( ( empty( $course_pricing['price'] ) ) || ( empty( $course_pricing['interval'] ) ) || ( empty( $course_pricing['frequency'] ) ) ) {
			$course_pricing['type']             = LEARNDASH_DEFAULT_COURSE_PRICE_TYPE;
			$course_pricing['interval']         = '';
			$course_pricing['frequency']        = '';
			$course_pricing['trial_price']      = '';
			$course_pricing['trial_interval']   = '';
			$course_pricing['trial_frequency']  = '';
			$course_pricing['repeats']          = '';
			$course_pricing['repeat_frequency'] = '';
		} else {
			if ( empty( $course_pricing['trial_price'] ) ) {
				$course_pricing['trial_price'] = 0;
			} elseif ( ( empty( $course_pricing['trial_interval'] ) ) || ( empty( $course_pricing['trial_frequency'] ) ) ) {
				$course_pricing['trial_price'] = '';
			}
		}
	}

	if ( 'free' == $price_type || '' === $course_pricing['price'] ) {
		$label = apply_filters( 'learndash_no_price_price_label', ( 'closed' === $price_type ? esc_html__( 'Closed', 'talemy' ) : esc_html__( 'Free', 'talemy' ) ) );
		$price_html .= '<div class="course-price"><span class="price-amount">' . $label . '</span></div>';

	} else if ( ! empty( $course_pricing['price'] ) ) {

		if ( 'subscribe' !== $price_type ) {
			
			$price_html .= '<div class="course-price">';
			$price_html .= '<span class="price-amount">' . talemy_get_ld_price_formatted( $course_pricing['price'] ) . '</span>';
			$price_html .= '</div>';

		} else {

			if ( ! empty( $course_pricing['trial_frequency'] ) && ! $short_form ) {
				$price_html .= '<div class="course-trial-price recurring">';
				$price_html .= '<span class="price-amount">' . talemy_get_ld_price_formatted( $course_pricing['trial_price'] ) . '</span>';

				if ( ( ! empty( $course_pricing['trial_interval'] ) ) && ( ! empty( $course_pricing['trial_frequency'] ) ) ) {
					$price_html .= '<span class="price-text">';
					$price_html .= sprintf(
						// translators: placeholders: Trial interval, Trial frequency.
						esc_html_x( 'Trial price for %1$s %2$s, then', 'placeholders: Trial interval, Trial frequency', 'talemy' ),
						absint( $course_pricing['trial_interval'] ),
						esc_html( $course_pricing['trial_frequency'] )
					);
					$price_html .= '</span>';
				}
				$price_html .= '</div>';

				if ( ( ! empty( $course_pricing['interval'] ) ) && ( ! empty( $course_pricing['frequency'] ) ) ) {
					$price_html .= '<div class="course-price recurring">';
					$price_html .= '<span class="price-amount">' . talemy_get_ld_price_formatted( $course_pricing['price'] ) . '</span>';

					$price_html .= '<span class="price-text">';
					$price_html .= sprintf(
						// translators: placeholders: %1$s Interval of recurring payments (number), %2$s Frequency of recurring payments: day, week, month or year.
						esc_html_x( 'Full price every %1$s %2$s afterward', 'Recurring duration message', 'talemy' ),
						absint( $course_pricing['interval'] ),
						esc_html( $course_pricing['frequency'] )
					);

					if ( ! empty( $course_pricing['repeats'] ) ) {
						$price_html .= ' ';
						$price_html .= sprintf(
							// translators: placeholders: %1$s Number of times the recurring payment repeats, %2$s Frequency of recurring payments: day, week, month, year.
							esc_html__( 'for %1$s %2$s', 'talemy' ),
							// Get correct total time by multiplying interval by number of repeats
							absint( $course_pricing['interval'] * $course_pricing['repeats'] ),
							esc_html( $course_pricing['repeat_frequency'] )
						);
					}
					$price_html .= '</span>';
					$price_html .= '</div>';
				}

			} else {
				$price_html .= '<div class="course-price">';
				$price_html .= '<span class="price-amount">' . talemy_get_ld_price_formatted( $course_pricing['price'] ) . '</span>';
				$price_html .= '<span class="recurring-duration">';

				if ( ( ! empty( $course_pricing['interval'] ) ) && ( ! empty( $course_pricing['frequency'] ) ) ) {

					if ( $short_form ) {

						if ( $course_pricing['interval'] > 1 ) {
							$price_html .= sprintf(
								// translators: Recurring duration message.
								esc_html_x( '/%1$s %2$s', 'Recurring duration message', 'talemy' ),
								esc_html( $course_pricing['interval'] ),
								esc_html( $course_pricing['frequency'] )
							);

						} else {
							$price_html .= sprintf(
								// translators: Recurring duration message.
								esc_html_x( '/%s', 'Recurring duration message', 'talemy' ),
								esc_html( $course_pricing['frequency'] )
							);
						}

					} else {
						$price_html .= sprintf(
							// translators: Recurring duration message.
							esc_html_x( 'Every %1$s %2$s', 'Recurring duration message', 'talemy' ),
							esc_html( $course_pricing['interval'] ),
							esc_html( $course_pricing['frequency'] )
						);

						if ( ( ! empty( $course_pricing['repeats'] ) ) && ( ! empty( $course_pricing['repeat_frequency'] ) ) ) {
							$price_html .= sprintf(
								// translators: placeholders: %1$s Number of times the recurring payment repeats, %2$s Frequency of recurring payments: day, week, month, year.
								esc_html__( ' for %1$s %2$s', 'talemy' ),
								// Get correct total time by multiplying interval by number of repeats
								absint( $course_pricing['interval'] * $course_pricing['repeats'] ),
								esc_html( $course_pricing['repeat_frequency'] )
							);
						}
					}
				}

				$price_html .= '</span>';
				$price_html .= '</div>';
			}
		}
	}

	return apply_filters( 'talemy_course_price', $price_html, $course_id, $short_form );
}

/**
 * Get group price
 *
 * @param  int    $group_id      group id
 * @param  array  $short_form    short form
 * @return string
 */
function talemy_get_ld_group_price( $group_id = false, $short_form = false ) {

	if ( empty( $group_id ) ) {
		$group_id = get_the_ID();
	}

	if ( empty( $group_id ) ) {
		return '';
	}

	$group_pricing = learndash_get_group_price( $group_id );
	$group_pricing = wp_parse_args(
		$group_pricing,
		[
			'type'             => LEARNDASH_DEFAULT_GROUP_PRICE_TYPE,
			'price'            => '',
			'interval'         => '',
			'frequency'        => '',
			'trial_price'      => '',
			'trial_interval'   => '',
			'trial_frequency'  => '',
			'repeats'          => '',
			'repeat_frequency' => '',
		]
	);
	$price_type = strtolower( $group_pricing['type'] );
	$price_html = '';

	if ( 'subscribe' === $price_type ) {
		if ( ( empty( $group_pricing['price'] ) ) || ( empty( $group_pricing['interval'] ) ) || ( empty( $group_pricing['frequency'] ) ) ) {
			$group_pricing['type']             = LEARNDASH_DEFAULT_GROUP_PRICE_TYPE;
			$group_pricing['interval']         = '';
			$group_pricing['frequency']        = '';
			$group_pricing['trial_price']      = '';
			$group_pricing['trial_interval']   = '';
			$group_pricing['trial_frequency']  = '';
			$group_pricing['repeats']          = '';
			$group_pricing['repeat_frequency'] = '';
		} else {
			if ( empty( $group_pricing['trial_price'] ) ) {
				$group_pricing['trial_price'] = 0;
			} elseif ( ( empty( $group_pricing['trial_interval'] ) ) || ( empty( $group_pricing['trial_frequency'] ) ) ) {
				$group_pricing['trial_price'] = '';
			}
		}
	}

	if ( 'free' == $price_type || '' === $group_pricing['price'] ) {
		$label = apply_filters( 'learndash_no_price_price_label', ( 'closed' === $price_type ? esc_html__( 'Closed', 'talemy' ) : esc_html__( 'Free', 'talemy' ) ) );
		$price_html .= '<div class="course-price"><span class="price-amount">' . $label . '</span></div>';

	} else if ( ! empty( $group_pricing['price'] ) ) {

		if ( 'subscribe' !== $price_type ) {
			
			$price_html .= '<div class="course-price">';
			$price_html .= '<span class="price-amount">' . talemy_get_ld_price_formatted( $group_pricing['price'] ) . '</span>';
			$price_html .= '</div>';

		} else {

			if ( ! empty( $group_pricing['trial_frequency'] ) && ! $short_form ) {
				$price_html .= '<div class="course-trial-price recurring">';
				$price_html .= '<span class="price-amount">' . talemy_get_ld_price_formatted( $group_pricing['trial_price'] ) . '</span>';

				if ( ( ! empty( $group_pricing['trial_interval'] ) ) && ( ! empty( $group_pricing['trial_frequency'] ) ) ) {
					$price_html .= '<span class="price-text">';
					$price_html .= sprintf(
						// translators: placeholders: Trial interval, Trial frequency.
						esc_html_x( 'Trial price for %1$s %2$s, then', 'placeholders: Trial interval, Trial frequency', 'talemy' ),
						absint( $group_pricing['trial_interval'] ),
						esc_html( $group_pricing['trial_frequency'] )
					);
					$price_html .= '</span>';
				}
				$price_html .= '</div>';

				if ( ( ! empty( $group_pricing['interval'] ) ) && ( ! empty( $group_pricing['frequency'] ) ) ) {
					$price_html .= '<div class="course-price recurring">';
					$price_html .= '<span class="price-amount">' . talemy_get_ld_price_formatted( $group_pricing['price'] ) . '</span>';

					$price_html .= '<span class="price-text">';
					$price_html .= sprintf(
						// translators: placeholders: %1$s Interval of recurring payments (number), %2$s Frequency of recurring payments: day, week, month or year.
						esc_html_x( 'Full price every %1$s %2$s afterward', 'Recurring duration message', 'talemy' ),
						absint( $group_pricing['interval'] ),
						esc_html( $group_pricing['frequency'] )
					);

					if ( ! empty( $group_pricing['repeats'] ) ) {
						$price_html .= ' ';
						$price_html .= sprintf(
							// translators: placeholders: %1$s Number of times the recurring payment repeats, %2$s Frequency of recurring payments: day, week, month, year.
							esc_html__( 'for %1$s %2$s', 'talemy' ),
							// Get correct total time by multiplying interval by number of repeats
							absint( $group_pricing['interval'] * $group_pricing['repeats'] ),
							esc_html( $group_pricing['repeat_frequency'] )
						);
					}
					$price_html .= '</span>';
					$price_html .= '</div>';
				}

			} else {
				$price_html .= '<div class="course-price">';
				$price_html .= '<span class="price-amount">' . talemy_get_ld_price_formatted( $group_pricing['price'] ) . '</span>';
				$price_html .= '<span class="recurring-duration">';

				if ( ( ! empty( $group_pricing['interval'] ) ) && ( ! empty( $group_pricing['frequency'] ) ) ) {

					if ( $short_form ) {

						if ( $group_pricing['interval'] > 1 ) {
							$price_html .= sprintf(
								// translators: Recurring duration message.
								esc_html_x( '/%1$s %2$s', 'Recurring duration message', 'talemy' ),
								esc_html( $group_pricing['interval'] ),
								esc_html( $group_pricing['frequency'] )
							);

						} else {
							$price_html .= sprintf(
								// translators: Recurring duration message.
								esc_html_x( '/%s', 'Recurring duration message', 'talemy' ),
								esc_html( $group_pricing['frequency'] )
							);
						}

					} else {
						$price_html .= sprintf(
							// translators: Recurring duration message.
							esc_html_x( 'Every %1$s %2$s', 'Recurring duration message', 'talemy' ),
							esc_html( $group_pricing['interval'] ),
							esc_html( $group_pricing['frequency'] )
						);

						if ( ( ! empty( $group_pricing['repeats'] ) ) && ( ! empty( $group_pricing['repeat_frequency'] ) ) ) {
							$price_html .= sprintf(
								// translators: placeholders: %1$s Number of times the recurring payment repeats, %2$s Frequency of recurring payments: day, week, month, year.
								esc_html__( ' for %1$s %2$s', 'talemy' ),
								// Get correct total time by multiplying interval by number of repeats
								absint( $group_pricing['interval'] * $group_pricing['repeats'] ),
								esc_html( $group_pricing['repeat_frequency'] )
							);
						}
					}
				}

				$price_html .= '</span>';
				$price_html .= '</div>';
			}
		}
	}

	return apply_filters( 'talemy_group_price', $price_html, $group_id, $short_form );
}

if ( ! function_exists( 'talemy_get_ld_option_course_cats' ) ) {
    /**
     * Get LearnDash course categoreis
     *
     * @param  string $type  value type: id (default) or slug
     * @return array course categoreis
     */
    function talemy_get_ld_option_course_cats( $type = 'id' ) {
        $cats       = get_terms( 'hide_empty=0&taxonomy=ld_course_category' );
        $cats_array = array();
        if ( ! empty( $cats ) && ! is_wp_error( $cats ) ) {
            foreach ( $cats as $cat ) {
                $value              = 'slug' == $type ? $cat->slug : $cat->term_id;
                $cats_array[$value] = wp_specialchars_decode( $cat->name );
            }
        }
        return $cats_array;
    }
}

if ( ! function_exists( 'talemy_get_ld_option_course_tags' ) ) {
    /**
     * Get LearnDash course tags
     * @return array course tags
     */
    function talemy_get_ld_option_course_tags() {
        $tags       = get_terms( 'hide_empty=0&taxonomy=ld_course_tag' );
        $tags_array = array();
        if ( ! empty( $tags ) && ! is_wp_error( $tags ) ) {
            foreach ( $tags as $tag ) {
                $tags_array[$tag->term_id] = wp_specialchars_decode( $tag->name );
            }
        }
        return $tags_array;
    }
}

if ( ! function_exists( 'talemy_get_ld_option_courses' ) ) {
    /**
     * Get LearnDash courses
     * @return array courses
     */
    function talemy_get_ld_option_courses() {
        $options = array();
        $posts   = get_posts( array(
            'post_type'      => 'sfwd-courses',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
        ) );
        foreach ( $posts as $post ) {
            $options[$post->ID] = $post->post_title;
        }
        return $options;
    }
}

if ( ! function_exists( 'talemy_get_ld_course_student_count' ) ) {
    /**
     * Get LearnDash course student count
     * @param  integer $course_id     course id
     * @param  integer $base_count    base count
     * @return integer                student count
     */
    function talemy_get_ld_course_student_count( $course_id, $base_count = 0 ) {
        $students_count = get_transient( 'talemy_ld_course_students_' . $course_id );

        if ( false !== $students_count ) {
            return $students_count;
        }

        $course_access_users = learndash_get_users_for_course( $course_id, [], false );
        $students_count = 0;

        if ( $course_access_users instanceof \WP_User_Query && property_exists( $course_access_users, 'total_users' ) && ! empty( $course_access_users ) ) {
          $students_count = $course_access_users->total_users;
        }
        $students_count += absint( $base_count );
        set_transient( 'talemy_ld_course_students_' . $course_id, $students_count, 60 );

        return $students_count;
    }
}

if ( ! function_exists( 'talemy_get_ld_group_student_count' ) ) {
    /**
     * Get LearnDash group student count
     * @param  integer $post_id       post id
     * @param  integer $base_count    base count
     * @return integer                student count
     */
    function talemy_get_ld_group_student_count( $post_id, $base_count = 0 ) {
        $group_users = learndash_get_groups_user_ids( $post_id );
        if (  ( empty( $group_users ) ) || ( ! is_array( $group_users ) ) ) {
            $group_users = array();
        }
        $count = count( $group_users );
        return $count + absint( $base_count );
    }
}

if ( ! function_exists( 'talemy_get_wc_product_price' ) ) {
    /**
     * Get Woocommerce product price
     */
    function talemy_get_wc_product_price( $product_id = null ) {
        if ( empty( $product_id ) ) {
            return '';
        }

        $product = wc_get_product( $product_id );

        if ( ! $product ) {
            return '';
        }

        return '<div class="course-price wc-course-price">' . $product->get_price_html() . '</div>';
    }
}

if ( ! function_exists( 'talemy_get_ld_wc_payment_button' ) ) {
    /**
     * Get WooCommerce payment button for a course
     */
    function talemy_get_ld_wc_payment_button( $course_id, $product_id = null ) {
        if ( empty( $course_id ) || empty( $product_id ) ) {
            return '';
        }

        $meta = learndash_get_setting( $course_id );

        switch ( get_post_type( $course_id ) ) {
        case 'sfwd-courses':$post_label_prefix = 'course';
            break;
        case 'groups':$post_label_prefix = 'group';
            break;
        }

        $price_type = ( isset( $meta[$post_label_prefix . '_price_type'] ) ) ? $meta[$post_label_prefix . '_price_type'] : '';

        if ( 'closed' !== $price_type ) {
            return '';
        }

        $button              = '';
        $custom_button_url   = ! empty( $meta['custom_button_url'] ) ? $meta['custom_button_url'] : '';
        $custom_button_label = ! empty( $meta['custom_button_label'] ) ? $meta['custom_button_label'] : '';

        if ( empty( $custom_button_label ) ) {
            $button_text = LearnDash_Custom_Label::get_label( 'button_take_this_course' );
        } else {
            $button_text = esc_attr( $custom_button_label );
        }

        if ( empty( $custom_button_url ) ) {
            $checkout_url = wc_get_cart_url();
            $button_url   = add_query_arg( 'add-to-cart', $product_id, $checkout_url );
        } else {
            $custom_button_url = trim( $custom_button_url );
            /**
             * If the value does NOT start with [http://, https://, /] we prepend the home URL.
             */
            if (  ( stripos( $custom_button_url, 'http://', 0 ) !== 0 ) && ( stripos( $custom_button_url, 'https://', 0 ) !== 0 ) && ( strpos( $custom_button_url, '/', 0 ) !== 0 ) ) {
                $button_url = get_home_url( null, $custom_button_url );
            } else {
                $button_url = $custom_button_url;
            }
        }

        $button_url = apply_filters( 'sf_learndash_wc_payment_button_url', $button_url, $course_id, $product_id );

        if ( ! empty( $button_url ) ) {
            $button = '<a class="btn btn-lg btn-primary btn-block" href="' . esc_url( $button_url ) . '">' . $button_text . '</a>';
        }

        return $button;
    }
}

if ( ! function_exists( 'talemy_ld_get_courses_progress' ) ) {
    function talemy_ld_get_courses_progress( $user_id, $sort_order = 'desc' ) {
        $course_completion_percentage = array();

        if ( ! $course_completion_percentage = wp_cache_get( $user_id, 'ld_courses_progress' ) ) {
            $course_progress = get_user_meta( $user_id, '_sfwd-course_progress', true );

            if ( ! empty( $course_progress ) && is_array( $course_progress ) ) {

                foreach ( $course_progress as $course_id => $coursep ) {
                    // We take default progress value as 1 % rather than 0%
                    $course_completion_percentage[$course_id] = 1; //

                    if ( 0 == $coursep['total'] ) {
                        continue;
                    }

                    $course_steps_count     = learndash_get_course_steps_count( $course_id );
                    $course_steps_completed = learndash_course_get_completed_steps( $user_id, $course_id, $coursep );

                    $completed_on = get_user_meta( $user_id, 'course_completed_' . $course_id, true );
                    if ( ! empty( $completed_on ) ) {

                        $coursep['completed'] = $course_steps_count;
                        $coursep['total']     = $course_steps_count;

                    } else {
                        $coursep['total']     = $course_steps_count;
                        $coursep['completed'] = $course_steps_completed;

                        if ( $coursep['completed'] > $coursep['total'] ) {
                            $coursep['completed'] = $coursep['total'];
                        }

                    }

                    // cannot divide by 0
                    if ( 0 == $coursep['total'] ) {
                        $course_completion_percentage[$course_id] = 0;
                    } else {
                        $course_completion_percentage[$course_id] = ceil(  ( $coursep['completed'] * 100 ) / $coursep['total'] );
                    }
                }
            }

            //Avoid running the queries multiple times if user's course progress is empty
            $course_completion_percentage = ! empty( $course_completion_percentage ) ? $course_completion_percentage : 'empty';

            wp_cache_set( $user_id, $course_completion_percentage, 'ld_courses_progress' );
        }

        $course_completion_percentage = 'empty' !== $course_completion_percentage ? $course_completion_percentage : array();

        if ( ! empty( $course_completion_percentage ) ) {
            // Sort.
            if ( 'asc' == $sort_order ) {
                asort( $course_completion_percentage );
            } else {
                arsort( $course_completion_percentage );
            }
        }

        return $course_completion_percentage;
    }
}

if ( ! function_exists( 'talemy_ld_get_custom_course_resume_url' ) ) {
    /**
     * Get all the URLs of current course ( lesson, topic, quiz )
     * @param $course_id
     *
     * @return array | string
     */
    function talemy_ld_get_custom_course_resume_url( $course_id ) {
        global $post;

        $course_price_type = learndash_get_course_meta_setting( $course_id, 'course_price_type' );
        if ( 'closed' == $course_price_type ) {
            $courses_progress = talemy_ld_get_courses_progress( get_current_user_id() );
            $user_courses     = learndash_user_get_enrolled_courses( get_current_user_id() );
            $course_progress  = isset( $courses_progress[$course_id] ) ? $courses_progress[$course_id] : null;
            if ( $course_progress <= 0 && ! in_array( $course_id, $user_courses ) ) {
                return get_the_permalink( $course_id );
            }
        }

        $navigation_urls = array();
        $lesson_list     = learndash_get_lesson_list( $course_id );

        if ( ! empty( $lesson_list ) ):

            foreach ( $lesson_list as $lesson ) {
                $lesson_topics   = learndash_get_topic_list( $lesson->ID );
                $course_progress = get_user_meta( get_current_user_id(), '_sfwd-course_progress', true );
                $completed       = ! empty( $course_progress[$course_id]['lessons'][$lesson->ID] ) && 1 === $course_progress[$course_id]['lessons'][$lesson->ID];

                $navigation_urls[] = array(
                    'url'      => get_permalink( $lesson->ID ),
                    'complete' => $completed ? 'yes' : 'no',
                );

                if ( ! empty( $lesson_topics ) ):
                    foreach ( $lesson_topics as $lesson_topic ) {

                        $completed = ! empty( $course_progress[$course_id]['topics'][$lesson->ID][$lesson_topic->ID] ) && 1 === $course_progress[$course_id]['topics'][$lesson->ID][$lesson_topic->ID];

                        $navigation_urls[] = array(
                            'url'      => get_permalink( $lesson_topic->ID ),
                            'complete' => $completed ? 'yes' : 'no',
                        );

                        $topic_quizzes = learndash_get_lesson_quiz_list( $lesson_topic->ID );

                        if ( ! empty( $topic_quizzes ) ):
                            foreach ( $topic_quizzes as $topic_quiz ) {
                                $navigation_urls[] = array(
                                    'url'      => get_permalink( $topic_quiz['post']->ID ),
                                    'complete' => learndash_is_quiz_complete( get_current_user_id(), $topic_quiz['post']->ID ) ? 'yes' : 'no',
                                );
                            }
                        endif;

                    }
                endif;

                $lesson_quizzes = learndash_get_lesson_quiz_list( $lesson->ID );

                if ( ! empty( $lesson_quizzes ) ):
                    foreach ( $lesson_quizzes as $lesson_quiz ) {
                        $navigation_urls[] = array(
                            'url'      => get_permalink( $lesson_quiz['post']->ID ),
                            'complete' => learndash_is_quiz_complete( get_current_user_id(), $lesson_quiz['post']->ID ) ? 'yes' : 'no',
                        );
                    }
                endif;
            }

        endif;

        $course_quizzes = learndash_get_course_quiz_list( $course_id );
        if ( ! empty( $course_quizzes ) ):
            foreach ( $course_quizzes as $course_quiz ) {
                $navigation_urls[] = array(
                    'url'      => get_permalink( $course_quiz['post']->ID ),
                    'complete' => learndash_is_quiz_complete( get_current_user_id(), $course_quiz['post']->ID ) ? 'yes' : 'no',
                );
            }
        endif;

        $key = array_search( 'no', array_column( $navigation_urls, 'complete' ) );
        if ( '' !== $key && isset( $navigation_urls[$key] ) ) {
            return $navigation_urls[$key]['url'];
        }
        return '';
    }
}

if ( ! function_exists( 'talemy_get_ld_course_resume_link' ) ) {
    /**
     * Get LearnDash course resume link
     *
     * @param  int $course_id  course id
     * @return string
     */
    function talemy_get_ld_course_resume_link( $course_id ) {
        if ( is_user_logged_in() ) {
            if ( ! empty( $course_id ) ) {
                $user           = wp_get_current_user();
                $step_course_id = $course_id;
                $course         = get_post( $step_course_id );
                $url            = talemy_ld_get_custom_course_resume_url( $course_id );

                if ( isset( $course ) && 'sfwd-courses' === $course->post_type ) {
                    //$last_know_step = get_user_meta( $user->ID, 'learndash_last_known_course_' . $step_course_id, true );
                    $last_know_step = '';

                    // User has not hit a LD module yet
                    if ( empty( $last_know_step ) ) {
                        if ( isset( $url ) && '' !== $url ) {
                            return $url;
                        } else {
                            return '';
                        }
                    }

                    //$step_course_id = 0;
                    // Sanity Check
                    if ( absint( $last_know_step ) ) {
                        $step_id = $last_know_step;
                    } else {
                        if ( isset( $url ) && '' !== $url ) {
                            return $url;
                        } else {
                            return '';
                        }
                    }

                    $last_know_post_object = get_post( $step_id );

                    // Make sure the post exists and that the user hit a page that was a post
                    // if $last_know_page_id returns '' then get post will return current pages post object
                    // so we need to make sure first that the $last_know_page_id is returning something and
                    // that the something is a valid post
                    if ( null !== $last_know_post_object ) {
                        $post_type = $last_know_post_object->post_type; // getting post_type of last page.
                        $label     = get_post_type_object( $post_type ); // getting Labels of the post type.
                        $title     = $last_know_post_object->post_title;

                        if ( function_exists( 'learndash_get_step_permalink' ) ) {
                            $permalink = learndash_get_step_permalink( $step_id, $step_course_id );
                        } else {
                            $permalink = get_permalink( $step_id );
                        }

                        return $permalink;
                    }
                }
            }
        } else {
            $course_price_type = learndash_get_course_meta_setting( $course_id, 'course_price_type' );
            if ( 'open' == $course_price_type ) {
                return talemy_ld_get_custom_course_resume_url( $course_id );
            }
        }

        return '';
    }
}

if ( ! function_exists( 'talemy_get_ld_course_retake_link' ) ) {
    /**
     * Get LearnDash course retake link
     *
     * @param  int $course_id  course id
     * @return string
     */
    function talemy_get_ld_course_retake_link( $course_id ) {
        $url            = '';
        $lesson_list    = learndash_get_lesson_list( $course_id );
        $course_quizzes = learndash_get_course_quiz_list( $course_id );

        if ( ! empty( $lesson_list[0] ) ) {
            $url = get_permalink( $lesson_list[0]->ID );
        } elseif ( ! empty( $course_quizzes[0] ) ) {
            $url = get_permalink( $course_quizzes[0]['post']->ID );
        }

        return $url;
    }
}