<?php
/**
 * WooCommerce integration
 *
 * @since   1.0.0
 * @package Talemy/Classes
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class Talemy_WooCommerce {

	/**
	 * Hooks
	 */
	public function __construct() {
		add_theme_support( 'wc-product-gallery-zoom' );
		add_theme_support( 'wc-product-gallery-lightbox' );
		add_theme_support( 'wc-product-gallery-slider' );

		add_filter( 'woocommerce_enqueue_styles', '__return_empty_array' );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ), 15 );

		add_filter('woocommerce_billing_fields', array( $this, 'billing_fields' ) );
		add_filter('woocommerce_shipping_fields', array( $this, 'shipping_fields' ) );

		add_filter( 'woocommerce_product_additional_information_heading', '__return_empty_string' );
		add_filter( 'woocommerce_product_description_heading', '__return_empty_string' );

		add_action( 'woocommerce_before_quantity_input_field', array( $this, 'before_quantity_input_field' ) );
		add_action( 'woocommerce_after_quantity_input_field', array( $this, 'after_quantity_input_field' ) );

		add_action( 'woocommerce_before_shop_loop', [$this, 'loop_top_start'], 12 );
		add_action( 'woocommerce_before_shop_loop', [$this, 'loop_top_end'], 35 );

		// Restructure products Loop.
		remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_product_link_close', 5 );
		remove_action( 'woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_title', 10 );
		remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_add_to_cart', 10 );
		remove_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_rating', 5 );

		add_action( 'woocommerce_before_shop_loop_item', array( $this, 'before_shop_loop_item' ) , 5 );
		add_action( 'woocommerce_before_shop_loop_item_title', array( $this, 'before_shop_loop_item_title' ), 30 );
		add_action( 'woocommerce_shop_loop_item_title', array( $this, 'shop_loop_item_title' ), 10 );
		add_action( 'woocommerce_after_shop_loop_item_title', array( $this, 'after_shop_loop_item_title' ), 20 );
		add_action( 'woocommerce_after_shop_loop_item_title', 'woocommerce_template_loop_rating', 15 );

		// Restructure products category Loop.
		remove_action( 'woocommerce_before_subcategory', 'woocommerce_template_loop_category_link_open', 5 );
		remove_action( 'woocommerce_shop_loop_subcategory_title', 'woocommerce_template_loop_category_title', 10 );
		remove_action( 'woocommerce_after_subcategory', 'woocommerce_template_loop_category_link_close', 10 );

		add_action( 'woocommerce_before_subcategory', array( $this, 'before_shop_loop_item' ) , 5 );
		add_action( 'woocommerce_shop_loop_subcategory_title', array( $this, 'before_shop_loop_item_title' ) , 10 );
		add_action( 'woocommerce_shop_loop_subcategory_title', array( $this, 'shop_loop_category_title' ) , 10 );
		add_action( 'woocommerce_after_subcategory', array( $this, 'after_shop_loop_category' ) , 10 );

		// Related & upsell products per page & columns
		add_filter( 'woocommerce_output_related_products_args', array( $this, 'related_products_args' ) );
		add_filter( 'woocommerce_upsell_display_args', array( $this, 'upsell_products_args' ) );

		add_filter( 'woocommerce_add_to_cart_fragments', array( $this, 'cart_item_count_fragment' ), 10, 1 );
		add_filter( 'woocommerce_gallery_thumbnail_size', function( $size ) {
			return 'thumbnail';
		} );

		// Cart page wrappers
		remove_action( 'woocommerce_cart_collaterals', 'woocommerce_cross_sell_display' );
		add_action( 'woocommerce_before_cart', [$this, 'cart_wrapper_start'] );
		add_action( 'woocommerce_after_cart', [$this, 'cart_wrapper_end'] );
		add_action( 'woocommerce_after_cart', 'woocommerce_cross_sell_display', 30 );

		// Checkout page wrappers
		add_action( 'woocommerce_checkout_before_customer_details', [$this, 'customer_details_start'], 0 );
		add_action( 'woocommerce_checkout_after_customer_details', [$this, 'customer_details_end'], 3000 );
		add_action( 'woocommerce_checkout_before_order_review_heading', [$this, 'order_review_start'], 0 );
		add_action( 'woocommerce_checkout_after_order_review', [$this, 'order_review_end'], 3000 );

		add_action( 'woocommerce_before_single_product_summary', [ $this, 'product_wrap_start'], 0 );
		add_action( 'woocommerce_before_single_product_summary', [ $this, 'product_column_start'], 30 );
		add_action( 'woocommerce_after_single_product_summary', [ $this, 'product_wrap_end'], 5 );
	}

	public function product_wrap_start() {
		talemy_breadcrumbs();
		echo '<div class="wc-product-wrap">';
		echo '<div class="wc-product-col">';
	}

	public function product_column_start() {
		echo '</div>';
		echo '<div class="wc-product-col sticky-top">';
	}

	public function product_wrap_end() {
		echo '</div>';
		echo '</div><!-- .wc-product-wrap -->';
	}

	/**
	 * Modify billing fields
	 *
	 * @param array $fields
	 * @return array
	 */
	public function billing_fields( $fields ) {
		$highlight_required = ( 'yes' == get_option( 'woocommerce_checkout_highlight_required_fields' ) );

		foreach ( $fields as $key => $field ) {
			$fields[$key]['placeholder'] = $field['label'];
			if ( $highlight_required && isset( $field['required'] ) && $field['required'] ) {
				$fields[$key]['placeholder'] = $field['label'] . ' *';
			}
		}
		return $fields;
	}

	/**
	 * Modify shipping fields
	 *
	 * @param array $fields
	 * @return array
	 */
	public function shipping_fields( $fields ) {
		$highlight_required = ( 'yes' == get_option( 'woocommerce_checkout_highlight_required_fields' ) );

		$shipping_fields = $fields;
		foreach ( $shipping_fields as $key => $field ) {
			$fields[$key]['placeholder'] = $field['label'];
			if ( $highlight_required && isset( $field['required'] ) && $field['required'] ) {
				$fields[$key]['placeholder'] = $field['label'] . ' *';
			}
		}
		return $fields;
	}

	/**
	 * Modify related products args
	 *
	 * @param array $args
	 * @return array
	 */
	public function related_products_args( $args ) {
	  	$args['posts_per_page'] = talemy_get_option( 'wc_related_count' );
	  	$args['columns'] = talemy_get_option( 'wc_related_columns' );
		return $args;
	}

	/**
	 * Modify upsell products args
	 *
	 * @param array $args
	 * @return array
	 */
	public function upsell_products_args( $args ) {
	  	$args['posts_per_page'] = talemy_get_option( 'wc_upsell_count' );
	  	$args['columns'] = talemy_get_option( 'wc_upsell_columns' );
		return $args;
	}

	/**
	 * Before shop loop item
	 */
	public function before_shop_loop_item() {
		$class = 'product-body';
		if ( 'hover' == talemy_get_option( 'wc_add_to_cart_display' ) ) {
			$class .= ' button-on-hover';
		}
		?>
		<div class="<?php echo esc_attr( $class ); ?>">
			<div class="product-thumb">
		<?php
	}

	/**
	 * Before shop loop item title
	 */
	public function before_shop_loop_item_title() {
		?>
		</a>
		</div><!-- .product-thumb -->
		<div class="product-info">
		<?php
	}

	/**
	 * Shop loop item title
	 */
	public function shop_loop_item_title() {
		global $product;

		$link = apply_filters( 'woocommerce_loop_product_link', get_the_permalink(), $product );

		echo '<h3 class="' . esc_attr( apply_filters( 'woocommerce_product_loop_title_classes', 'product-title' ) ) . '"><a href="' . esc_url( $link ) . '" class="woocommerce-LoopProduct-link woocommerce-loop-product__link">' . get_the_title() . '</a></h3>';

		echo '<div class="product-meta">';
	}

	/**
	 * After shop loop item title
	 */
	public function after_shop_loop_item_title() {
		?>
				</div><!-- .product-meta -->
				<div class="product-buttons">
					<?php if ( function_exists( 'tinv_get_option' ) ) : ?>
					<?php echo do_shortcode( "[ti_wishlists_addtowishlist loop=yes]" ); ?>
					<?php endif; ?>
					<?php woocommerce_template_loop_add_to_cart(); ?>
				</div><!-- .product-buttons -->
			</div><!-- .product-info -->
		</div><!-- .product-body -->
		<?php
	}

	/**
	 * Shop loop category title
	 */
	public function shop_loop_category_title( $category ) {
		?>
		<h3 class="woocommerce-loop-category__title">
			<a href="<?php echo esc_url( get_term_link( $category, 'product_cat' ) ); ?>">
			<?php
			echo esc_html( $category->name );

			if ( $category->count > 0 ) {
				echo apply_filters( 'woocommerce_subcategory_count_html', ' <mark class="count">(' . esc_html( $category->count ) . ')</mark>', $category ); // WPCS: XSS ok.
			}
			?>
			</a>
		</h3>
		<?php
	}

	/**
	 * After shop loop category
	 */
	public function after_shop_loop_category() {
		?>
			</div>
		</div>
		<?php
	}

	/**
	 * Before quantity input field
	 */
	public function before_quantity_input_field() {
		echo '<input class="minus" type="button" value="-">';
	}

	/**
	 * After quantity input field
	 */
	public function after_quantity_input_field() {
		echo '<input class="plus" type="button" value="+">';
	}

	/**
	 * Updates cart item count
	 * 
	 * @param  array $fragments AJAX fragments handled by WooCommerce.
	 * @return array
	 */
	public function cart_item_count_fragment( $fragments ) {
		$item_count = WC()->cart->get_cart_contents_count();
	    $fragments['span.cart-item-count'] = '<span class="cart-item-count item-count" style="' . esc_attr( $item_count == '0' ? 'display:none;' : '' ) . '">' . esc_html( $item_count ) . '</span>';
	    return $fragments;
	}

	public function widget_shopping_cart_proceed_to_checkout() {
		echo '<a href="' . esc_url( wc_get_checkout_url() ) . '" class="button alt checkout wc-forward">' . esc_html__( 'Checkout', 'talemy' ) . '</a>';
	}

	public function loop_top_start() {
		?>
        <div class="wc-listing-top">
		<?php
		if ( ! talemy_get_option( 'wc_shop_sort' ) ) {
			remove_action( 'woocommerce_before_shop_loop', 'woocommerce_catalog_ordering', 30 );
		}

		if ( ! talemy_get_option( 'wc_shop_results_count' ) ) {
			remove_action( 'woocommerce_before_shop_loop', 'woocommerce_result_count', 20 );
		}
    }

	public function loop_top_end() {
		?>
        </div>
		<?php
    }

	public function cart_wrapper_start() {
		?>
        <div class="wc-cart-form">
		<?php
    }

	public function cart_wrapper_end() {
		?>
        </div><!-- .wc-cart-form -->
		<?php
    }

	public function customer_details_start() {
		?>
		<div class="wc-customer-details">
		<?php
	}

	public function customer_details_end() {
		?>
		</div>
		<?php
	}

	public function order_review_start() {
		?>
		<div class="wc-order-review">
		<?php
	}

	public function order_review_end() {
		?>
		</div>
		<?php
	}

	public function enqueue_scripts() {
		$suffix = !TALEMY_DEV_MODE ? '.min' : '';

		wp_register_style(
			'talemy-woocommerce',
			TALEMY_THEME_URI . 'assets/css/woocommerce' . $suffix . '.css',
			false,
			TALEMY_THEME_VERSION
		);

		wp_enqueue_style( 'talemy-woocommerce' );
		wp_style_add_data( 'talemy-woocommerce', 'rtl', 'replace' );

		wp_register_style(
			'talemy-wc-blocks',
			TALEMY_THEME_URI . 'assets/css/wc-blocks.css',
			[ 'wc-blocks-packages-style' ],
			TALEMY_THEME_VERSION
		);
		wp_enqueue_style( 'talemy-wc-blocks' );	
	}
}

if ( class_exists( 'WooCommerce' ) ) {
	new Talemy_WooCommerce();
}

