<?php
/**
 * bbPress integration
 *
 * @since   1.1.7
 * @package Talemy/Classes
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'bbPress' ) ) {
    return;
}
class Talemy_bbPress {

    /**
	 * Hooks
	 */
	public function __construct() {
        // Enqueue scripts
        add_action( 'wp_enqueue_scripts', array( $this, 'scripts' ) );

        // bbPress page
        add_filter( 'is_bbpress', array( $this, 'bbp_page' ) );

        // Disable bbpress breadcrumbs
        add_filter( 'bbp_no_breadcrumb', '__return_true' );

        // Remove bbpress forum & topic description
        add_filter( 'bbp_get_single_forum_description', '__return_empty_string' );
        add_filter( 'bbp_get_single_topic_description', '__return_empty_string' );

        // Add search form
        add_action( 'bbp_template_before_single_forum', array( $this, 'add_search_form' ) );
        add_action( 'bbp_template_before_single_topic', array( $this, 'add_search_form' ) );
        add_action( 'bbp_template_before_search', array( $this, 'add_search_form' ) );

        // Replace subscribe separater
        add_filter( 'bbp_get_user_subscribe_link', array( $this, 'replace_separater' ), 10, 4 );

        // Add forum section to course page
        if ( talemy_get_option( 'ld_course_section_forums' ) ) {
            add_filter( 'talemy_course_sections', array( $this, 'add_forums_section' ) );
            add_action( 'talemy_course_section_forums', array( $this, 'add_forums_section_content' ) );
        }
    }

    /**
     * Print search form
     *
     * @return void
     */
    public function add_search_form() {
        ?>
        <?php if ( bbp_allow_search() ) : ?>
            <div class="bbp-search-form">
                <?php bbp_get_template_part( 'form', 'search' ); ?>
            </div>
        <?php endif; ?>
        <?php
    }

    /**
     * Replace separater
     */
    public function replace_separater( $html, $r, $user_id, $topic_id ) {
        return str_replace( '&nbsp;|&nbsp;', '<span class="sep"></span>', $html );
    }

    /**
     * Enqueue scripts
     *
     * @return void
     */
    public function scripts() {
        $suffix = !TALEMY_DEV_MODE ? '.min' : '';
        wp_register_style(
            'talemy-bbpress',
            TALEMY_THEME_URI . 'assets/css/bbpress' . $suffix . '.css',
            false,
            TALEMY_THEME_VERSION
        );

        if ( is_bbpress() ) {
            wp_enqueue_style( 'talemy-bbpress' );
            wp_style_add_data( 'talemy-bbpress', 'rtl', 'replace' );
        }
    }

    /**
     * Check if post has bbpress shortcodes
     *
     * @param bool $is_bbpress
     * @return bool
     */
    public function bbp_page( $is_bbpress ) {
        global $post;
        if ( $post instanceof WP_Post && ( has_shortcode( $post->post_content, 'bbp-login' ) || has_shortcode( $post->post_content, 'bbp-register' ) || has_shortcode( $post->post_content, 'bbp-lost-pass' ) || has_shortcode( $post->post_content, 'bbp-single-forum' ) ) ) {
            $is_bbpress = true;
        }
        return $is_bbpress;
    }

    /**
     * Add forum section
     *
     * @param array $sections course sections
     * @return array course sections
     */
    public function add_forums_section( $sections ) {
        $sections['forums'] = esc_html__( 'Forums', 'talemy' );
        return $sections;
    }

    /**
     * Print forum section content
     *
     * @param array $course_id Course ID
     * @return void
     */
    public function add_forums_section_content( $course_id ) {
        wp_enqueue_style( 'talemy-bbpress' );
        
        $forums = get_posts( array(
            'post_type' => 'forum',
            'fields' => 'ids',
            'posts_per_page' => -1,
            'meta_key' => '_ld_associated_courses'
        ) );
        
        if ( !empty( $forums ) ) {
            foreach( $forums as $forum_id ) {
                $associated_courses = get_post_meta( $forum_id, '_ld_associated_courses', true );
                if ( is_array( $associated_courses ) && in_array( $course_id, $associated_courses ) ) {
                    echo do_shortcode( '[bbp-single-forum id="'. $forum_id .'"]' );
                }
            }
        }
    }
}

new Talemy_bbPress();