<?php
/**
 * Talemy Breadcrumbs
 *
 * @since   1.0
 * @package Talemy\Classes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
// Exit if accessed directly

class Talemy_Breadcrumbs {

	var $settings = [];
	var $labels   = [];

	public function __construct( $args = [] ) {
		$defaults = [
			'show_page_title'     => true,
			'show_taxonomy_title' => true,
			'show_home_icon'      => false
		];
		$this->settings = wp_parse_args( $args, $defaults );
		$this->labels   = apply_filters( 'talemy_breadcrumbs_labels', [
			'home'   => esc_html__( 'Home', 'talemy' ),
			'blog'   => esc_html__( 'Blog', 'talemy' ),
			'404'    => esc_html__( '404 Not found', 'talemy' ),
			// Translators: %s is the search query.
			'search' => esc_html__( 'Search results for %s', 'talemy' ),
		] );
	}

	public function render() {
		$items = $this->get_breadcrumbs();

		$separator = '<i class="sep fas fa-chevron-right"></i>';
		
		if ( count( $items ) < 1 ) {
			return '';
		}

		$items = apply_filters( 'talemy_breadcrumb_items', $items );

		echo '<nav class="breadcrumbs" itemscope itemtype="https://schema.org/BreadcrumbList">';

		for ( $i = 0; $i < count( $items ); $i++ ) {
			if (  ( count( $items ) - 1 ) === $i ) {

				echo '<span class="item last-item" itemtype="https://schema.org/ListItem" itemscope itemprop="itemListElement">';
				echo '<meta itemprop="position" content="' . ( $i + 1 ) . '">';

				if ( isset( $items[$i]['icon'] ) ) {
					echo sprintf( $items[$i]['icon'] );
				}
				echo '<span itemprop="name">';
				echo esc_html( $items[$i]['name'] );
				echo '</span>';

				if ( isset( $items[$i]['url'] ) ) {
					echo '<meta itemprop="url" content="' . esc_attr( $items[$i]['url'] ) . '"/>';
				}

				echo '</span>';

			} else if ( 0 === $i ) {

				echo '<span class="item first-item" itemprop="itemListElement" itemtype="https://schema.org/ListItem">';
				echo '<meta itemprop="position" content="' . ( $i + 1 ) . '">';

				if ( isset( $items[$i]['url'] ) ) {
					echo '<a href="' . esc_attr( $items[$i]['url'] ) . '" itemprop="item">';

					if ( isset( $items[$i]['icon'] ) ) {
						echo sprintf( $items[$i]['icon'] );
					}

					echo '<span itemprop="name">';
					echo esc_html( $items[$i]['name'] );
					echo '</span>';

					echo '</a>';
				} else {
					echo '<span itemprop="name">';
					echo esc_html( $items[$i]['name'] );
					echo '</span>';
				}

				if (
					isset( $items[$i]['url'] )
				) {
					echo '<meta itemprop="url" content="' . esc_attr( $items[$i]['url'] ) . '"/>';
				}

				echo '</span>';
				echo sprintf( $separator );

			} else {
				echo '<span class="item" itemprop="item">';

				echo '<meta itemprop="position" content="' . ( $i + 1 ) . '">';

				if ( isset( $items[$i]['url'] ) ) {
					echo '<a href="' . esc_attr( $items[$i]['url'] ) . '" itemprop="item">';

					if ( isset( $items[$i]['icon'] ) ) {
						echo sprintf( $items[$i]['icon'] );
					}

					echo '<span itemprop="name">';
					echo esc_html( $items[$i]['name'] );
					echo '</span>';

					echo '</a>';
				} else {
					echo '<span itemprop="name">';
					echo esc_html( $items[$i]['name'] );
					echo '</span>';
				}

				if (
					isset( $items[$i]['url'] )
				) {
					echo '<meta itemprop="url" content="' . esc_attr( $items[$i]['url'] ) . '"/>';
				}

				echo '</span>';
				echo sprintf( $separator );
			}
		}

		echo '</nav>';
	}

	private function get_breadcrumbs() {
		if ( is_admin() ) {
			return [];
		}

		if ( did_action( 'wp' ) === 0 ) {
			return [];
		}

		$home_icon = '';

		if ( $this->settings['show_home_icon'] ) {
			$home_icon = '<span class="fas fa-home"></span>';
		}

		$items[] = [
			'name' => $this->labels['home'],
			'url'  => esc_url( home_url( '/' ) ),
			'type' => 'front_page',
			'icon' => $home_icon,
		];

		if ( is_404() ) {
			$items[] = [
				'type' => '404',
				'name' => $this->labels['404'],
				'url'  => '',
			];

		} elseif ( is_search() ) {
			$items[] = [
				'type' => 'search',
				'name' => sprintf( $this->labels['search'], get_search_query() ),
				'url'  => get_search_link(),
			];

		} elseif ( is_page() ) {
			global $post;

			$items = array_merge(
				$items,
				array_reverse( $this->get_page_hierarchy( $post->ID ) )
			);

		} elseif ( is_single() ) {
			global $post;

			$taxonomies = get_object_taxonomies( $post->post_type, 'objects' );

			$primary_taxonomy_hash = [
				'post'    => 'category',
				'product' => 'product_cat',
			];

			$slugs = [];

			if ( isset( $primary_taxonomy_hash[$post->post_type] ) ) {
				foreach ( $taxonomies as $key => $tax ) {
					if ( $tax->name === $primary_taxonomy_hash[$post->post_type] ) {
						$slugs[] = $tax->name;
						break;
					}
				}
			}

			$items = array_merge(
				$items,
				$this->get_custom_post_type_archive()
			);

			if ( ! empty( $taxonomies ) ) {
				if ( empty( $slugs ) ) {
					foreach ( $taxonomies as $key => $tax ) {
						if ( true === $tax->show_ui && true === $tax->public && false !== $tax->hierarchical ) {
							array_push( $slugs, $tax->name );
						}
					}
				}

				$terms = wp_get_post_terms( $post->ID, $slugs );

				if ( ! empty( $terms ) ) {
					$lowest_term = $this->get_lowest_taxonomy_terms(
						$post, $terms,
						$slugs[0]
					);

					$term = $lowest_term[0];

					$items = array_merge(
						$items,
						array_reverse(
							$this->get_term_hierarchy(
								$term->term_id,
								$term->taxonomy
							)
						)
					);
				}
			}

			$items = array_merge(
				$items,
				array_reverse( $this->get_page_hierarchy( $post->ID ) )
			);

		} elseif ( is_archive() ) {

			if ( is_category() ) {
				$term_id = get_query_var( 'cat' );
				$items   = array_merge(
					$items,
					array_reverse( $this->get_term_hierarchy( $term_id, 'category' ) )
				);

			} elseif ( is_tag() ) {
				$term_id = get_query_var( 'tag' );
				$term    = get_term_by( 'slug', $term_id, 'post_tag' );

				if ( empty( $term ) || is_wp_error( $term ) ) {
					return [];
				}

				$items[] = [
					'type'     => 'taxonomy',
					'name'     => $term->name,
					'url'      => get_term_link( $term_id, 'post_tag' ),
					'taxonomy' => 'post_tag',
				];

			} elseif ( is_tax() ) {
				$term_id  = get_queried_object()->term_id;
				$taxonomy = get_queried_object()->taxonomy;
				$items = array_merge(
					$items,
					$this->get_custom_post_type_archive(),
					array_reverse( $this->get_term_hierarchy( $term_id, $taxonomy ) )
				);

			} elseif ( is_author() ) {
				$author_data = get_user_by( 'id', get_the_author_meta( 'ID' ) );
				$items[]     = [
					'type' => 'author',
					'name' => $author_data->display_name,
					'url'  => get_author_posts_url( $author_data->ID, $author_data->user_nicename ),
					'id'   => $author_data->ID,
				];

			} elseif ( is_date() ) {

				if ( get_option( 'permalink_structure' ) ) {
					$day   = get_query_var( 'day' );
					$month = get_query_var( 'monthnum' );
					$year  = get_query_var( 'year' );
				} else {
					$m     = get_query_var( 'm' );
					$year  = substr( $m, 0, 4 );
					$month = substr( $m, 4, 2 );
					$day   = substr( $m, 6, 2 );
				}

				$item_year              = [];
				$item_year['name']      = get_the_time( 'Y' );
				$item_year['url']       = get_year_link( $year );
				$item_year['date_type'] = 'yearly';
				$item_year['year']      = $year;

				$items[] = $item_year;

				if ( is_day() ) {
					$item_month              = [];
					$item_month['name']      = get_the_time( 'F' );
					$item_month['url']       = get_month_link( $year, $month );
					$item_month['date_type'] = 'monthly';
					$item_month['month']     = $month;
					$item_month['year']      = $year;

					$item_day              = [];
					$item_day['name']      = get_the_time( 'j' );
					$item_day['url']       = get_day_link( $year, $month, $day );
					$item_day['date_type'] = 'daily';
					$item_day['day']       = $day;
					$item_day['month']     = $month;
					$item_day['year']      = $year;

					$items[] = $item_month;
					$items[] = $item_day;

				} elseif ( is_month() ) {
					$item_month              = [];
					$item_month['name']      = get_the_time( 'F' );
					$item_month['url']       = get_month_link( $year, $month );
					$item_month['date_type'] = 'monthly';
					$item_month['month']     = $month;
					$item_month['year']      = $year;

					$items[] = $item_month;
				}

			} else {
				$post_type = get_query_var( 'post_type' );

				if ( $post_type ) {
					if ( is_array( $post_type ) ) {
						$post_type = $post_type[0];
					}
					$post_type_obj   = get_post_type_object( $post_type );
					$archive         = [];
					$archive['name'] = $post_type_obj->labels->name;
					$archive['url']  = get_post_type_archive_link( $post_type );
					$items[]         = $archive;
				}
			}
		}

		foreach ( $items as $key => $item ) {
			if ( empty( $item['name'] ) ) {
				$items[$key]['name'] = esc_html__( 'No title', 'talemy' );
			}
		}

		return $items;
	}

	/**
	 * Determine if the page has parents and in case it has, adds all page parents hierarchy
	 *
	 * @param $id , page id
	 *
	 * @return array
	 */
	private function get_page_hierarchy( $id, $has_single_check = true ) {
		$page = get_post( $id );

		if ( empty( $page ) || is_wp_error( $page ) ) {
			return [];
		}

		$return   = [];
		$page_obj = [];

		$page_obj['type']      = 'post';
		$page_obj['post_type'] = $page->post_type;
		$page_obj['name']      = $page->post_title;
		$page_obj['id']        = $id;
		$page_obj['url']       = get_permalink( $id );

		$return[] = $page_obj;

		if ( $page->post_parent > 0 ) {
			$return = array_merge(
				$return,
				$this->get_page_hierarchy( $page->post_parent )
			);
		}

		return $return;
	}

	/**
	 * Determine if the term has parents and in case it has, adds all term parents hierarchy
	 *
	 * @param $id , term id
	 * @param $taxonomy , term taxonomy name
	 *
	 * @return array
	 */
	private function get_term_hierarchy( $id, $taxonomy ) {
		$term = get_term( $id, $taxonomy );

		if ( empty( $term ) || is_wp_error( $term ) ) {
			return [];
		}

		$return   = [];
		$term_obj = [];

		$term_obj['type']     = 'taxonomy';
		$term_obj['name']     = $term->name;
		$term_obj['id']       = $id;
		$term_obj['url']      = get_term_link( $id, $taxonomy );
		$term_obj['taxonomy'] = $taxonomy;

		$return[] = $term_obj;

		if ( $term->parent > 0 ) {
			$return = array_merge(
				$return,
				$this->get_term_hierarchy( $term->parent, $taxonomy )
			);
		}

		return $return;
	}

	private function get_custom_post_type_archive() {
		$return = [];

		$post_type        = get_post_type();
		$post_type_object = get_post_type_object( $post_type );

		if (
			$post_type_object
			&& 'blog' !== $post_type && 'product' !== $post_type &&
			$post_type_object->has_archive
		) {
			// Add support for a non-standard label of 'archive_title' (special use case).
			$label = ! empty(
				$post_type_object->labels->archive_title
			) ? $post_type_object->labels->archive_title : $post_type_object->labels->name;

			// Core filter hook.
			$label = apply_filters(
				'post_type_archive_title',
				$label,
				$post_type_object->name
			);

			$return[] = [
				'name' => $label,
				'url'  => get_post_type_archive_link( $post_type ),
			];
		}

		return $return;
	}

	/**
	 * Returns the lowest hierarchical term
	 * @return array
	 */
	private function get_lowest_taxonomy_terms( $post, $terms, $taxonomy ) {
		$post_id = $post->ID;

		$primary_term = null;

		if ( class_exists( 'WPSEO_Primary_Term' ) ) {
			$primary_term = new WPSEO_Primary_Term( $taxonomy, $post_id );
			$primary_term = get_term( $primary_term->get_primary_term() );
		}

		// B. The SEO Framework
		if ( function_exists( 'the_seo_framework' ) ) {
			$primary_term = the_seo_framework()->get_primary_term(
				$post_id,
				$taxonomy
			);
		}

		// C. RankMath
		if ( class_exists( 'RankMath' ) ) {
			$primary_cat_id = get_post_meta( $post_id, "rank_math_primary_{$taxonomy}", true );
			$primary_term   = ( ! empty( $primary_cat_id ) ) ? get_term( $primary_cat_id, $taxonomy ) : '';
		}

		// D. SEOPress
		if ( function_exists( 'seopress_init' ) && 'category' == $taxonomy ) {
			$primary_cat_id = get_post_meta( $post_id, '_seopress_robots_primary_cat', true );
			$primary_term   = ( ! empty( $primary_cat_id ) ) ? get_term( $primary_cat_id, 'category' ) : '';
		}

		if ( $primary_term && ! is_wp_error( $primary_term ) ) {
			return [$primary_term];
		}

		// if terms is not array or its empty don't proceed
		if ( ! is_array( $terms ) || empty( $terms ) ) {
			return false;
		}

		return $this->filter_terms( $terms );
	}

	private function filter_terms( $terms ) {
		$return_terms = [];
		$term_ids     = [];

		foreach ( $terms as $t ) {
			$term_ids[] = $t->term_id;
		}

		foreach ( $terms as $t ) {
			if ( false == $t->parent || ! in_array( $t->parent, $term_ids ) ) {
				// remove this term
			} else {
				$return_terms[] = $t;
			}
		}

		if ( count( $return_terms ) ) {
			return $this->filter_terms( $return_terms );
		} else {
			return $terms;
		}
	}
}
