<?php

/**
 * Display a help tip.
 *
 * @param  string $tip        Help tip text.
 * @param  bool   $allow_html Allow sanitized HTML if true or escape.
 * @return string
 */
function sf_help_tip( $tip, $allow_html = false ) {
	if ( $allow_html ) {
		$tip = sf_sanitize_tooltip( $tip );
	} else {
		$tip = esc_attr( $tip );
	}

	return '<span class="sf-help-tip" data-tip="' . $tip . '"></span>';
}

/**
 * Retrieve a list of all published pages
 *
 * On large sites this can be expensive, so only load if on the settings page or $force is set to true
 *
 * @param bool $force Force the pages to be loaded even if not on settings
 * @return array $pages_options An array of the pages
 */
function sf_get_option_pages( $force = false ) {

	$pages_options = array( '' => '' ); // Blank option

	if( ( ! isset( $_GET['page'] ) || 'sf_settings' != $_GET['page'] ) && ! $force ) {
		return $pages_options;
	}

	$pages = get_pages();
	if ( $pages ) {
		foreach ( $pages as $page ) {
			$pages_options[ $page->ID ] = $page->post_title;
		}
	}

	return $pages_options;
}

/**
 * Get all post types
 * @return array  post type options
 */
function sf_get_option_post_types() {
    $post_types = get_post_types();
    $options = array();
    foreach ( $post_types as $name => $slug ) {
        $options[ $slug ] = $name;
    }
    return $options;
}

/**
 * Get a list of categories
 * @return array  category options
 */
function sf_get_option_cats() {
	$cats = get_categories( 'hide_empty=0' );
	$options = array();
	foreach ( $cats as $cat ) {
		$options[ $cat->term_id ] = htmlspecialchars_decode( $cat->cat_name );
	}
	return $options;
}


/**
 * Get a list of exclude cats options
 * @return array  category options
 */
function sf_get_option_exclude_cats() {
	$cats = get_categories( 'hide_empty=0' );
	$options = array();
	foreach ( $cats as $cat ) {
		$options[ '-' . $cat->term_id ] = htmlspecialchars_decode( $cat->name );
	}
	return $options;
}

/**
 * Get a list of tags as option
 * @return array  tag options
 */
function sf_get_option_tags() {
	$tags = get_tags( 'hide_empty=0' );
	$options = array();
	foreach ( $tags as $tag ) {
		$options[ $tag->term_id ] = htmlspecialchars_decode( $tag->name );
	}
	return $options;
}

/**
 * Get a list of terms as option
 * @return array  term options
 */
function sf_get_option_terms( $taxonomy = 'category' ) {
	$terms = get_terms( array(
		'taxonomy' => $taxonomy,
		'hide_empty' => 0,
	));
	$options = array();
	if ( !is_wp_error( $terms ) && !empty( $terms ) ) {
		foreach ( $terms as $term ) {
			$options[ $term->term_id ] = htmlspecialchars_decode( $term->name );
		}
	}
	return $options;
}

/**
 * Get a list of posts
 * @return array  post options
 */
function sf_get_option_posts( $post_type = 'post' ) {
	$options = array();
    $posts = get_posts( array(
        'post_type' => $post_type,
        'post_status' => 'publish',
        'posts_per_page' => -1,
		'no_found_rows' => 1
    ));
    foreach( $posts as $post ) {
        $options[ $post->ID ] = $post->post_title;
    }
    return $options;
}

/**
 * Get all users
 * @return array  users
 */
function sf_get_option_users( $args = [] ) {
	$options = array();
	$args    = apply_filters( 'sf_get_option_users_args', $args );
	$users   = get_users( $args );
	if ( ! empty( $users ) ) {
		foreach ( $users as $user ) {
			$options[ $user->ID ] = $user->display_name;
		}
	}
    return $options;
}

/**
 * Get options with author and admin role
 * @return array  users
 */
function sf_get_option_authors() {
    return sf_get_option_users( [
        'role__in' => [
            'author',
            'administrator'
        ]
    ] );
}

/**
 * Get user roles
 *
 * @return array user roles
 */
function sf_get_option_user_roles() {
	global $wp_roles;
	$roles = wp_list_pluck( $wp_roles->roles, 'name' );
	unset( $roles['administrator'] );
	$options = apply_filters( 'sf_get_option_user_roles', $roles );
	return $options;
}
/**
 * Get button styles
 * @return array  button styles
 */
function sf_get_option_button_styles() {
    return apply_filters( 'sf_option_button_styles', [
		'primary' => __( 'Primary', 'spirit' ),
		'secondary' => __( 'Secondary', 'spirit' ),
		'light' => __( 'Light', 'spirit' ),
		'dark' => __( 'Dark', 'spirit' ),
		'info' => __( 'Info', 'spirit' ),
		'success' => __( 'Success', 'spirit' ),
		'warning' => __( 'Warning', 'spirit' ),
		'danger' => __( 'Danger', 'spirit' ),
		'outline-primary' => __( 'Outline Primary', 'spirit' ),
		'outline-secondary' => __( 'Outline Secondary', 'spirit' ),
		'outline-light' => __( 'Outline Light', 'spirit' ),
		'outline-dark' => __( 'Outline Dark', 'spirit' ),
		'outline-info' => __( 'Outline Info', 'spirit' ),
		'outline-success' => __( 'Outline Success', 'spirit' ),
		'outline-warning' => __( 'Outline Warning', 'spirit' ),
		'outline-danger' => __( 'Outline Danger', 'spirit' )
    ] );
}

/**
 * Get button sizes
 * @return array  button sizes
 */
function sf_get_option_button_sizes() {
    return apply_filters( 'sf_option_button_sizes', [
		'sm' => __( 'Small', 'spirit' ),
		'' => __( 'Medium', 'spirit' ),
		'lg' => __( 'Large', 'spirit' ),
    ] );
}

/**
 * Get entrance animations
 *
 * @return array animation names
 */
function sf_get_option_entrance_animation() {
	return apply_filters( 'sf_option_entrance_animation', [
		'' => esc_html__( 'None', 'spirit' ),
		'fadeIn' => esc_html__( 'Fade In', 'spirit' ),
		'fadeInUp' => esc_html__( 'Fade In Up', 'spirit' ),
		'fadeInLeft' => esc_html__( 'Fade In Left', 'spirit' ),
		'fadeInRight' => esc_html__( 'Fade In Right', 'spirit' ),
		'fadeInDown' => esc_html__( 'Fade In Down', 'spirit' ),
		'fadeInUpDown' => esc_html__( 'Fade In Up & Down', 'spirit' ),
		'fadeInLeftRight' => esc_html__( 'Fade In Left & Right', 'spirit' ),
		'fadeInRightLeft' => esc_html__( 'Fade In Right & Left', 'spirit' ),
		'fadeInDownUp' => esc_html__( 'Fade In Down & Up', 'spirit' ),
		'zoomIn' => esc_html__( 'Zoom In', 'spirit' ),
		'slideInUp' => esc_html__( 'Slide In Up', 'spirit' )
	] );
}

/**
 * Get animation speed
 *
 * @return array animation speeds
 */
function sf_get_option_animation_speed() {
	$speed_options = [];
	
	for ( $i = 1; $i < 11; $i++ ) { 
		$speed_options[ $i * 100 ] = $i * 100;
	}

	return apply_filters( 'sf_option_animation_speed', $speed_options );
}

/**
 * Get a fontawesome icons from json file
 * @return array icons
 */
function sf_get_font_icons() {
	$fonts_json = file_get_contents( SF_FRAMEWORK_DIR . 'includes/data/fontawesome.json' );
	$fonts_array = json_decode( $fonts_json, true );
	return apply_filters( 'sf_font_icons', $fonts_array );
}

/**
 * Get mega menu columns array
 * @return array columns
 */
function sf_get_megamenu_column_names() {
	return array( "1/1","1/2 ( 2/4 ) ( 3/6 ) ( 4/8 )","1/3 ( 2/6 )","2/3 ( 4/6 )","1/4 ( 2/8 )","3/4 ( 6/8 )","1/5","2/5","3/5","4/5","1/6","5/6","1/7","2/7","3/7","4/7","5/7","6/7","1/8","3/8","5/8","7/8" );
}

if ( ! function_exists( 'sf_get_social_icon_names' ) ) {
	/**
	 * Get social icon names
	 * @return array
	 */
	function sf_get_social_icon_names() {
		return apply_filters( 'sf_social_icon_names', array(
			'facebook'      => 'Facebook',
			'twitter'       => 'X (Twitter)',
			'googleplus'    => 'Google Plus',
			'instagram'     => 'Instagram',
			'pinterest'     => 'Pinterest',
			'behance'       => 'Behance',
			'deviantart'    => 'Deviantart',
			'digg'          => 'Digg',
			'dribbble'      => 'Dribbble',
			'flickr'        => 'Flickr',
			'github'        => 'Github',
			'linkedin'      => 'Linkedin',
			'quora'         => 'Quora',
			'reddit'        => 'Reddit',
			'skype'         => 'Skype',
			'soundcloud'    => 'Soundcloud',
			'spotify'       => 'Spotify',
			'stackoverflow' => 'Stackoverflow',
			'steam'         => 'Steam',
			'stumbleupon'   => 'StumbleUpon',
			'tumblr'        => 'Tumblr',
			'telegram'      => 'Telegram',
			'vimeo'         => 'Vimeo',
			'whatsapp'      => 'WhatsApp',
			'windows'       => 'Windows',
			'wordpress'     => 'Wordpress',
			'youtube'       => 'Youtube',
			'weixin'        => 'Weixin',
			'tiktok'		=> 'Tiktok',
			'twitch'        => 'Twitch',
			'vk'            => 'VK',
			'viber'         => 'Viber',
			'rss'           => 'RSS',
			'email'         => esc_html__( 'Email', 'spirit' ),
			'phone'         => esc_html__( 'Phone', 'spirit' )
		) );
	}
}

/**
 * Get social icon classes
 * @return array
 */
function sf_get_social_icon_classes() {
	return apply_filters( 'sf_social_icon_classes', array(
		'facebook'      => 's-facebook',
		'twitter'       => 's-twitter',
		'googleplus'    => 's-google-plus',
		'instagram'     => 's-instagram',
		'pinterest'     => 's-pinterest',
		'behance'       => 's-behance',
		'deviantart'    => 's-deviantart',
		'digg'          => 's-digg',
		'dribbble'      => 's-dribbble',
		'flickr'        => 's-flickr',
		'github'        => 's-github',
		'linkedin'      => 's-linkedin',
		'quora'         => 's-quora',
		'reddit'        => 's-reddit',
		'skype'         => 's-skype',
		'soundcloud'    => 's-soundcloud',
		'spotify'       => 's-spotify',
		'stackoverflow' => 's-stack-overflow',
		'steam'         => 's-steam',
		'stumbleupon'   => 's-stumbleupon',
		'tumblr'        => 's-tumblr',
		'telegram'      => 's-telegram-plane',
		'vimeo'         => 's-vimeo',
		'whatsapp'      => 's-whatsapp',
		'windows'       => 's-windows',
		'wordpress'     => 's-wordpress',
		'youtube'       => 's-youtube',
		'weixin'        => 's-weixin',
		'tiktok'		=> 's-tiktok',
		'twitch'        => 's-twitch',
		'vk'            => 's-vk',
		'viber'         => 's-viber',
		'rss'           => 's-rss',
		'email'         => 's-envelope',
		'phone'         => 's-phone'
	) );
}

/**
 * Get social icon classes
 * @return array
 */
function sf_get_social_icon_colors() {
	return apply_filters( 'sf_social_icon_colors', [
		'facebook'   => '#375288',
		'twitter'    => '#1dadeb',
		'instagram'  => '#ed1376',
		'pinterest'  => '#b81621',
		'behance'    => '#1b64f6',
		'deviantart' => '#00e59b',
		'linkedin'   => '#1c86c6',
		'reddit'     => '#fc471e',
		'vk'         => '#5382b6',
		'telegram'   => '#229cce',
		'viber'      => '#7f509e',
		'whatsapp'   => '#5bba67',
		'email'      => '#392c44',
	] );
}

/**
 * Get share buttons
 * @return array
 */
function sf_get_share_networks() {
	return apply_filters( 'sf_share_networks', [
		'facebook'  => 'Facebook',
		'twitter'   => 'X (Twitter)',
		'pinterest' => 'Pinterest',
		'linkedin'  => 'LinkedIn',
		'reddit'    => 'Reddit',
		'vk'        => 'VKontakte',
		'telegram'  => 'Telegram',
		'viber'     => 'Viber',
		'whatsapp'  => 'WhatsApp',
		'email'     => esc_html( 'Email', 'spirit' ),
	] );
}

/**
 * Get form icons
 *
 * @param string $key
 * @return array|string
 */
function sf_get_form_icon_class( $key ) {
	$form_icons = apply_filters( 'sf_form_icon_class', [
		'username' => 'fas fa-user',
		'password' => 'fas fa-key',
		'email' => 'fas fa-envelope',
		'show_password' => 'fas fa-eye',
		'hide_password' => 'fas fa-eye-slash',
		'username_or_email' => 'fas fa-envelope',
		'first_name' => 'fas fa-user',
		'last_name' => 'fas fa-user',
		'tel' => 'fas fa-phone',
	] );

	if ( isset( $form_icons[ $key ] ) ) {
		return $form_icons[ $key ];
	}

	return '';
}

/**
 * Get social icon classes
 * @return array
 */
function sf_get_social_svg_icons() {
	return [
		'facebook'       => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 512"><path d="M279.14 288l14.22-92.66h-88.91v-60.13c0-25.35 12.42-50.06 52.24-50.06h40.42V6.26S260.43 0 225.36 0c-73.22 0-121.08 44.38-121.08 124.72v70.62H22.89V288h81.39v224h100.17V288z"/></svg>',
		'twitter'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M64 32C28.7 32 0 60.7 0 96V416c0 35.3 28.7 64 64 64H384c35.3 0 64-28.7 64-64V96c0-35.3-28.7-64-64-64H64zm297.1 84L257.3 234.6 379.4 396H283.8L209 298.1 123.3 396H75.8l111-126.9L69.7 116h98l67.7 89.5L313.6 116h47.5zM323.3 367.6L153.4 142.9H125.1L296.9 367.6h26.3z"/></svg>',
		'instagram'      => '<svg enable-background="new 0 0 14 14" viewBox="0 0 14 14" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"><clipPath id="a"><path d="m1.6 1.6h10.8v10.8h-10.8z"/></clipPath><g clip-path="url(#a)"><path clip-rule="evenodd" d="m9.7 1.6h-5.4c-1.5 0-2.7 1.2-2.7 2.7v5.4c0 1.5 1.2 2.7 2.7 2.7h5.4c1.5 0 2.7-1.2 2.7-2.7v-5.4c0-1.5-1.2-2.7-2.7-2.7zm-.5 5.1c.1.5 0 .9-.2 1.3-.3.4-.6.7-1 .9s-.9.3-1.3.2c-.5-.1-.9-.3-1.2-.6s-.5-.7-.6-1.2c-.1-.4 0-.9.2-1.3s.5-.7.9-.9.9-.3 1.3-.2c.5.1.9.3 1.2.6.4.3.6.7.7 1.2zm1-2.1c.4 0 .8-.4.8-.8s-.4-.8-.8-.8-.8.3-.8.8.4.8.8.8z" fill-rule="evenodd"/></g></svg>',
		'youtube'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path d="M549.655 124.083c-6.281-23.65-24.787-42.276-48.284-48.597C458.781 64 288 64 288 64S117.22 64 74.629 75.486c-23.497 6.322-42.003 24.947-48.284 48.597-11.412 42.867-11.412 132.305-11.412 132.305s0 89.438 11.412 132.305c6.281 23.65 24.787 41.5 48.284 47.821C117.22 448 288 448 288 448s170.78 0 213.371-11.486c23.497-6.321 42.003-24.171 48.284-47.821 11.412-42.867 11.412-132.305 11.412-132.305s0-89.438-11.412-132.305zm-317.51 213.508V175.185l142.739 81.205-142.739 81.201z"/></svg>',
		'vimeo'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M403.2 32H44.8C20.1 32 0 52.1 0 76.8v358.4C0 459.9 20.1 480 44.8 480h358.4c24.7 0 44.8-20.1 44.8-44.8V76.8c0-24.7-20.1-44.8-44.8-44.8zM377 180.8c-1.4 31.5-23.4 74.7-66 129.4-44 57.2-81.3 85.8-111.7 85.8-18.9 0-34.8-17.4-47.9-52.3-25.5-93.3-36.4-148-57.4-148-2.4 0-10.9 5.1-25.4 15.2l-15.2-19.6c37.3-32.8 72.9-69.2 95.2-71.2 25.2-2.4 40.7 14.8 46.5 51.7 20.7 131.2 29.9 151 67.6 91.6 13.5-21.4 20.8-37.7 21.8-48.9 3.5-33.2-25.9-30.9-45.8-22.4 15.9-52.1 46.3-77.4 91.2-76 33.3.9 49 22.5 47.1 64.7z"/></svg>',
		'facebook_group' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512"><!--! Font Awesome Pro 6.4.0 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license (Commercial License) Copyright 2023 Fonticons, Inc. --><path d="M96 128a128 128 0 1 1 256 0A128 128 0 1 1 96 128zM0 482.3C0 383.8 79.8 304 178.3 304h91.4C368.2 304 448 383.8 448 482.3c0 16.4-13.3 29.7-29.7 29.7H29.7C13.3 512 0 498.7 0 482.3zM609.3 512H471.4c5.4-9.4 8.6-20.3 8.6-32v-8c0-60.7-27.1-115.2-69.8-151.8c2.4-.1 4.7-.2 7.1-.2h61.4C567.8 320 640 392.2 640 481.3c0 17-13.8 30.7-30.7 30.7zM432 256c-31 0-59-12.6-79.3-32.9C372.4 196.5 384 163.6 384 128c0-26.8-6.6-52.1-18.3-74.3C384.3 40.1 407.2 32 432 32c61.9 0 112 50.1 112 112s-50.1 112-112 112z"/></svg>',
		'pinterest'      => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 496 512"><path d="M496 256c0 137-111 248-248 248-25.6 0-50.2-3.9-73.4-11.1 10.1-16.5 25.2-43.5 30.8-65 3-11.6 15.4-59 15.4-59 8.1 15.4 31.7 28.5 56.8 28.5 74.8 0 128.7-68.8 128.7-154.3 0-81.9-66.9-143.2-152.9-143.2-107 0-163.9 71.8-163.9 150.1 0 36.4 19.4 81.7 50.3 96.1 4.7 2.2 7.2 1.2 8.3-3.3.8-3.4 5-20.3 6.9-28.1.6-2.5.3-4.7-1.7-7.1-10.1-12.5-18.3-35.3-18.3-56.6 0-54.7 41.4-107.6 112-107.6 60.9 0 103.6 41.5 103.6 100.9 0 67.1-33.9 113.6-78 113.6-24.3 0-42.6-20.1-36.7-44.8 7-29.5 20.5-61.3 20.5-82.6 0-19-10.2-34.9-31.4-34.9-24.9 0-44.9 25.7-44.9 60.2 0 22 7.4 36.8 7.4 36.8s-24.5 103.8-29 123.2c-5 21.4-3 51.6-.9 71.2C65.4 450.9 0 361.1 0 256 0 119 111 8 248 8s248 111 248 248z"/></svg>',
		'linkedin'       => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M416 32H31.9C14.3 32 0 46.5 0 64.3v383.4C0 465.5 14.3 480 31.9 480H416c17.6 0 32-14.5 32-32.3V64.3c0-17.8-14.4-32.3-32-32.3zM135.4 416H69V202.2h66.5V416zm-33.2-243c-21.3 0-38.5-17.3-38.5-38.5S80.9 96 102.2 96c21.2 0 38.5 17.3 38.5 38.5 0 21.3-17.2 38.5-38.5 38.5zm282.1 243h-66.4V312c0-24.8-.5-56.7-34.5-56.7-34.6 0-39.9 27-39.9 54.9V416h-66.4V202.2h63.7v29.2h.9c8.9-16.8 30.6-34.5 62.9-34.5 67.2 0 79.7 44.3 79.7 101.9V416z"/></svg>',
		'behance'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path d="M232 237.2c31.8-15.2 48.4-38.2 48.4-74 0-70.6-52.6-87.8-113.3-87.8H0v354.4h171.8c64.4 0 124.9-30.9 124.9-102.9 0-44.5-21.1-77.4-64.7-89.7zM77.9 135.9H151c28.1 0 53.4 7.9 53.4 40.5 0 30.1-19.7 42.2-47.5 42.2h-79v-82.7zm83.3 233.7H77.9V272h84.9c34.3 0 56 14.3 56 50.6 0 35.8-25.9 47-57.6 47zm358.5-240.7H376V94h143.7v34.9zM576 305.2c0-75.9-44.4-139.2-124.9-139.2-78.2 0-131.3 58.8-131.3 135.8 0 79.9 50.3 134.7 131.3 134.7 61.3 0 101-27.6 120.1-86.3H509c-6.7 21.9-34.3 33.5-55.7 33.5-41.3 0-63-24.2-63-65.3h185.1c.3-4.2.6-8.7.6-13.2zM390.4 274c2.3-33.7 24.7-54.8 58.5-54.8 35.4 0 53.2 20.8 56.2 54.8H390.4z"/></svg>',
		'discord'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512"><path d="M524.531,69.836a1.5,1.5,0,0,0-.764-.7A485.065,485.065,0,0,0,404.081,32.03a1.816,1.816,0,0,0-1.923.91,337.461,337.461,0,0,0-14.9,30.6,447.848,447.848,0,0,0-134.426,0,309.541,309.541,0,0,0-15.135-30.6,1.89,1.89,0,0,0-1.924-.91A483.689,483.689,0,0,0,116.085,69.137a1.712,1.712,0,0,0-.788.676C39.068,183.651,18.186,294.69,28.43,404.354a2.016,2.016,0,0,0,.765,1.375A487.666,487.666,0,0,0,176.02,479.918a1.9,1.9,0,0,0,2.063-.676A348.2,348.2,0,0,0,208.12,430.4a1.86,1.86,0,0,0-1.019-2.588,321.173,321.173,0,0,1-45.868-21.853,1.885,1.885,0,0,1-.185-3.126c3.082-2.309,6.166-4.711,9.109-7.137a1.819,1.819,0,0,1,1.9-.256c96.229,43.917,200.41,43.917,295.5,0a1.812,1.812,0,0,1,1.924.233c2.944,2.426,6.027,4.851,9.132,7.16a1.884,1.884,0,0,1-.162,3.126,301.407,301.407,0,0,1-45.89,21.83,1.875,1.875,0,0,0-1,2.611,391.055,391.055,0,0,0,30.014,48.815,1.864,1.864,0,0,0,2.063.7A486.048,486.048,0,0,0,610.7,405.729a1.882,1.882,0,0,0,.765-1.352C623.729,277.594,590.933,167.465,524.531,69.836ZM222.491,337.58c-28.972,0-52.844-26.587-52.844-59.239S193.056,219.1,222.491,219.1c29.665,0,53.306,26.82,52.843,59.239C275.334,310.993,251.924,337.58,222.491,337.58Zm195.38,0c-28.971,0-52.843-26.587-52.843-59.239S388.437,219.1,417.871,219.1c29.667,0,53.307,26.82,52.844,59.239C470.715,310.993,447.538,337.58,417.871,337.58Z"/></svg>',
		'dribbble'       => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M256 8C119.252 8 8 119.252 8 256s111.252 248 248 248 248-111.252 248-248S392.748 8 256 8zm163.97 114.366c29.503 36.046 47.369 81.957 47.835 131.955-6.984-1.477-77.018-15.682-147.502-6.818-5.752-14.041-11.181-26.393-18.617-41.614 78.321-31.977 113.818-77.482 118.284-83.523zM396.421 97.87c-3.81 5.427-35.697 48.286-111.021 76.519-34.712-63.776-73.185-116.168-79.04-124.008 67.176-16.193 137.966 1.27 190.061 47.489zm-230.48-33.25c5.585 7.659 43.438 60.116 78.537 122.509-99.087 26.313-186.36 25.934-195.834 25.809C62.38 147.205 106.678 92.573 165.941 64.62zM44.17 256.323c0-2.166.043-4.322.108-6.473 9.268.19 111.92 1.513 217.706-30.146 6.064 11.868 11.857 23.915 17.174 35.949-76.599 21.575-146.194 83.527-180.531 142.306C64.794 360.405 44.17 310.73 44.17 256.323zm81.807 167.113c22.127-45.233 82.178-103.622 167.579-132.756 29.74 77.283 42.039 142.053 45.189 160.638-68.112 29.013-150.015 21.053-212.768-27.882zm248.38 8.489c-2.171-12.886-13.446-74.897-41.152-151.033 66.38-10.626 124.7 6.768 131.947 9.055-9.442 58.941-43.273 109.844-90.795 141.978z"/></svg>',
		'flickr'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M400 32H48C21.5 32 0 53.5 0 80v352c0 26.5 21.5 48 48 48h352c26.5 0 48-21.5 48-48V80c0-26.5-21.5-48-48-48zM144.5 319c-35.1 0-63.5-28.4-63.5-63.5s28.4-63.5 63.5-63.5 63.5 28.4 63.5 63.5-28.4 63.5-63.5 63.5zm159 0c-35.1 0-63.5-28.4-63.5-63.5s28.4-63.5 63.5-63.5 63.5 28.4 63.5 63.5-28.4 63.5-63.5 63.5z"/></svg>',
		'github'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 496 512"><path d="M165.9 397.4c0 2-2.3 3.6-5.2 3.6-3.3.3-5.6-1.3-5.6-3.6 0-2 2.3-3.6 5.2-3.6 3-.3 5.6 1.3 5.6 3.6zm-31.1-4.5c-.7 2 1.3 4.3 4.3 4.9 2.6 1 5.6 0 6.2-2s-1.3-4.3-4.3-5.2c-2.6-.7-5.5.3-6.2 2.3zm44.2-1.7c-2.9.7-4.9 2.6-4.6 4.9.3 2 2.9 3.3 5.9 2.6 2.9-.7 4.9-2.6 4.6-4.6-.3-1.9-3-3.2-5.9-2.9zM244.8 8C106.1 8 0 113.3 0 252c0 110.9 69.8 205.8 169.5 239.2 12.8 2.3 17.3-5.6 17.3-12.1 0-6.2-.3-40.4-.3-61.4 0 0-70 15-84.7-29.8 0 0-11.4-29.1-27.8-36.6 0 0-22.9-15.7 1.6-15.4 0 0 24.9 2 38.6 25.8 21.9 38.6 58.6 27.5 72.9 20.9 2.3-16 8.8-27.1 16-33.7-55.9-6.2-112.3-14.3-112.3-110.5 0-27.5 7.6-41.3 23.6-58.9-2.6-6.5-11.1-33.3 2.6-67.9 20.9-6.5 69 27 69 27 20-5.6 41.5-8.5 62.8-8.5s42.8 2.9 62.8 8.5c0 0 48.1-33.6 69-27 13.7 34.7 5.2 61.4 2.6 67.9 16 17.7 25.8 31.5 25.8 58.9 0 96.5-58.9 104.2-114.8 110.5 9.2 7.9 17 22.9 17 46.4 0 33.7-.3 75.4-.3 83.6 0 6.5 4.6 14.4 17.3 12.1C428.2 457.8 496 362.9 496 252 496 113.3 383.5 8 244.8 8zM97.2 352.9c-1.3 1-1 3.3.7 5.2 1.6 1.6 3.9 2.3 5.2 1 1.3-1 1-3.3-.7-5.2-1.6-1.6-3.9-2.3-5.2-1zm-10.8-8.1c-.7 1.3.3 2.9 2.3 3.9 1.6 1 3.6.7 4.3-.7.7-1.3-.3-2.9-2.3-3.9-2-.6-3.6-.3-4.3.7zm32.4 35.6c-1.6 1.3-1 4.3 1.3 6.2 2.3 2.3 5.2 2.6 6.5 1 1.3-1.3.7-4.3-1.3-6.2-2.2-2.3-5.2-2.6-6.5-1zm-11.4-14.7c-1.6 1-1.6 3.6 0 5.9 1.6 2.3 4.3 3.3 5.6 2.3 1.6-1.3 1.6-3.9 0-6.2-1.4-2.3-4-3.3-5.6-2z"/></svg>',
		'googleplus'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M256,8C119.1,8,8,119.1,8,256S119.1,504,256,504,504,392.9,504,256,392.9,8,256,8ZM185.3,380a124,124,0,0,1,0-248c31.3,0,60.1,11,83,32.3l-33.6,32.6c-13.2-12.9-31.3-19.1-49.4-19.1-42.9,0-77.2,35.5-77.2,78.1S142.3,334,185.3,334c32.6,0,64.9-19.1,70.1-53.3H185.3V238.1H302.2a109.2,109.2,0,0,1,1.9,20.7c0,70.8-47.5,121.2-118.8,121.2ZM415.5,273.8v35.5H380V273.8H344.5V238.3H380V202.8h35.5v35.5h35.2v35.5Z"/></svg>',
		'medium'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512"><path d="M180.5,74.262C80.813,74.262,0,155.633,0,256S80.819,437.738,180.5,437.738,361,356.373,361,256,280.191,74.262,180.5,74.262Zm288.25,10.646c-49.845,0-90.245,76.619-90.245,171.095s40.406,171.1,90.251,171.1,90.251-76.619,90.251-171.1H559C559,161.5,518.6,84.908,468.752,84.908Zm139.506,17.821c-17.526,0-31.735,68.628-31.735,153.274s14.2,153.274,31.735,153.274S640,340.631,640,256C640,171.351,625.785,102.729,608.258,102.729Z"/></svg>',
		'patreon'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M512 194.8c0 101.3-82.4 183.8-183.8 183.8-101.7 0-184.4-82.4-184.4-183.8 0-101.6 82.7-184.3 184.4-184.3C429.6 10.5 512 93.2 512 194.8zM0 501.5h90v-491H0v491z"/></svg>',
		'quora'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M440.5 386.7h-29.3c-1.5 13.5-10.5 30.8-33 30.8-20.5 0-35.3-14.2-49.5-35.8 44.2-34.2 74.7-87.5 74.7-153C403.5 111.2 306.8 32 205 32 105.3 32 7.3 111.7 7.3 228.7c0 134.1 131.3 221.6 249 189C276 451.3 302 480 351.5 480c81.8 0 90.8-75.3 89-93.3zM297 329.2C277.5 300 253.3 277 205.5 277c-30.5 0-54.3 10-69 22.8l12.2 24.3c6.2-3 13-4 19.8-4 35.5 0 53.7 30.8 69.2 61.3-10 3-20.7 4.2-32.7 4.2-75 0-107.5-53-107.5-156.7C97.5 124.5 130 71 205 71c76.2 0 108.7 53.5 108.7 157.7.1 41.8-5.4 75.6-16.7 100.5z"/></svg>',
		'reddit'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M201.5 305.5c-13.8 0-24.9-11.1-24.9-24.6 0-13.8 11.1-24.9 24.9-24.9 13.6 0 24.6 11.1 24.6 24.9 0 13.6-11.1 24.6-24.6 24.6zM504 256c0 137-111 248-248 248S8 393 8 256 119 8 256 8s248 111 248 248zm-132.3-41.2c-9.4 0-17.7 3.9-23.8 10-22.4-15.5-52.6-25.5-86.1-26.6l17.4-78.3 55.4 12.5c0 13.6 11.1 24.6 24.6 24.6 13.8 0 24.9-11.3 24.9-24.9s-11.1-24.9-24.9-24.9c-9.7 0-18 5.8-22.1 13.8l-61.2-13.6c-3-.8-6.1 1.4-6.9 4.4l-19.1 86.4c-33.2 1.4-63.1 11.3-85.5 26.8-6.1-6.4-14.7-10.2-24.1-10.2-34.9 0-46.3 46.9-14.4 62.8-1.1 5-1.7 10.2-1.7 15.5 0 52.6 59.2 95.2 132 95.2 73.1 0 132.3-42.6 132.3-95.2 0-5.3-.6-10.8-1.9-15.8 31.3-16 19.8-62.5-14.9-62.5zM302.8 331c-18.2 18.2-76.1 17.9-93.6 0-2.2-2.2-6.1-2.2-8.3 0-2.5 2.5-2.5 6.4 0 8.6 22.8 22.8 87.3 22.8 110.2 0 2.5-2.2 2.5-6.1 0-8.6-2.2-2.2-6.1-2.2-8.3 0zm7.7-75c-13.6 0-24.6 11.1-24.6 24.9 0 13.6 11.1 24.6 24.6 24.6 13.8 0 24.9-11.1 24.9-24.6 0-13.8-11-24.9-24.9-24.9z"/></svg>',
		'skype'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M424.7 299.8c2.9-14 4.7-28.9 4.7-43.8 0-113.5-91.9-205.3-205.3-205.3-14.9 0-29.7 1.7-43.8 4.7C161.3 40.7 137.7 32 112 32 50.2 32 0 82.2 0 144c0 25.7 8.7 49.3 23.3 68.2-2.9 14-4.7 28.9-4.7 43.8 0 113.5 91.9 205.3 205.3 205.3 14.9 0 29.7-1.7 43.8-4.7 19 14.6 42.6 23.3 68.2 23.3 61.8 0 112-50.2 112-112 .1-25.6-8.6-49.2-23.2-68.1zm-194.6 91.5c-65.6 0-120.5-29.2-120.5-65 0-16 9-30.6 29.5-30.6 31.2 0 34.1 44.9 88.1 44.9 25.7 0 42.3-11.4 42.3-26.3 0-18.7-16-21.6-42-28-62.5-15.4-117.8-22-117.8-87.2 0-59.2 58.6-81.1 109.1-81.1 55.1 0 110.8 21.9 110.8 55.4 0 16.9-11.4 31.8-30.3 31.8-28.3 0-29.2-33.5-75-33.5-25.7 0-42 7-42 22.5 0 19.8 20.8 21.8 69.1 33 41.4 9.3 90.7 26.8 90.7 77.6 0 59.1-57.1 86.5-112 86.5z"/></svg>',
		'snapchat'       => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M496.926,366.6c-3.373-9.176-9.8-14.086-17.112-18.153-1.376-.806-2.641-1.451-3.72-1.947-2.182-1.128-4.414-2.22-6.634-3.373-22.8-12.09-40.609-27.341-52.959-45.42a102.889,102.889,0,0,1-9.089-16.12c-1.054-3.013-1-4.724-.248-6.287a10.221,10.221,0,0,1,2.914-3.038c3.918-2.591,7.96-5.22,10.7-6.993,4.885-3.162,8.754-5.667,11.246-7.44,9.362-6.547,15.909-13.5,20-21.278a42.371,42.371,0,0,0,2.1-35.191c-6.2-16.318-21.613-26.449-40.287-26.449a55.543,55.543,0,0,0-11.718,1.24c-1.029.224-2.059.459-3.063.72.174-11.16-.074-22.94-1.066-34.534-3.522-40.758-17.794-62.123-32.674-79.16A130.167,130.167,0,0,0,332.1,36.443C309.515,23.547,283.91,17,256,17S202.6,23.547,180,36.443a129.735,129.735,0,0,0-33.281,26.783c-14.88,17.038-29.152,38.44-32.673,79.161-.992,11.594-1.24,23.435-1.079,34.533-1-.26-2.021-.5-3.051-.719a55.461,55.461,0,0,0-11.717-1.24c-18.687,0-34.125,10.131-40.3,26.449a42.423,42.423,0,0,0,2.046,35.228c4.105,7.774,10.652,14.731,20.014,21.278,2.48,1.736,6.361,4.24,11.246,7.44,2.641,1.711,6.5,4.216,10.28,6.72a11.054,11.054,0,0,1,3.3,3.311c.794,1.624.818,3.373-.36,6.6a102.02,102.02,0,0,1-8.94,15.785c-12.077,17.669-29.363,32.648-51.434,44.639C32.355,348.608,20.2,352.75,15.069,366.7c-3.868,10.528-1.339,22.506,8.494,32.6a49.137,49.137,0,0,0,12.4,9.387,134.337,134.337,0,0,0,30.342,12.139,20.024,20.024,0,0,1,6.126,2.741c3.583,3.137,3.075,7.861,7.849,14.78a34.468,34.468,0,0,0,8.977,9.127c10.019,6.919,21.278,7.353,33.207,7.811,10.776.41,22.989.881,36.939,5.481,5.778,1.91,11.78,5.605,18.736,9.92C194.842,480.951,217.707,495,255.973,495s61.292-14.123,78.118-24.428c6.907-4.24,12.872-7.9,18.489-9.758,13.949-4.613,26.163-5.072,36.939-5.481,11.928-.459,23.187-.893,33.206-7.812a34.584,34.584,0,0,0,10.218-11.16c3.434-5.84,3.348-9.919,6.572-12.771a18.971,18.971,0,0,1,5.753-2.629A134.893,134.893,0,0,0,476.02,408.71a48.344,48.344,0,0,0,13.019-10.193l.124-.149C498.389,388.5,500.708,376.867,496.926,366.6Zm-34.013,18.277c-20.745,11.458-34.533,10.23-45.259,17.137-9.114,5.865-3.72,18.513-10.342,23.076-8.134,5.617-32.177-.4-63.239,9.858-25.618,8.469-41.961,32.822-88.038,32.822s-62.036-24.3-88.076-32.884c-31-10.255-55.092-4.241-63.239-9.858-6.609-4.563-1.24-17.211-10.341-23.076-10.739-6.907-24.527-5.679-45.26-17.075-13.206-7.291-5.716-11.8-1.314-13.937,75.143-36.381,87.133-92.552,87.666-96.719.645-5.046,1.364-9.014-4.191-14.148-5.369-4.96-29.189-19.7-35.8-24.316-10.937-7.638-15.748-15.264-12.2-24.638,2.48-6.485,8.531-8.928,14.879-8.928a27.643,27.643,0,0,1,5.965.67c12,2.6,23.659,8.617,30.392,10.242a10.749,10.749,0,0,0,2.48.335c3.6,0,4.86-1.811,4.612-5.927-.768-13.132-2.628-38.725-.558-62.644,2.84-32.909,13.442-49.215,26.04-63.636,6.051-6.932,34.484-36.976,88.857-36.976s82.88,29.92,88.931,36.827c12.611,14.421,23.225,30.727,26.04,63.636,2.071,23.919.285,49.525-.558,62.644-.285,4.327,1.017,5.927,4.613,5.927a10.648,10.648,0,0,0,2.48-.335c6.745-1.624,18.4-7.638,30.4-10.242a27.641,27.641,0,0,1,5.964-.67c6.386,0,12.4,2.48,14.88,8.928,3.546,9.374-1.24,17-12.189,24.639-6.609,4.612-30.429,19.343-35.8,24.315-5.568,5.134-4.836,9.1-4.191,14.149.533,4.228,12.511,60.4,87.666,96.718C468.629,373.011,476.119,377.524,462.913,384.877Z"/></svg>',
		'soundcloud'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 640 512"><path d="M111.4 256.3l5.8 65-5.8 68.3c-.3 2.5-2.2 4.4-4.4 4.4s-4.2-1.9-4.2-4.4l-5.6-68.3 5.6-65c0-2.2 1.9-4.2 4.2-4.2 2.2 0 4.1 2 4.4 4.2zm21.4-45.6c-2.8 0-4.7 2.2-5 5l-5 105.6 5 68.3c.3 2.8 2.2 5 5 5 2.5 0 4.7-2.2 4.7-5l5.8-68.3-5.8-105.6c0-2.8-2.2-5-4.7-5zm25.5-24.1c-3.1 0-5.3 2.2-5.6 5.3l-4.4 130 4.4 67.8c.3 3.1 2.5 5.3 5.6 5.3 2.8 0 5.3-2.2 5.3-5.3l5.3-67.8-5.3-130c0-3.1-2.5-5.3-5.3-5.3zM7.2 283.2c-1.4 0-2.2 1.1-2.5 2.5L0 321.3l4.7 35c.3 1.4 1.1 2.5 2.5 2.5s2.2-1.1 2.5-2.5l5.6-35-5.6-35.6c-.3-1.4-1.1-2.5-2.5-2.5zm23.6-21.9c-1.4 0-2.5 1.1-2.5 2.5l-6.4 57.5 6.4 56.1c0 1.7 1.1 2.8 2.5 2.8s2.5-1.1 2.8-2.5l7.2-56.4-7.2-57.5c-.3-1.4-1.4-2.5-2.8-2.5zm25.3-11.4c-1.7 0-3.1 1.4-3.3 3.3L47 321.3l5.8 65.8c.3 1.7 1.7 3.1 3.3 3.1 1.7 0 3.1-1.4 3.1-3.1l6.9-65.8-6.9-68.1c0-1.9-1.4-3.3-3.1-3.3zm25.3-2.2c-1.9 0-3.6 1.4-3.6 3.6l-5.8 70 5.8 67.8c0 2.2 1.7 3.6 3.6 3.6s3.6-1.4 3.9-3.6l6.4-67.8-6.4-70c-.3-2.2-2-3.6-3.9-3.6zm241.4-110.9c-1.1-.8-2.8-1.4-4.2-1.4-2.2 0-4.2.8-5.6 1.9-1.9 1.7-3.1 4.2-3.3 6.7v.8l-3.3 176.7 1.7 32.5 1.7 31.7c.3 4.7 4.2 8.6 8.9 8.6s8.6-3.9 8.6-8.6l3.9-64.2-3.9-177.5c-.4-3-2-5.8-4.5-7.2zm-26.7 15.3c-1.4-.8-2.8-1.4-4.4-1.4s-3.1.6-4.4 1.4c-2.2 1.4-3.6 3.9-3.6 6.7l-.3 1.7-2.8 160.8s0 .3 3.1 65.6v.3c0 1.7.6 3.3 1.7 4.7 1.7 1.9 3.9 3.1 6.4 3.1 2.2 0 4.2-1.1 5.6-2.5 1.7-1.4 2.5-3.3 2.5-5.6l.3-6.7 3.1-58.6-3.3-162.8c-.3-2.8-1.7-5.3-3.9-6.7zm-111.4 22.5c-3.1 0-5.8 2.8-5.8 6.1l-4.4 140.6 4.4 67.2c.3 3.3 2.8 5.8 5.8 5.8 3.3 0 5.8-2.5 6.1-5.8l5-67.2-5-140.6c-.2-3.3-2.7-6.1-6.1-6.1zm376.7 62.8c-10.8 0-21.1 2.2-30.6 6.1-6.4-70.8-65.8-126.4-138.3-126.4-17.8 0-35 3.3-50.3 9.4-6.1 2.2-7.8 4.4-7.8 9.2v249.7c0 5 3.9 8.6 8.6 9.2h218.3c43.3 0 78.6-35 78.6-78.3.1-43.6-35.2-78.9-78.5-78.9zm-296.7-60.3c-4.2 0-7.5 3.3-7.8 7.8l-3.3 136.7 3.3 65.6c.3 4.2 3.6 7.5 7.8 7.5 4.2 0 7.5-3.3 7.5-7.5l3.9-65.6-3.9-136.7c-.3-4.5-3.3-7.8-7.5-7.8zm-53.6-7.8c-3.3 0-6.4 3.1-6.4 6.7l-3.9 145.3 3.9 66.9c.3 3.6 3.1 6.4 6.4 6.4 3.6 0 6.4-2.8 6.7-6.4l4.4-66.9-4.4-145.3c-.3-3.6-3.1-6.7-6.7-6.7zm26.7 3.4c-3.9 0-6.9 3.1-6.9 6.9L227 321.3l3.9 66.4c.3 3.9 3.1 6.9 6.9 6.9s6.9-3.1 6.9-6.9l4.2-66.4-4.2-141.7c0-3.9-3-6.9-6.9-6.9z"/></svg>',
		'spotify'        => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 496 512"><path d="M248 8C111.1 8 0 119.1 0 256s111.1 248 248 248 248-111.1 248-248S384.9 8 248 8zm100.7 364.9c-4.2 0-6.8-1.3-10.7-3.6-62.4-37.6-135-39.2-206.7-24.5-3.9 1-9 2.6-11.9 2.6-9.7 0-15.8-7.7-15.8-15.8 0-10.3 6.1-15.2 13.6-16.8 81.9-18.1 165.6-16.5 237 26.2 6.1 3.9 9.7 7.4 9.7 16.5s-7.1 15.4-15.2 15.4zm26.9-65.6c-5.2 0-8.7-2.3-12.3-4.2-62.5-37-155.7-51.9-238.6-29.4-4.8 1.3-7.4 2.6-11.9 2.6-10.7 0-19.4-8.7-19.4-19.4s5.2-17.8 15.5-20.7c27.8-7.8 56.2-13.6 97.8-13.6 64.9 0 127.6 16.1 177 45.5 8.1 4.8 11.3 11 11.3 19.7-.1 10.8-8.5 19.5-19.4 19.5zm31-76.2c-5.2 0-8.4-1.3-12.9-3.9-71.2-42.5-198.5-52.7-280.9-29.7-3.6 1-8.1 2.6-12.9 2.6-13.2 0-23.3-10.3-23.3-23.6 0-13.6 8.4-21.3 17.4-23.9 35.2-10.3 74.6-15.2 117.5-15.2 73 0 149.5 15.2 205.4 47.8 7.8 4.5 12.9 10.7 12.9 22.6 0 13.6-11 23.3-23.2 23.3z"/></svg>',
		'stackoverflow'  => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 384 512"><path d="M290.7 311L95 269.7 86.8 309l195.7 41zm51-87L188.2 95.7l-25.5 30.8 153.5 128.3zm-31.2 39.7L129.2 179l-16.7 36.5L293.7 300zM262 32l-32 24 119.3 160.3 32-24zm20.5 328h-200v39.7h200zm39.7 80H42.7V320h-40v160h359.5V320h-40z"/></svg>',
		'steam'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 496 512"><path d="M496 256c0 137-111.2 248-248.4 248-113.8 0-209.6-76.3-239-180.4l95.2 39.3c6.4 32.1 34.9 56.4 68.9 56.4 39.2 0 71.9-32.4 70.2-73.5l84.5-60.2c52.1 1.3 95.8-40.9 95.8-93.5 0-51.6-42-93.5-93.7-93.5s-93.7 42-93.7 93.5v1.2L176.6 279c-15.5-.9-30.7 3.4-43.5 12.1L0 236.1C10.2 108.4 117.1 8 247.6 8 384.8 8 496 119 496 256zM155.7 384.3l-30.5-12.6a52.79 52.79 0 0 0 27.2 25.8c26.9 11.2 57.8-1.6 69-28.4 5.4-13 5.5-27.3.1-40.3-5.4-13-15.5-23.2-28.5-28.6-12.9-5.4-26.7-5.2-38.9-.6l31.5 13c19.8 8.2 29.2 30.9 20.9 50.7-8.3 19.9-31 29.2-50.8 21zm173.8-129.9c-34.4 0-62.4-28-62.4-62.3s28-62.3 62.4-62.3 62.4 28 62.4 62.3-27.9 62.3-62.4 62.3zm.1-15.6c25.9 0 46.9-21 46.9-46.8 0-25.9-21-46.8-46.9-46.8s-46.9 21-46.9 46.8c.1 25.8 21.1 46.8 46.9 46.8z"/></svg>',
		'stumbleupon'    => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M502.9 266v69.7c0 62.1-50.3 112.4-112.4 112.4-61.8 0-112.4-49.8-112.4-111.3v-70.2l34.3 16 51.1-15.2V338c0 14.7 12 26.5 26.7 26.5S417 352.7 417 338v-72h85.9zm-224.7-58.2l34.3 16 51.1-15.2V173c0-60.5-51.1-109-112.1-109-60.8 0-112.1 48.2-112.1 108.2v162.4c0 14.9-12 26.7-26.7 26.7S86 349.5 86 334.6V266H0v69.7C0 397.7 50.3 448 112.4 448c61.6 0 112.4-49.5 112.4-110.8V176.9c0-14.7 12-26.7 26.7-26.7s26.7 12 26.7 26.7v30.9z"/></svg>',
		'telegram'       => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 496 512"><path d="M248,8C111.033,8,0,119.033,0,256S111.033,504,248,504,496,392.967,496,256,384.967,8,248,8ZM362.952,176.66c-3.732,39.215-19.881,134.378-28.1,178.3-3.476,18.584-10.322,24.816-16.948,25.425-14.4,1.326-25.338-9.517-39.287-18.661-21.827-14.308-34.158-23.215-55.346-37.177-24.485-16.135-8.612-25,5.342-39.5,3.652-3.793,67.107-61.51,68.335-66.746.153-.655.3-3.1-1.154-4.384s-3.59-.849-5.135-.5q-3.283.746-104.608,69.142-14.845,10.194-26.894,9.934c-8.855-.191-25.888-5.006-38.551-9.123-15.531-5.048-27.875-7.717-26.8-16.291q.84-6.7,18.45-13.7,108.446-47.248,144.628-62.3c68.872-28.647,83.183-33.623,92.511-33.789,2.052-.034,6.639.474,9.61,2.885a10.452,10.452,0,0,1,3.53,6.716A43.765,43.765,0,0,1,362.952,176.66Z"/></svg>',
		'tiktok'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M448,209.91a210.06,210.06,0,0,1-122.77-39.25V349.38A162.55,162.55,0,1,1,185,188.31V278.2a74.62,74.62,0,1,0,52.23,71.18V0l88,0a121.18,121.18,0,0,0,1.86,22.17h0A122.18,122.18,0,0,0,381,102.39a121.43,121.43,0,0,0,67,20.14Z"/></svg>',
		'tumblr'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 512"><path d="M309.8 480.3c-13.6 14.5-50 31.7-97.4 31.7-120.8 0-147-88.8-147-140.6v-144H17.9c-5.5 0-10-4.5-10-10v-68c0-7.2 4.5-13.6 11.3-16 62-21.8 81.5-76 84.3-117.1.8-11 6.5-16.3 16.1-16.3h70.9c5.5 0 10 4.5 10 10v115.2h83c5.5 0 10 4.4 10 9.9v81.7c0 5.5-4.5 10-10 10h-83.4V360c0 34.2 23.7 53.6 68 35.8 4.8-1.9 9-3.2 12.7-2.2 3.5.9 5.8 3.4 7.4 7.9l22 64.3c1.8 5 3.3 10.6-.4 14.5z"/></svg>',
		'twitch'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M391.17,103.47H352.54v109.7h38.63ZM285,103H246.37V212.75H285ZM120.83,0,24.31,91.42V420.58H140.14V512l96.53-91.42h77.25L487.69,256V0ZM449.07,237.75l-77.22,73.12H294.61l-67.6,64v-64H140.14V36.58H449.07Z"/></svg>',
		'viber'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M444 49.9C431.3 38.2 379.9.9 265.3.4c0 0-135.1-8.1-200.9 52.3C27.8 89.3 14.9 143 13.5 209.5c-1.4 66.5-3.1 191.1 117 224.9h.1l-.1 51.6s-.8 20.9 13 25.1c16.6 5.2 26.4-10.7 42.3-27.8 8.7-9.4 20.7-23.2 29.8-33.7 82.2 6.9 145.3-8.9 152.5-11.2 16.6-5.4 110.5-17.4 125.7-142 15.8-128.6-7.6-209.8-49.8-246.5zM457.9 287c-12.9 104-89 110.6-103 115.1-6 1.9-61.5 15.7-131.2 11.2 0 0-52 62.7-68.2 79-5.3 5.3-11.1 4.8-11-5.7 0-6.9.4-85.7.4-85.7-.1 0-.1 0 0 0-101.8-28.2-95.8-134.3-94.7-189.8 1.1-55.5 11.6-101 42.6-131.6 55.7-50.5 170.4-43 170.4-43 96.9.4 143.3 29.6 154.1 39.4 35.7 30.6 53.9 103.8 40.6 211.1zm-139-80.8c.4 8.6-12.5 9.2-12.9.6-1.1-22-11.4-32.7-32.6-33.9-8.6-.5-7.8-13.4.7-12.9 27.9 1.5 43.4 17.5 44.8 46.2zm20.3 11.3c1-42.4-25.5-75.6-75.8-79.3-8.5-.6-7.6-13.5.9-12.9 58 4.2 88.9 44.1 87.8 92.5-.1 8.6-13.1 8.2-12.9-.3zm47 13.4c.1 8.6-12.9 8.7-12.9.1-.6-81.5-54.9-125.9-120.8-126.4-8.5-.1-8.5-12.9 0-12.9 73.7.5 133 51.4 133.7 139.2zM374.9 329v.2c-10.8 19-31 40-51.8 33.3l-.2-.3c-21.1-5.9-70.8-31.5-102.2-56.5-16.2-12.8-31-27.9-42.4-42.4-10.3-12.9-20.7-28.2-30.8-46.6-21.3-38.5-26-55.7-26-55.7-6.7-20.8 14.2-41 33.3-51.8h.2c9.2-4.8 18-3.2 23.9 3.9 0 0 12.4 14.8 17.7 22.1 5 6.8 11.7 17.7 15.2 23.8 6.1 10.9 2.3 22-3.7 26.6l-12 9.6c-6.1 4.9-5.3 14-5.3 14s17.8 67.3 84.3 84.3c0 0 9.1.8 14-5.3l9.6-12c4.6-6 15.7-9.8 26.6-3.7 14.7 8.3 33.4 21.2 45.8 32.9 7 5.7 8.6 14.4 3.8 23.6z"/></svg>',
		'vk'             => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M31.4907 63.4907C0 94.9813 0 145.671 0 247.04V264.96C0 366.329 0 417.019 31.4907 448.509C62.9813 480 113.671 480 215.04 480H232.96C334.329 480 385.019 480 416.509 448.509C448 417.019 448 366.329 448 264.96V247.04C448 145.671 448 94.9813 416.509 63.4907C385.019 32 334.329 32 232.96 32H215.04C113.671 32 62.9813 32 31.4907 63.4907ZM75.6 168.267H126.747C128.427 253.76 166.133 289.973 196 297.44V168.267H244.16V242C273.653 238.827 304.64 205.227 315.093 168.267H363.253C359.313 187.435 351.46 205.583 340.186 221.579C328.913 237.574 314.461 251.071 297.733 261.227C316.41 270.499 332.907 283.63 346.132 299.751C359.357 315.873 369.01 334.618 374.453 354.747H321.44C316.555 337.262 306.614 321.61 292.865 309.754C279.117 297.899 262.173 290.368 244.16 288.107V354.747H238.373C136.267 354.747 78.0267 284.747 75.6 168.267Z"/></svg>',
		'weixin'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path d="M385.2 167.6c6.4 0 12.6.3 18.8 1.1C387.4 90.3 303.3 32 207.7 32 100.5 32 13 104.8 13 197.4c0 53.4 29.3 97.5 77.9 131.6l-19.3 58.6 68-34.1c24.4 4.8 43.8 9.7 68.2 9.7 6.2 0 12.1-.3 18.3-.8-4-12.9-6.2-26.6-6.2-40.8-.1-84.9 72.9-154 165.3-154zm-104.5-52.9c14.5 0 24.2 9.7 24.2 24.4 0 14.5-9.7 24.2-24.2 24.2-14.8 0-29.3-9.7-29.3-24.2.1-14.7 14.6-24.4 29.3-24.4zm-136.4 48.6c-14.5 0-29.3-9.7-29.3-24.2 0-14.8 14.8-24.4 29.3-24.4 14.8 0 24.4 9.7 24.4 24.4 0 14.6-9.6 24.2-24.4 24.2zM563 319.4c0-77.9-77.9-141.3-165.4-141.3-92.7 0-165.4 63.4-165.4 141.3S305 460.7 397.6 460.7c19.3 0 38.9-5.1 58.6-9.9l53.4 29.3-14.8-48.6C534 402.1 563 363.2 563 319.4zm-219.1-24.5c-9.7 0-19.3-9.7-19.3-19.6 0-9.7 9.7-19.3 19.3-19.3 14.8 0 24.4 9.7 24.4 19.3 0 10-9.7 19.6-24.4 19.6zm107.1 0c-9.7 0-19.3-9.7-19.3-19.6 0-9.7 9.7-19.3 19.3-19.3 14.5 0 24.4 9.7 24.4 19.3.1 10-9.9 19.6-24.4 19.6z"/></svg>',
		'weibo'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M407 177.6c7.6-24-13.4-46.8-37.4-41.7-22 4.8-28.8-28.1-7.1-32.8 50.1-10.9 92.3 37.1 76.5 84.8-6.8 21.2-38.8 10.8-32-10.3zM214.8 446.7C108.5 446.7 0 395.3 0 310.4c0-44.3 28-95.4 76.3-143.7C176 67 279.5 65.8 249.9 161c-4 13.1 12.3 5.7 12.3 6 79.5-33.6 140.5-16.8 114 51.4-3.7 9.4 1.1 10.9 8.3 13.1 135.7 42.3 34.8 215.2-169.7 215.2zm143.7-146.3c-5.4-55.7-78.5-94-163.4-85.7-84.8 8.6-148.8 60.3-143.4 116s78.5 94 163.4 85.7c84.8-8.6 148.8-60.3 143.4-116zM347.9 35.1c-25.9 5.6-16.8 43.7 8.3 38.3 72.3-15.2 134.8 52.8 111.7 124-7.4 24.2 29.1 37 37.4 12 31.9-99.8-55.1-195.9-157.4-174.3zm-78.5 311c-17.1 38.8-66.8 60-109.1 46.3-40.8-13.1-58-53.4-40.3-89.7 17.7-35.4 63.1-55.4 103.4-45.1 42 10.8 63.1 50.2 46 88.5zm-86.3-30c-12.9-5.4-30 .3-38 12.9-8.3 12.9-4.3 28 8.6 34 13.1 6 30.8.3 39.1-12.9 8-13.1 3.7-28.3-9.7-34zm32.6-13.4c-5.1-1.7-11.4.6-14.3 5.4-2.9 5.1-1.4 10.6 3.7 12.9 5.1 2 11.7-.3 14.6-5.4 2.8-5.2 1.1-10.9-4-12.9z"/></svg>',
		'whatsapp'       => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M380.9 97.1C339 55.1 283.2 32 223.9 32c-122.4 0-222 99.6-222 222 0 39.1 10.2 77.3 29.6 111L0 480l117.7-30.9c32.4 17.7 68.9 27 106.1 27h.1c122.3 0 224.1-99.6 224.1-222 0-59.3-25.2-115-67.1-157zm-157 341.6c-33.2 0-65.7-8.9-94-25.7l-6.7-4-69.8 18.3L72 359.2l-4.4-7c-18.5-29.4-28.2-63.3-28.2-98.2 0-101.7 82.8-184.5 184.6-184.5 49.3 0 95.6 19.2 130.4 54.1 34.8 34.9 56.2 81.2 56.1 130.5 0 101.8-84.9 184.6-186.6 184.6zm101.2-138.2c-5.5-2.8-32.8-16.2-37.9-18-5.1-1.9-8.8-2.8-12.5 2.8-3.7 5.6-14.3 18-17.6 21.8-3.2 3.7-6.5 4.2-12 1.4-32.6-16.3-54-29.1-75.5-66-5.7-9.8 5.7-9.1 16.3-30.3 1.8-3.7.9-6.9-.5-9.7-1.4-2.8-12.5-30.1-17.1-41.2-4.5-10.8-9.1-9.3-12.5-9.5-3.2-.2-6.9-.2-10.6-.2-3.7 0-9.7 1.4-14.8 6.9-5.1 5.6-19.4 19-19.4 46.3 0 27.3 19.9 53.7 22.6 57.4 2.8 3.7 39.1 59.7 94.8 83.8 35.2 15.2 49 16.5 66.6 13.9 10.7-1.6 32.8-13.4 37.4-26.4 4.6-13 4.6-24.1 3.2-26.4-1.3-2.5-5-3.9-10.5-6.6z"/></svg>',
		'wordpress'      => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M61.7 169.4l101.5 278C92.2 413 43.3 340.2 43.3 256c0-30.9 6.6-60.1 18.4-86.6zm337.9 75.9c0-26.3-9.4-44.5-17.5-58.7-10.8-17.5-20.9-32.4-20.9-49.9 0-19.6 14.8-37.8 35.7-37.8.9 0 1.8.1 2.8.2-37.9-34.7-88.3-55.9-143.7-55.9-74.3 0-139.7 38.1-177.8 95.9 5 .2 9.7.3 13.7.3 22.2 0 56.7-2.7 56.7-2.7 11.5-.7 12.8 16.2 1.4 17.5 0 0-11.5 1.3-24.3 2l77.5 230.4L249.8 247l-33.1-90.8c-11.5-.7-22.3-2-22.3-2-11.5-.7-10.1-18.2 1.3-17.5 0 0 35.1 2.7 56 2.7 22.2 0 56.7-2.7 56.7-2.7 11.5-.7 12.8 16.2 1.4 17.5 0 0-11.5 1.3-24.3 2l76.9 228.7 21.2-70.9c9-29.4 16-50.5 16-68.7zm-139.9 29.3l-63.8 185.5c19.1 5.6 39.2 8.7 60.1 8.7 24.8 0 48.5-4.3 70.6-12.1-.6-.9-1.1-1.9-1.5-2.9l-65.4-179.2zm183-120.7c.9 6.8 1.4 14 1.4 21.9 0 21.6-4 45.8-16.2 76.2l-65 187.9C426.2 403 468.7 334.5 468.7 256c0-37-9.4-71.8-26-102.1zM504 256c0 136.8-111.3 248-248 248C119.2 504 8 392.7 8 256 8 119.2 119.2 8 256 8c136.7 0 248 111.2 248 248zm-11.4 0c0-130.5-106.2-236.6-236.6-236.6C125.5 19.4 19.4 125.5 19.4 256S125.6 492.6 256 492.6c130.5 0 236.6-106.1 236.6-236.6z"/></svg>',
		'yelp'           => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 384 512"><path d="M42.9 240.32l99.62 48.61c19.2 9.4 16.2 37.51-4.5 42.71L30.5 358.45a22.79 22.79 0 0 1-28.21-19.6 197.16 197.16 0 0 1 9-85.32 22.8 22.8 0 0 1 31.61-13.21zm44 239.25a199.45 199.45 0 0 0 79.42 32.11A22.78 22.78 0 0 0 192.94 490l3.9-110.82c.7-21.3-25.5-31.91-39.81-16.1l-74.21 82.4a22.82 22.82 0 0 0 4.09 34.09zm145.34-109.92l58.81 94a22.93 22.93 0 0 0 34 5.5 198.36 198.36 0 0 0 52.71-67.61A23 23 0 0 0 364.17 370l-105.42-34.26c-20.31-6.5-37.81 15.8-26.51 33.91zm148.33-132.23a197.44 197.44 0 0 0-50.41-69.31 22.85 22.85 0 0 0-34 4.4l-62 91.92c-11.9 17.7 4.7 40.61 25.2 34.71L366 268.63a23 23 0 0 0 14.61-31.21zM62.11 30.18a22.86 22.86 0 0 0-9.9 32l104.12 180.44c11.7 20.2 42.61 11.9 42.61-11.4V22.88a22.67 22.67 0 0 0-24.5-22.8 320.37 320.37 0 0 0-112.33 30.1z"/></svg>',
		'rss'            => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><!--! Font Awesome Pro 6.4.0 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license (Commercial License) Copyright 2023 Fonticons, Inc. --><path d="M0 64C0 46.3 14.3 32 32 32c229.8 0 416 186.2 416 416c0 17.7-14.3 32-32 32s-32-14.3-32-32C384 253.6 226.4 96 32 96C14.3 96 0 81.7 0 64zM0 416a64 64 0 1 1 128 0A64 64 0 1 1 0 416zM32 160c159.1 0 288 128.9 288 288c0 17.7-14.3 32-32 32s-32-14.3-32-32c0-123.7-100.3-224-224-224c-17.7 0-32-14.3-32-32s14.3-32 32-32z"/></svg>',
		'email'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><!--! Font Awesome Pro 6.4.0 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license (Commercial License) Copyright 2023 Fonticons, Inc. --><path d="M48 64C21.5 64 0 85.5 0 112c0 15.1 7.1 29.3 19.2 38.4L236.8 313.6c11.4 8.5 27 8.5 38.4 0L492.8 150.4c12.1-9.1 19.2-23.3 19.2-38.4c0-26.5-21.5-48-48-48H48zM0 176V384c0 35.3 28.7 64 64 64H448c35.3 0 64-28.7 64-64V176L294.4 339.2c-22.8 17.1-54 17.1-76.8 0L0 176z"/></svg>',
		'phone'          => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><!--! Font Awesome Pro 6.4.0 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license (Commercial License) Copyright 2023 Fonticons, Inc. --><path d="M164.9 24.6c-7.7-18.6-28-28.5-47.4-23.2l-88 24C12.1 30.2 0 46 0 64C0 311.4 200.6 512 448 512c18 0 33.8-12.1 38.6-29.5l24-88c5.3-19.4-4.6-39.7-23.2-47.4l-96-40c-16.3-6.8-35.2-2.1-46.3 11.6L304.7 368C234.3 334.7 177.3 277.7 144 207.3L193.3 167c13.7-11.2 18.4-30 11.6-46.3l-40-96z"/></svg>',
		'windows' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M0 93.7l183.6-25.3v177.4H0V93.7zm0 324.6l183.6 25.3V268.4H0v149.9zm203.8 28L448 480V268.4H203.8v177.9zm0-380.6v180.1H448V32L203.8 65.7z"/></svg>',
		'deviantart' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 320 512"><path d="M320 93.2l-98.2 179.1 7.4 9.5H320v127.7H159.1l-13.5 9.2-43.7 84c-.3 0-8.6 8.6-9.2 9.2H0v-93.2l93.2-179.4-7.4-9.2H0V102.5h156l13.5-9.2 43.7-84c.3 0 8.6-8.6 9.2-9.2H320v93.1z"/></svg>',
		'digg' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M81.7 172.3H0v174.4h132.7V96h-51v76.3zm0 133.4H50.9v-92.3h30.8v92.3zm297.2-133.4v174.4h81.8v28.5h-81.8V416H512V172.3H378.9zm81.8 133.4h-30.8v-92.3h30.8v92.3zm-235.6 41h82.1v28.5h-82.1V416h133.3V172.3H225.1v174.4zm51.2-133.3h30.8v92.3h-30.8v-92.3zM153.3 96h51.3v51h-51.3V96zm0 76.3h51.3v174.4h-51.3V172.3z"/></svg>'
	];
}

function sf_social_svg_icon( $name = '' ) {
	$svg_icons = sf_get_social_svg_icons();
	if ( isset( $svg_icons[$name] ) ) {
		echo $svg_icons[$name];
	}
}

/**
 * Get social icons data
 *
 * @param array $links social links
 * @return array
 */
function sf_get_social_icons_data( $links = [] ) {

	if ( empty( $links ) ) {
		return [];
	}

	$icons_data = [];
	$icon_names = sf_get_social_icon_names();
	$svg_icons  = sf_get_social_svg_icons();

	foreach ( $links as $key => $url ) {
		$icons_data[] = [
			'icon'  => @$svg_icons[$key],
			'class' => 's-' . $key,
			'url'   => $url,
			'title' => @$icon_names[$key],
		];
	}

	return $icons_data;
}

/**
 * Output or return social icons
 *
 * @param array $social_icons
 * @param string $wrapper_class
 * @param boolean $echo
 * @return void|string
 */
function sf_social_icons( $social_icons = [], $wrapper_class = '', $echo = true ) {
	$out = '';
	if ( ! empty( $social_icons ) ) {
		$out = '<div class="' . esc_attr( $wrapper_class ) . '">';
		$out .= '<div class="social-list">';
		foreach ( $social_icons as $icon ) {
			$out .= '<a href="' . esc_url( $icon['url'] ) . '" class="' . esc_attr( $icon['class'] ) . '" title="' . esc_attr( $icon['title'] ) . '" target="_blank" rel="noopener noreferrer nofollow">' . $icon['icon'] . '</a>';
		}
		$out .= '</div>';
		$out .= '</div>';
	}

	if ( $echo ) {
		echo $out;
	} else {
		return $out;
	}
}

/**
 * Get post title link from post id
 * 
 * @param  int     $post_id  post id
 * @param  array   $atts     attributes
 * @return string            html
 */
function sf_get_post_title_link( $post_id = null, $atts = array() ) {

	$out = $attributes = '';
	
	if ( $post_id && is_string( get_post_status( $post_id ) ) ) {
		
		if ( !empty( $atts ) && is_array( $atts ) ) {
			foreach ( $atts as $name => $value ) {
				$attributes .= $name . '="' . esc_attr( $value ) . '" ';
			}
		}

		$out .= '<a href="'. esc_url( get_the_permalink( $post_id ) ) .'" ' . $attributes . '>'. get_the_title( $post_id ) .'</a>';
	}

	return $out;
}

/**
 * Check if reCaptcha is active
 * 
 * @return Boolean
 */
function sf_is_captcha_enabled() {
	$recaptcha  = SF()->get_setting( 'recaptcha' );
	$site_key   = SF()->get_setting( 'recaptcha_site_key' );
	$secret_key = SF()->get_setting( 'recaptcha_secret_key' );
	return $recaptcha && ! empty( $site_key ) && ! empty( $secret_key );
}

/**
 * Verify reCaptcha
 *
 * @param string $token
 * @return bool
 */
function sf_is_human( $token, $action = '' ) {
	$is_human = false;

	if ( empty( $token ) ) {
		return $is_human;
	}

	$source   = SF()->get_setting( 'recaptcha_source' );
	$source   = ! empty( $source ) ? $source : 'google.com';
	$source   = apply_filters( 'sf_recaptcha_source', 'google.com' );
	$version  = SF()->get_setting( 'recaptcha_version' );
	$endpoint = sprintf( 'https://www.%s/recaptcha/api/siteverify', $source );
	$request  = array(
		'body' => array(
			'secret'   => SF()->get_setting( 'recaptcha_secret_key' ),
			'response' => $token,
			'remoteip' => isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ? $_SERVER['HTTP_X_FORWARDED_FOR'] : $_SERVER['REMOTE_ADDR'],
		)
	);
	$response = wp_safe_remote_post( esc_url_raw( $endpoint ), $request );
	
	if ( 200 != wp_remote_retrieve_response_code( $response ) ) {
		return $is_human;
	}

	$response_body = wp_remote_retrieve_body( $response );
	$result        = json_decode( $response_body, true );
	$is_human      = isset( $result['success'] ) && true == $result['success'];

	if ( $is_human && '3' === $version ) {
		$score_threshold = SF()->get_setting( 'recaptcha_v3_threshold' );
		$score_threshold = '' === $score_threshold ? 0.5 : $score_threshold;
		$is_human = $result['action'] == $action && $result['score'] >= $score_threshold;
	}

	return apply_filters( 'sf_recaptcha_verify_response', $is_human, $response_body );
}

/**
 * Print reCaptcha widget
 */
function sf_output_form_captcha_widget() {
    $site_key = SF()->get_setting( 'recaptcha_site_key' );
    $version  = SF()->get_setting( 'recaptcha_version' );
    if ( '2' === $version ) {
        echo '<div class="g-recaptcha" data-sitekey="'. esc_attr( $site_key ) .'"></div>';
    } else {
        echo '<input type="hidden" name="sf_recaptcha_key" value="' . esc_attr( $site_key ) . '" />';
    }
}

/**
 * Print preloader
 */
function sf_output_preloader() {
    ?>
    <div class="preloader">
        <svg class="google-circular" viewBox="25 25 50 50">
            <circle class="path" cx="50" cy="50" r="20" fill="none" stroke-width="4" stroke-miterlimit="10"/>
        </svg>
    </div>
    <?php
}

/**
 * Get column class
 * @param  array $settings  widget settings
 * @param  string $additional_classes  
 * @return string           column class
 */
function sf_get_column_class( $settings, $additional_classes = '' ) {
	$classes = $additional_classes;

	if ( ! isset( $settings['columns_tablet'] ) ) {
		$settings['columns_tablet'] = 2;
	}
	if ( ! isset( $settings['columns_mobile'] ) ) {
		$settings['columns_mobile'] = 1;
	}

	switch ( $settings['columns'] ) {
		case 1: $classes .= ' col-lg-12'; break;
		case 2: $classes .= ' col-lg-6'; break;
		case 3: $classes .= ' col-lg-4'; break;
		case 4: $classes .= ' col-lg-3'; break;
		case 6: $classes .= ' col-lg-2'; break;
		default: $classes .= ' col-lg-4'; break;
	}
	switch ( $settings['columns_tablet'] ) {
		case 1: $classes .= ' col-md-12'; break;
		case 2: $classes .= ' col-md-6'; break;
		case 3: $classes .= ' col-md-4'; break;
		case 4: $classes .= ' col-md-3'; break;
		case 6: $classes .= ' col-md-2'; break;
		default: $classes .= ' col-md-6'; break;
	}
	switch ( $settings['columns_mobile'] ) {
		case 1: $classes .= ' col-12'; break;
		case 2: $classes .= ' col-6'; break;
		case 3: $classes .= ' col-4'; break;
		case 4: $classes .= ' col-3'; break;
		case 6: $classes .= ' col-2'; break;
		default: break;
	}

	return apply_filters( 'sf_get_column_class', trim ( $classes ) );
}

/**
 * Get row class
 * @param  array $settings  widget settings
 * @return string           row class
 */
function sf_get_row_class( $settings ) {
	$classes = 'row';

	if ( isset( $settings['columns_xl'] ) ) {
		switch ( $settings['columns_xl'] ) {
			case 1: $classes .= ' row-cols-xl-1'; break;
			case 2: $classes .= ' row-cols-xl-2'; break;
			case 3: $classes .= ' row-cols-xl-3'; break;
			case 4: $classes .= ' row-cols-xl-4'; break;
			case 5: $classes .= ' row-cols-xl-5'; break;
			case 6: $classes .= ' row-cols-xl-6'; break;
		}
	}

	switch ( $settings['columns'] ) {
		case 1: $classes .= ' row-cols-lg-1'; break;
		case 2: $classes .= ' row-cols-lg-2'; break;
		case 3: $classes .= ' row-cols-lg-3'; break;
		case 4: $classes .= ' row-cols-lg-4'; break;
		case 5: $classes .= ' row-cols-lg-5'; break;
		case 6: $classes .= ' row-cols-lg-6'; break;
		default: $classes .= ' row-cols-lg-3'; break;
	}

	if ( isset( $settings['columns_tablet'] ) ) {
		switch ( $settings['columns_tablet'] ) {
			case 1: $classes .= ' row-cols-sm-1'; break;
			case 2: $classes .= ' row-cols-sm-2'; break;
			case 3: $classes .= ' row-cols-sm-3'; break;
			case 4: $classes .= ' row-cols-sm-4'; break;
			case 5: $classes .= ' row-cols-sm-5'; break;
			case 6: $classes .= ' row-cols-sm-6'; break;
			default: $classes .= ' row-cols-sm-2'; break;
		}
	}

	if ( isset( $settings['columns_mobile'] ) ) {
		switch ( $settings['columns_mobile'] ) {
			case 1: $classes .= ' row-cols-1'; break;
			case 2: $classes .= ' row-cols-2'; break;
			case 3: $classes .= ' row-cols-3'; break;
			case 4: $classes .= ' row-cols-4'; break;
			case 5: $classes .= ' row-cols-5'; break;
			case 6: $classes .= ' row-cols-6'; break;
			default: $classes .= ' row-cols-1'; break;
		}
	}
	return apply_filters( 'sf_get_row_class', $classes );
}

/**
 * Return or output form icon
 *
 * @param string $key
 * @param array $args
 * @return void|string
 */
function sf_form_icon( $key = '', $attributes = [], $tag = 'i', $echo = true ) {
	$icon_class = sf_get_form_icon_class( $key );
	$return     = '';

	if ( ! empty( $icon_class ) ) {
		if ( empty( $attributes['class'] ) ) {
			$attributes['class'] = $icon_class;
		} else {
			if ( is_array( $attributes['class'] ) ) {
				$attributes['class'][] = implode( ' ', $attributes['class'] ) . ' '. $icon_class;
			} else {
				$attributes['class'] .= ' ' . $icon_class;
			}
		}

		$html_attrs = '';
		foreach ( $attributes as $name => $value ) {
			$html_attrs .= ' ' . $name . '="' . esc_attr( $value ) . '"';
		}

		$return = sprintf( '<%1$s %2$s aria-hidden="true"></%1$s>', $tag, $html_attrs );
	}

	if ( $echo ) {
		echo $return;
	} else {
		return $return;
	}
}

function sf_get_data_animation_settings( $settings ) {
	if ( empty( $settings['animation'] ) ) {
		return false;
	}

	$animation 		 = $settings['animation'];
	$animation_speed = isset( $settings['animation_speed'] ) ? $settings['animation_speed'] : '';
	$animation_delay = isset( $settings['animation_delay'] ) ? $settings['animation_delay'] : '';
	$animation_on 	 = isset( $settings['animation_on'] ) ? $settings['animation_on'] : [];

	switch ( $animation ) {
		case 'fadeInLeftRight':
			$animation     = str_replace( 'Right', '', $settings['animation'] );
			$animation_alt = str_replace( 'Left', '', $settings['animation'] );
			break;
		case 'fadeInRightLeft':
			$animation     = str_replace( 'Left', '', $settings['animation'] );
			$animation_alt = str_replace( 'Right', '', $settings['animation'] );
			break;
		case 'fadeInUpDown':
			$animation     = str_replace( 'Down', '', $settings['animation'] );
			$animation_alt = str_replace( 'Up', '', $settings['animation'] );
			break;
		case 'fadeInDownUp':
			$animation     = str_replace( 'Up', '', $settings['animation'] );
			$animation_alt = str_replace( 'Down', '', $settings['animation'] );
			break;
	}

	$animation_alt = ! empty( $animation_alt ) ? 'anim-' . $animation_alt : '';

	return wp_json_encode( [
		'animation'       => 'anim-' . $animation,
		'animation_alt'   => $animation_alt,
		'animation_speed' => $animation_speed,
		'animation_delay' => $animation_delay,
		'animation_on' 	  => $animation_on
	] );
}

/**
 * Get attachment id from url
 *
 * @param string $url
 * @return int
 */
function sf_get_attachment_id_from_url( $url ) {
	global $wpdb;
	$attachment_id = false;

	if ( ! empty( $url ) ) {
		$upload_dir_paths = wp_upload_dir();

		// Make sure the upload path base directory exists in the attachment URL, to verify that we're working with a media library image.
		if ( false !== strpos( $url, $upload_dir_paths['baseurl'] ) ) {

			// If this is the URL of an auto-generated thumbnail, get the URL of the original image.
			$url = preg_replace( '/-\d+x\d+(?=\.(jpg|jpeg|png|gif)$)/i', '', $url );

			// Remove the upload path base directory from the attachment URL.
			$url = str_replace( $upload_dir_paths['baseurl'] . '/', '', $url );

			// Finally, run a custom database query to get the attachment ID from the modified attachment URL.
			$attachment_id = $wpdb->get_var( $wpdb->prepare( "SELECT wposts.ID FROM $wpdb->posts wposts, $wpdb->postmeta wpostmeta WHERE wposts.ID = wpostmeta.post_id AND wpostmeta.meta_key = '_wp_attached_file' AND wpostmeta.meta_value = '%s' AND wposts.post_type = 'attachment'", $url ) );
		}
	}
	return $attachment_id;
}