<?php
/**
 * SF Fonts
 *
 * @package     classes
 * @category    fonts
 * @author      ThemeSpirit
 * @version     1.0.0
 */

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.
class SF_Fonts {

	/**
	 * Theme fonts option name
	 * @var string
	 */
	const OPTION_NAME = 'sf_fonts';

	/**
	 * Custom fonts option name
	 * @var string
	 */
	const CUSTOM_FONTS_OPTION_NAME = 'sf_custom_fonts';

	/**
	 * Adobe fonts option names
	 * @var string
	 */
	const ADOBE_FONTS_OPTION_NAME = 'sf_adobe_fonts';
	const ADOBE_FONTS_KIT_ID_OPTION_NAME = 'sf_adobe_fonts_kit_id';

	/**
	 * Adobe fonts link
	 * @var string
	 */	
	const ADOBE_FONTS_LINK = 'https://use.typekit.net/%s.css';

	/**
	 * Google fonts array
	 * @var array
	 */
	static $google_fonts = array();

	/**
	 * default fonts used by the theme
	 * @var array
	 */
	static $default_fonts = array(
		array(
			'family' => 'Poppins',
			'variants' => '300,regular,italic,600,600italic,700',
			'source' => 'Google'
		)
	);

	/**
	 * Add theme default fonts
	 */
	static function add_default_fonts() {
		if ( empty( get_option( self::OPTION_NAME ) ) ) {
			self::$default_fonts = apply_filters( 'sf_config_theme_fonts', self::$default_fonts );
			update_option( self::OPTION_NAME, self::$default_fonts );
		}
	}

	/**
	 * Get theme fonts
	 * @return array
	 */
	static function get_fonts() {
		$fonts = get_option( self::OPTION_NAME );

		if ( ! empty( $fonts ) ) {
			return $fonts;
		}
		return array();
	}

	/**
	 * Get custom fonts
	 * @return array
	 */
	static function get_custom_fonts() {
		$fonts = get_option( self::CUSTOM_FONTS_OPTION_NAME );

		if ( ! empty( $fonts ) ) {
			return $fonts;
		}
		return [];
	}

	/**
	 * Get Google fonts data from a json file
	 * @return array
	 */
	static function get_google_fonts() {
		if ( empty( self::$google_fonts ) ) {
			ob_start();
			include( SF_FRAMEWORK_DIR . 'includes/libs/kirki/kirki-packages/googlefonts/src/webfonts.json' );
			$fonts_json = ob_get_clean();
			$fonts = json_decode( $fonts_json, true );
			self::$google_fonts = $fonts['items'];
		}
		return self::$google_fonts;
	}

	/**
	 * Get font variants
	 * @param  string $family font name
	 * @return array          font variants
	 */
	static function get_variants( $font_family = '' ) {
	    if ( $font_family === '' ) {
	        return array();
	    }

	    $fonts = self::get_google_fonts();
	    
	    if ( isset( $fonts[$font_family]['variants'] ) ) {
	        return $fonts[$font_family]['variants'];
	    }
	    return array();
	}

	static function get_adobe_fonts_kit_id() {
		$kit_id = get_option( self::ADOBE_FONTS_KIT_ID_OPTION_NAME, false );	
		return ( false !== $kit_id ? $kit_id : '' );
	}

	static function get_adobe_fonts() {
		$fonts = get_option( self::ADOBE_FONTS_OPTION_NAME, false );	
		return ( false !== $fonts ? (array) $fonts : [] );
	}

	static function get_adobe_fonts_css() {
		$kit_id = self::get_adobe_fonts_kit_id();
		if ( ! empty( $kit_id ) ) {
			return sprintf( SF_Fonts::ADOBE_FONTS_LINK, $kit_id );
		}
		return '';
	}

	static function get_normalized_adobe_fonts_variants( $variants ) {
		$return = [];
		foreach ( $variants as $variant ) {
			$return[] = str_replace(
				[ 'n1', 'n2', 'n3', 'n4', 'n5', 'n6', 'n7', 'n8', 'n9', 'i1', 'i2', 'i3', 'i4', 'i5', 'i6', 'i7', 'i8', 'i9' ],
				[ '100', '200', '300', 'regular', '500', '600', '700', '800', '900', '100italic', '200italic', '300italic', 'italic', '500italic', '600italic', '700italic', '800italic', '900italic' ],
				$variant
			);
		}
		return $return;
	}

	/**
	 * Convert a variant to standard format
	 * @param  string $variant string
	 * @return string
	 */
	static function to_variant( $variant = '' ) {
		if ( empty( $variant ) ) {
			return '';
		}
		$style = preg_replace( '/[0-9\s]/', '', $variant );
		
		if ( $style != 'Italic' ) {
			return preg_replace( '/\D/', '', $variant );
		} else {
			return preg_replace( '/\s/', '', $variant );
		}
	}

	/**
	 * Get custom font CSS style
	 * @param  string $font_family font name
	 * @return string              font face css
	 */
	static function get_custom_font_css( $font_family = null ) {
		$custom_fonts = get_option( self::CUSTOM_FONTS_OPTION_NAME );
		$output_css = '';

		foreach ( $custom_fonts as $font ) {
			if ( $font_family == $font['family'] ) {
				$output_css .= '@font-face{font-family:"' . $font['family'] . '";';
				if ( !empty( $font['eot'] ) ) {
					$output_css .= 'src:url("' . esc_url( $font['eot'] ) . '");';
				}
				if ( !empty( $font['woff2'] ) ) {
					$output_css .='src:url("' . esc_url( $font['woff2'] ) . '") format("woff2"),';
				}
				if ( !empty( $font['woff'] ) ) {
					$output_css .='src:url("' . esc_url( $font['woff'] ) . '") format("woff"),';
				}
				if ( !empty( $font['ttf'] ) ) {
					$output_css .='src:url("' . esc_url( $font['ttf'] ) . '") format("truetype"),';
				}
				if ( !empty( $font['svg'] ) ) {
					$output_css .='src:url("' . esc_url( $font['svg'] ) . '#' . $font_family . '") format("svg");';
				}
				$output_css .= '}';
				$output_css = str_replace( ',src:', ',', $output_css );
				$output_css = str_replace( ',}', ';}', $output_css );
				break;
			}
		}
		return $output_css;
	}

	/**
	 * Get custom fonts CSS style for frontend
	 * @return string  CSS
	 */
	static function get_frontend_style() {
		$theme_fonts = get_option( self::OPTION_NAME );
		$output_css = '';

		if ( ! empty( $theme_fonts ) ) {
			foreach ( $theme_fonts as $font ) {
				if ( isset( $font['source'] ) && 'Custom' == $font['source'] ) {
					$output_css .= self::get_custom_font_css( $font['family'] );
				}
			}
		}
		return $output_css;
	}
}
