<?php
defined( 'ABSPATH' ) || exit; // Exit if accessed directly.
/**
 * SF Fonts Manager
 */
class SF_Fonts_Manager {

	function __construct() {
		add_action( 'wp_ajax_sf_fonts_manager', array( $this, 'handle_ajax_actions' ) );
	}

	public function handle_ajax_actions() {
		$action = isset( $_POST['font_action'] ) ? sanitize_text_field( $_POST['font_action'] ) : ''; 

		if ( 'add_to_font_list' === $action ) {
			$font_source = isset( $_POST['font_source'] ) ? sanitize_text_field( $_POST['font_source'] ) : '';
			$font_name = isset( $_POST['font_name'] ) ? sanitize_text_field( $_POST['font_name'] ) : '';

			if ( 'google' == $font_source ) {
				$this->add_google_font_to_list( $font_name );
			} else if ( 'custom' == $font_source ) {
				$this->add_custom_font_to_list( $font_name );
			}

		} else if ( 'save_fonts' === $action ) {
			$this->save_fonts();

		} else if ( 'add_custom_font' === $action ) {
			$this->add_custom_font();

		} else if ( 'remove_custom_font' === $action ) {
			$this->remove_custom_font();

		} else if ( 'sync_adobe_fonts' === $action ) {
			$this->sync_adobe_fonts();
		} else if ( 'remove_adobe_fonts' === $action ) {
			$this->remove_adobe_fonts();
		}
		
		wp_send_json_error( [ 'message' => '<span class="error">' . __( 'Unknown Error', 'spirit' ) . '</span>' ] );
	}

	public function add_google_font_to_list( $font_name ) {
		$variants         = SF_Fonts::get_variants( $font_name );
		$checked_variants = [ '400', 'regular' ];
		$attr_disabled    = count( $variants ) > 1 ? '' : ' disabled';

		$preview = '<li><span class="sf-font-specimen" style="font-family:'. $font_name .'">Grumpy wizards make toxic brew for the evil Queen and Jack.</span><span class="sf-font-family">'. $font_name .'</span><a href="javascript:void(0)" class="sf-remove-font">X</a></li>';

		ob_start();
		?>
		<li>
			<h2 class="sf-font-family"><?php echo esc_html( $font_name ); ?></h2>
			<div class="sf-font-source"><?php printf( esc_html__( 'Source: %s', 'spirit' ), 'Google' ); ?></div>
			<div class="sf-font-variant-subset">
				<div class="sf-font-variants">
					<h3><?php esc_html_e( 'Variants', 'spirit' ); ?>:</h3>
					<ul><?php foreach ( $variants as $variant ) : ?>
						<li><?php $attr_checked = in_array( $variant, $checked_variants ) ? ' checked' : ''; ?>
							<label>
								<input type="checkbox" data-variant="<?php echo esc_attr( $variant ); ?>" class="sf-check-variant"<?php echo esc_html( $attr_checked . $attr_disabled ); ?>>
								<?php echo esc_html( $variant ); ?>
							</label>
						</li>
					<?php endforeach; ?>
					</ul>
				</div>
				<div class="clear"></div>
			</div>
		</li><?php
		
		$details = ob_get_clean();

		wp_send_json_success( [
			'success' => true,
			'preview' => $preview,
			'details' => $details,
		] );
	}

	public function add_custom_font_to_list( $font_name ) {
		$preview = '<li><style>' . SF_Fonts::get_custom_font_css( $font_name ) . '</style><span class="sf-font-specimen" style="font-family:\'' . esc_attr( $font_name ) . '\';">Grumpy wizards make toxic brew for the evil Queen and Jack.</span><span class="sf-font-family">' . esc_html( $font_name ) . '</span><a href="javascript:void(0)" class="sf-remove-font">X</a></li>';
		
		$details = '<li><h2 class="sf-font-family">' . esc_html( $font_name ) . '</h2><div class="sf-font-source">' . esc_html__( 'Source: Custom', 'spirit' ) . '</div></li>';

		wp_send_json_success( [
			'success' => true,
			'preview' => $preview,
			'details' => $details,
		] );
	}

    /**
     * Save default font list
     */
    public function save_fonts() {
    	$font_data = isset( $_POST['font_data'] ) ? (array) $_POST['font_data']: array();

    	if ( ! empty( $font_data ) ) {
    		update_option( SF_Fonts::OPTION_NAME, $font_data );
			wp_send_json_success( [
				'message' => '<span class="success">' . esc_html__( 'Fonts Saved.', 'spirit' ) . '</span>'
			] );
    	}

		wp_send_json_error();
    }

	/**
	 * Add custom font - upload font files & save font
	 */
	public function add_custom_font() {
        if ( empty( $_POST['custom_font_nonce'] ) || ! wp_verify_nonce( $_POST['custom_font_nonce'], 'add_custom_font' ) ) {
            wp_send_json_error( array( 'message'=>'<span class="error">' . __( 'Token is invalid or expired. Please refresh and try again.', 'spirit' ) . '</span>' ) );
        }

		if ( empty( $_FILES ) ) {
			wp_send_json_error( [
				'message' => '<span class="error">' . esc_html__( 'No font file was choosen.', 'spirit' )  . '</span>'
			] );
		}

		$font_name = isset( $_POST['font_name'] ) ? sanitize_text_field( $_POST['font_name'] ) : '';
		if ( empty( $font_name ) ) {
			wp_send_json_error( [
				'message' => '<span class="error">' . esc_html__( 'Please enter a font name.', 'spirit' )  . '</span>'
			] );
		}

		if ( ! function_exists( 'wp_handle_upload' ) ) {
			require_once( ABSPATH . 'wp-admin/includes/file.php' );
		}

		add_filter( 'upload_dir', array( $this, 'upload_dir' ) );
		add_filter( 'upload_mimes', array( $this, 'upload_mimes' ) );
		add_filter( 'wp_check_filetype_and_ext', array( $this, 'disable_mime_for_ttf_files' ), 10, 4 );

		$upload = wp_upload_dir();
		$new_font = array();
		$new_font['family'] = $font_name;
		$custom_fonts = get_option( SF_Fonts::CUSTOM_FONTS_OPTION_NAME );

		// check if font exists
		if ( ! empty( $custom_fonts ) ) {
			foreach ( $custom_fonts as $font ) {
				if ( $font['family'] == $font_name ) {
					wp_send_json_error( [
						'message' => '<span class="error">' . esc_html__( 'Font already exist.', 'spirit' )  . '</span>'
					] );
				}
			}
		}

		foreach ( $_FILES as $key => $file ) {
			if ( !@file_exists( trailingslashit( $upload['path'] ) . $file['name'] ) ) {
				$overrides = array( 'test_form' => false );
				$upload_file = wp_handle_upload( $file, $overrides );
				if ( isset( $upload_file['error'] ) ) {
					wp_send_json_error( [
						'message' => '<span class="error">' . $upload_file['error'] . '</span>'
					]);
				}
			}
			$new_font[ $key ] = trailingslashit( $upload['url'] ) . sanitize_text_field( $file['name'] );
		}
		remove_filter( 'upload_dir', array( $this, 'upload_dir' ) );
		remove_filter( 'upload_mimes', array( $this, 'upload_mimes' ) );
		remove_filter( 'wp_check_filetype_and_ext', array( $this, 'disable_mime_for_ttf_files' ) );

		// update font option
		$custom_fonts = get_option( SF_Fonts::CUSTOM_FONTS_OPTION_NAME );
		$custom_fonts = ! empty( $custom_fonts) && is_array( $custom_fonts ) ? $custom_fonts : array();
		$custom_fonts[] = $new_font;
		update_option( SF_Fonts::CUSTOM_FONTS_OPTION_NAME, $custom_fonts );

		wp_send_json_success( [
			'message' => '<span class="success">' . esc_html__( 'Font has been added.', 'spirit' ) . '</span>'
		] );
	}

	/**
	 * Remove custom font
	 */
	public function remove_custom_font() {
		$font_name = isset( $_POST['font_name'] ) ? sanitize_text_field( $_POST['font_name'] ) : '';
		$upload = wp_upload_dir();
		
		if ( ! empty( $font_name ) ) {
			$custom_fonts = get_option( SF_Fonts::CUSTOM_FONTS_OPTION_NAME ); $i = 0;
			foreach ( $custom_fonts as $font ) {
				if ( $font['family'] == $font_name ) {
					foreach ( $font as $key => $value ) {
						if ( $key !== 'family' && ! empty( $value ) ) {
							$file_name = basename( $value );
							$file_name = preg_replace( '/.[^.]*$/', '', $file_name );
							$file_path = trailingslashit( $upload['basedir'] ) . 'sf-fonts/' . $file_name . '.' . $key;
							if ( @file_exists( $file_path ) ) {
								@unlink( $file_path );
							}
						}
					}
					unset( $custom_fonts[ $i ] );
					$custom_fonts = array_values( $custom_fonts );
					update_option( SF_Fonts::CUSTOM_FONTS_OPTION_NAME, $custom_fonts );
					break;
				}
				$i ++;
			}

			$fonts = get_option( SF_Fonts::OPTION_NAME ); $j = 0;
			foreach ( $fonts as $font ) {
				if ( $font['family'] == $font_name ) {
					unset( $fonts[ $j ] );
					$fonts = array_values( $fonts );
					update_option( SF_Fonts::OPTION_NAME, $fonts );
					break;
				}
				$j ++;
			}
		}

		exit;
	}

	public function sync_adobe_fonts() {
        if ( empty( $_POST['sync_adobe_fonts_nonce'] ) || ! wp_verify_nonce( $_POST['sync_adobe_fonts_nonce'], 'sync_adobe_fonts' ) ) {
            wp_send_json_error( array( 'message'=>'<span class="error">' . __( 'Token is invalid or expired. Please refresh and try again.', 'spirit' ) . '</span>' ) );
        }
		
		$kit_id = isset( $_POST['kit_id'] ) ? sanitize_text_field( $_POST['kit_id'] ): '';

		if ( empty( $kit_id ) ) {
			wp_send_json_error( [
				'message' => '<span class="error">' . __( 'Please enter your Kit ID', 'spirit' ) . '</span>'
			] );
		}

		$api_base = 'https://typekit.com/api/v1/json/kits';

		$response = wp_remote_get( $api_base . '/' . $kit_id . '/published' );

		// Response is a WP_Error object
		if ( is_wp_error( $response ) ) {
			wp_send_json_error( [
				'message' => '<span class="error">' . __( 'Unknown Error', 'spirit' ) . '</span>'
			] );
		}

		// Response code is not success
		$response_code = (int) wp_remote_retrieve_response_code( $response );
		$response_body = json_decode( wp_remote_retrieve_body( $response ) );
		if ( 200 !== $response_code ) {
			switch ( $response_code ) {
				case 404:
					$error_msg = __( 'Project not found.', 'spirit' );
					break;
				default:
					$error_msg = $response_code;
					if ( isset( $response_body->errors ) ) {
						$error_msg .= ': ' . implode( ', ', $response_body->errors );
					}
					break;
			}

			if ( ! empty( $error_msg ) ) {
				wp_send_json_error( [
					'message' => '<span class="error">' . $error_msg . '</span>'
				] );
			}
		}

		if ( ! $response_body ) {
			wp_send_json_error( [
				'message' => '<span class="error">' . __( 'No project data was returned.', 'spirit' ) . '</span>'
			] );
		}

		/*
		 * Expected Json response example
		 * {
		 *    "kit": {
		 *        "id": "nmm7qvq",
		 *        "families": [
		 *             {
		 *                 "id": "hmqz",
		 *                 "name": "Adobe Caslon Pro",
		 *                 "slug": "adobe-caslon-pro",
		 *	               "css_names": [
		 *                    "adobe-caslon-pro"
		 * 	                ],
		 *                 "css_stack": "\"adobe-caslon-pro\",serif",
		 *                 "variations": [ "n6","i6","i7" ]
		 * 	            }
		 * 	        ]
		 * 	    }
		 * 	}
		 */
		if ( ! is_object( $response_body ) || ! isset( $response_body->kit ) || ! isset( $response_body->kit->families ) || ! is_array( $response_body->kit->families ) ) {
			wp_send_json_error();
		}

		$adobe_fonts = [];
		foreach ( $response_body->kit->families as $font_family ) {
			$font_data = (array) $font_family;
			if ( isset( $font_data['variations'] ) ) {
				$font_data['variations'] = SF_Fonts::get_normalized_adobe_fonts_variants( $font_data['variations'] );
			}
			$adobe_fonts[] = $font_data;

		}

		update_option( SF_Fonts::ADOBE_FONTS_OPTION_NAME, $adobe_fonts );
		update_option( SF_Fonts::ADOBE_FONTS_KIT_ID_OPTION_NAME, $kit_id );

		$html_font_list = '';

		if ( ! empty( $adobe_fonts ) ) :

			ob_start();
			?>
			<ul class="sf-font-list">
				<?php foreach ( $adobe_fonts as $font ) :
						$font_family = $font['name'];
						$preview_link = 'https://fonts.adobe.com/fonts/' . $font['slug'] . '/';
						$font_css_name = isset( $font['css_names'][0] ) ? $font['css_names'][0] : '';
					?>
					<li>
						<span class="sf-font-name"><?php echo esc_html( $font['name'] ); ?></span>
						<div class="sf-font-actions">
							<a href="<?php echo esc_url( $preview_link ); ?>" target="_blank" class="sf-preview-font"><?php esc_html_e( 'Preview', 'spirit' ) ?></a>
						</div>
					</li>
				<?php endforeach; ?>
			</ul>
			<?php
			$html_font_list = ob_get_clean();
		
		endif;

		wp_send_json_success( [
			'html' => $html_font_list
		]);
	}

	public function remove_adobe_fonts() {
		if ( empty( $_POST['sync_adobe_fonts_nonce'] ) || ! wp_verify_nonce( $_POST['sync_adobe_fonts_nonce'], 'sync_adobe_fonts' ) ) {
            wp_send_json_error( array( 'message'=>'<span class="error">' . __( 'Token is invalid or expired. Please refresh and try again.', 'spirit' ) . '</span>' ) );
        }

		delete_option( SF_Fonts::ADOBE_FONTS_OPTION_NAME );
		delete_option( SF_Fonts::ADOBE_FONTS_KIT_ID_OPTION_NAME );

		wp_send_json_success();
	}

	public function upload_dir( $upload ) {
		$upload['subdir'] = '/sf-fonts';
		$upload['path'] = $upload['basedir'] . $upload['subdir'];
		$upload['url'] = $upload['baseurl'] . $upload['subdir'];
		return $upload;
	}

	public function upload_mimes( $mimes ) {
		$mimes['woff'] = 'application/font-woff';
		$mimes['ttf'] = 'application/x-font-truetype';
		$mimes['eot'] = 'application/vnd.ms-fontobject';
		$mimes['svg'] = 'image/svg+xml';
		$mimes['woff2'] = 'application/font-woff2';
		$mimes['otf'] = 'application/x-font-opentype';
		return $mimes;
	}

	/**
	 * Disable Mime type check for TTF font files
	 *
	 * A bug was introduced in WordPress 4.7.1 which caused stricter checks on mime types
	 * However, files can have multiple mime types which doesn't appear to be supported yet.
	 * Once this bug is resolved we'll remove this patch.
	 *
	 * @trac https://core.trac.wordpress.org/ticket/39550
	 *
	 * @param array  $data     File data array containing 'ext', 'type', and 'proper_filename' keys.
	 * @param string $file     Full path to the file.
	 * @param string $filename The name of the file (may differ from $file due to $file being in a tmp directory).
	 * @param array  $mimes    Key is the file extension with value as the mime type.
	 *
	 * @return array
	 *
	 * @since 4.1
	 */
	public function disable_mime_for_ttf_files( $data, $file, $filename, $mimes ) {
		$wp_filetype = wp_check_filetype( $filename, $mimes );
		if ( strtolower( $wp_filetype['ext'] ) === 'ttf' ) {
			$ext             = $wp_filetype['ext'];
			$type            = $wp_filetype['type'];
			$proper_filename = $data['proper_filename'];
			return compact( 'ext', 'type', 'proper_filename' );
		}
		return $data;
	}
}

new SF_Fonts_Manager();

/**
 * Custom fonts optroup.
 *
 * @param array $custom_choice The custom choice.
 *
 * @return array The updated custom choice.
 */
function sf_custom_fonts_group( $custom_choice ) {
	$custom_fonts = get_option( SF_Fonts::CUSTOM_FONTS_OPTION_NAME );
	$variants     = [];
	$children	  = [];

	if ( ! empty( $custom_fonts ) ) {

		foreach ( $custom_fonts as $key => $custom_font ) {
			$children[] = [
				'id'   => $custom_font['family'],
				'text' => $custom_font['family'],
			];
			$variants[@$custom_font['family']] = ['regular'];
		}

		$custom_choice['families']['custom_fonts'] = [
			'text'     => esc_attr__( 'Custom Fonts', 'spirit' ),
			'children' => $children,
		];

		$custom_choice['variants'] = $variants;

	}

	return $custom_choice;

}

add_filter( 'sf_kirki_font_choices', 'sf_custom_fonts_group', 10 );

/**
 * Extend Kirki font choice with Adobe Fonts font group.
 *
 * @param array $custom_choice The custom choice.
 *
 * @return array The updated custom choice.
 */
function sf_adobe_fonts_group( $custom_choice ) {
	$adobe_fonts = SF_Fonts::get_adobe_fonts();
	$variants    = [];

	if ( ! empty( $adobe_fonts ) ) {

		foreach ( $adobe_fonts as $key => $font ) {
			$children[] = [
				'id'   => $font['slug'],
				'text' => $font['name'],
			];
			$variants[$font['slug']] = SF_Fonts::get_normalized_adobe_fonts_variants( $font['variations'] );
		}

		$custom_choice['families']['typekit'] = [
			'text'     => esc_attr__( 'Adobe Fonts', 'spirit' ),
			'children' => $children,
		];

		$custom_choice['variants'] = $variants;

	}

	return $custom_choice;

}

add_filter( 'sf_kirki_font_choices', 'sf_adobe_fonts_group', 20 );