<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

abstract class SF_Slider_Base extends Widget_Base {

    public function get_categories() {
        return [ 'sf-addons' ];
    }

    public function get_keywords() {
        return [ 'sf' ];
    }

    public function get_script_depends() {
        return [
            'swiper',
			'sf-frontend',
			'imagesloaded'
        ];
	}
	
	public function get_style_depends() {
		return [
			'swiper'
		];
	}
	
	public function register_slider_controls_section( $condition = [] ) {
		
		$this->start_controls_section(
			'section_slider',
			[
				'label' => __( 'Slider', 'spirit' ),
				'tab' => Controls_Manager::TAB_CONTENT,
				'condition' => $condition
			]
		);

		$this->add_responsive_control(
			'slides_to_show',
			[
				'label' => __( 'Sliders Per View', 'spirit' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					1 => __( '1', 'spirit' ),
					2 => __( '2', 'spirit' ),
					3 => __( '3', 'spirit' ),
					4 => __( '4', 'spirit' ),
					5 => __( '5', 'spirit' ),
					6 => __( '6', 'spirit' ),
					'auto' => __( 'Auto', 'spirit' ),
					'' => __( 'Default', 'spirit' ),
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'desktop_default' => '',
				'tablet_default' => '',
				'mobile_default' => 1,
				'render_type' => 'template',
			]
		);

		$this->add_control(
			'slides_to_scroll',
			[
				'label' => __( 'Sliders Per Scroll', 'spirit' ),
				'type' => Controls_Manager::SELECT,
				'description' => __( 'Set how many slides are scrolled per swipe.', 'spirit' ),
				'options' => [
					1 => __( '1', 'spirit' ),
					2 => __( '2', 'spirit' ),
					3 => __( '3', 'spirit' ),
					4 => __( '4', 'spirit' ),
					5 => __( '5', 'spirit' ),
					6 => __( '6', 'spirit' ),
					'' => __( 'Default', 'spirit' )
				],
				'condition' => [
					'slides_to_show!' => '1',
				],
				'default' => '',
			]
		);

        $this->add_control(
            'navigation',
            [
                'type' => Controls_Manager::SWITCHER,
                'label' => __( 'Arrows', 'spirit' ),
                'default' => 'yes',
                'label_off' => __( 'Hide', 'spirit' ),
                'label_on' => __( 'Show', 'spirit' ),
                'prefix_class' => 'sf-arrows--',
                'render_type' => 'template',
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'pagination',
            [
                'label' => __( 'Pagination', 'spirit' ),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    '' => __( 'None', 'spirit' ),
                    'bullets' => __( 'Dots', 'spirit' ),
                ],
                'prefix_class' => 'sf-pagination--type-',
                'render_type' => 'template',
                'frontend_available' => true
            ]
        );

		$this->add_control(
			'autoplay',
			[
				'label' => __( 'Autoplay', 'spirit' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'separator' => 'before'
			]
		);

		$this->add_control(
			'autoplay_speed',
			[
				'label' => __( 'Autoplay Speed', 'spirit' ),
				'type' => Controls_Manager::NUMBER,
				'default' => 5000,
				'condition' => [
					'autoplay' => 'yes'
				]
			]
		);

		$this->add_control(
			'loop',
			[
				'label' => __( 'Infinite Loop', 'spirit' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes'
			]
		);

        $this->add_control(
            'fade',
            [
                'label' => __( 'Fade', 'spirit' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
                'condition' => [
                    'slides_to_show' => '1'
                ]
            ]
        );

		$this->add_control(
			'speed',
			[
				'label' => __( 'Animation Speed', 'spirit' ),
				'type' => Controls_Manager::NUMBER,
				'default' => 300,
				'condition' => [
                    'slides_to_show' => '1'
                ]
			]
		);

		$this->end_controls_section();
	}

	public function register_navigation_style_controls_section( $condition = [] ) {

        $this->start_controls_section(
            'section_navigation_style',
            [
                'label' => esc_attr__( 'Navigation', 'spirit' ),
				'tab' => Controls_Manager::TAB_STYLE,
				'condition' => $condition
            ]
        );

		$this->add_control(
			'arrows_skin',
			[
				'label' => __( 'Arrows Skin', 'spirit' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'default',
				'prefix_class' => 'sf-arrows--skin-',
				'options' => [
					'default' => __( 'Default', 'spirit' ),
					'1' => __( 'Style 1', 'spirit' ),
					'2' => __( 'Style 2', 'spirit' ),
					'3' => __( 'Style 3', 'spirit' ),
					'4' => __( 'Style 4', 'spirit' )
				],
				'render_type' => 'template'
			]
		);

        $this->add_control(
            'arrows_color',
            [
                'label' => __( 'Arrows Color', 'spirit' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'dark',
                'prefix_class' => 'sf-arrows--color-',
                'options' => [
                    'light' => __( 'Light', 'spirit' ),
                    'dark' => __( 'Dark', 'spirit' )
                ]
            ]
        );

		$this->add_control(
			'arrows_position',
			[
				'label' => __( 'Arrows Position', 'spirit' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'default',
				'prefix_class' => 'sf-arrows--position-',
				'options' => [
					'default' => __( 'Default', 'spirit' ),
					'top_right' => __( 'Top Right', 'spirit' ),
					'top_left' => __( 'Top Left', 'spirit' ),
					'bottom_right' => __( 'Bottom Right', 'spirit' ),
					'bottom_left' => __( 'Bottom Left', 'spirit' )
				]
			]
		);


		$this->add_responsive_control(
			'arrows_offset',
			[
				'label' => __( 'Arrows Top Position', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'unit' => '%',
					'size' => 50,
				],
				'tablet_default' => [
					'unit' => '%',
				],
				'mobile_default' => [
					'unit' => '%',
				],
				'range' => [
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'size_units' => [ '%' ],
				'selectors' => [
					'{{WRAPPER}}.sf-arrows--position-default .sf-swiper-btn' => 'top: {{SIZE}}{{UNIT}};'
				],
			]
		);

		$this->add_responsive_control(
			'arrows_custom_position',
			[
				'label' => __( 'Arrows Horizontal Position', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ '%', 'px' ],
				'selectors' => [
					'{{WRAPPER}}.sf-arrows--position-default. .sf-swiper-btn-prev' => 'left: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}}.sf-arrows--position-default. .sf-swiper-btn-next' => 'right: {{SIZE}}{{UNIT}};',
				]
			]
		);

        $this->add_control(
            'arrows_hide_under',
            [
                'label' => __( 'Hide Under', 'spirit' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no'
            ]
		);
		
		$this->add_control(
			'arrows_hide_under_width',
			[
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 2000
					]
				],
				'condition' => [
					'arrows_hide_under' => 'yes'
				]
			]
		);

		$this->add_control(
			'bullets_color',
			[
				'label' => __( 'Bullets Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .sf-swiper-pagination .swiper-pagination-bullet' => 'background: {{VALUE}};'
				],
				'separator' => 'before'
			]
		);

		$this->add_control(
			'bullets_active_color',
			[
				'label' => __( 'Bullets Active Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .sf-swiper-pagination .swiper-pagination-bullet-active' => 'background: {{VALUE}};'
				]
			]
		);

		$this->add_responsive_control(
			'bullets_offset',
			[
				'label' => __( 'Bullets Offset', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'px' => 50
				],
				'selectors' => [
					'{{WRAPPER}} .sf-swiper-pagination.swiper-pagination-bullets' => 'margin-top: {{SIZE}}{{UNIT}}!important;',
				]
			]
		);

        $this->end_controls_section();
	}

    /**
     * Get slider settings
     * @param  array $settings  widget settings
     * @return array            slider settings
     */
    public function get_slider_settings( $settings ) {
        $slider_settings = [];

        $spacing_desktop = ( '' != $settings['spacing']['size'] ) ? $settings['spacing']['size'] : 30;
        $spacing_tablet = isset( $settings['spacing_tablet'] ) && '' != $settings['spacing_tablet']['size'] ? $settings['spacing_tablet']['size'] : $spacing_desktop;
        $spacing_mobile = isset( $settings['spacing_mobile'] ) && '' != $settings['spacing_mobile']['size'] ? $settings['spacing_mobile']['size'] : $spacing_tablet;

		// $slider_settings['handleElementorBreakpoints'] = true;
        $slider_settings['autoHeight'] = true;
		$slider_settings['watchOverflow'] = true;
		$slider_settings['slidesPerView'] = ! empty( $settings['slides_to_show_mobile'] ) ? $settings['slides_to_show_mobile'] : 1;
        $slider_settings['slidesPerGroup'] = ! empty( $settings['slides_to_scroll'] ) ? intval( $settings['slides_to_scroll'] ) : 1;
        $slider_settings['spaceBetween'] = $spacing_mobile;
        $slider_settings['loop'] = ( $settings['loop'] == 'yes' );
        $slider_settings['speed'] = isset( $settings['speed'] ) ? $settings['speed'] : 300;

		$breakpoint_tablet_min = Plugin::$instance->breakpoints->get_device_min_breakpoint('desktop');
		$breakpoint_mobile_min = Plugin::$instance->breakpoints->get_device_min_breakpoint('tablet');

        $slider_settings['breakpoints'] = [
            $breakpoint_tablet_min => [
                'slidesPerView' => ! empty( $settings['slides_to_show'] ) ? intval( $settings['slides_to_show'] ) : 3,
                'spaceBetween' => $spacing_desktop,
            ],
            $breakpoint_mobile_min => [
                'slidesPerView' => ! empty( $settings['slides_to_show_tablet'] ) ? intval( $settings['slides_to_show_tablet'] ) : 2,
                'spaceBetween' => $spacing_tablet,
            ]
        ];

		if ( ! empty( $settings['slides_to_show'] ) && 'auto' === $settings['slides_to_show'] ) {
			$slider_settings['loopedSlides'] = 1;
		}
        
        if ( 'yes' == $settings['autoplay'] ) {
        	if ( ! empty( $settings['autoplay_speed'] ) ) {
        		$slider_settings['autoplay'] = [
        			'delay' => intval( $settings['autoplay_speed'] )
        		];
        	} else {
        		$slider_settings['autoplay'] = true;
        	}
        }

        if ( 'yes' == $settings['fade'] ) {
        	$slider_settings['effect'] = 'fade';
		}
		
		if ( isset( $settings['center_slides'] ) && 'yes' == $settings['center_slides'] ) {
			$slider_settings['centeredSlides'] = true;
		}

        if ( ! empty( $settings['navigation'] ) ) {
            $slider_settings['navigation'] = [
            	'prevEl' => $settings['prev_btn_selector'],
                'nextEl' => $settings['next_btn_selector'],
            ];
        }

        if ( 'bullets' == $settings['pagination'] ) {
            $slider_settings['pagination'] = [
                'el' => $settings['pagination_selector'],
                'clickable' => true,
                'type' => 'bullets',
            ];
		}
		
		if ( 'yes' === $settings['arrows_hide_under'] ) {
			$slider_settings['arrowsHideUnderWidth'] = ! empty( $settings['arrows_hide_under_width']['size'] ) ? intval( $settings['arrows_hide_under_width']['size'] ) : '';
		}
        
        return $slider_settings;
	}
	
    /**
     * Get slider navigation html
     * @param  array $settings  widget settings
     */
    public function slider_navigation( $settings ) {
		$prev_btn_id   = str_replace( '#', '', $settings['prev_btn_selector'] );
		$next_btn_id   = str_replace( '#', '', $settings['next_btn_selector'] );
		$pagination_id = str_replace( '#', '', $settings['pagination_selector'] );
		
		if ( 'bullets' == $settings['pagination'] ) :
		?>
			<div id="<?php echo esc_attr( $pagination_id ) ?>" class="sf-swiper-pagination"></div>
		<?php
		endif;

		if ( isset( $settings['navigation'] ) && $settings['navigation'] ) :
			if ( 'default' === $settings['arrows_skin'] ) {
				$prev_btn_text = '<span class="prev">' . esc_html__( 'Prev', 'spirit' ) . '</span>';
				$next_btn_text = '<span class="next">' . esc_html__( 'Next', 'spirit' ) . '</span>';
			} else {
				$prev_btn_text = '<span class="sf-swiper-btn-icon sicon-arrow-left" aria-hidden="true"></span>';
				$next_btn_text = '<span class="sf-swiper-btn-icon sicon-arrow-right" aria-hidden="true"></span>';
			}
		?>
			<div id="<?php echo esc_attr( $prev_btn_id ) ?>" class="sf-swiper-btn sf-swiper-btn-prev">
				<?php echo apply_filters( 'sf_slider_prev_btn_text', $prev_btn_text ); ?>
			</div>
			<div id="<?php echo esc_attr( $next_btn_id ) ?>" class="sf-swiper-btn sf-swiper-btn-next">
				<?php echo apply_filters( 'sf_slider_next_btn_text', $next_btn_text ); ?>
			</div>
		<?php endif;
	}
}