<?php

namespace Elementor;

if ( !defined('ABSPATH') ) exit; // Exit if accessed directly

class SF_Products extends SF_Slider_Base {

    public function get_name() {
        return 'sf-products';
    }

    public function get_title() {
        return esc_attr__( 'Woo Products', 'spirit' );
    }

    public function get_icon() {
        return 'eicon-products sf-addons-label';
    }

    protected function register_controls() {

        $this->start_controls_section(
            'section_layout',
            [
                'label' => esc_attr__( 'Products', 'spirit' ),
            ]
        );

        $this->add_control(
            'layout',
            [
                'label' => __( 'Layout', 'spirit' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'grid' => __( 'Grid', 'spirit' ),
                    'carousel' => __( 'Carousel', 'spirit' ),
                ],
                'default' => 'carousel',
            ]
        );

        $this->add_responsive_control(
            'columns',
            [
                'label' => __( 'Columns', 'spirit' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    1 => __( '1', 'spirit' ),
                    2 => __( '2', 'spirit' ),
                    3 => __( '3', 'spirit' ),
                    4 => __( '4', 'spirit' ),
                    6 => __( '6', 'spirit' ),
                    '' => __( 'Default', 'spirit' ),
                ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'desktop_default' => 4,
                'tablet_default' => 2,
                'mobile_default' => 1,
                'condition' => [
                    'layout!' => 'carousel',
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_query',
            [
                'label' => esc_attr__( 'Query', 'spirit' ),
            ]
        );

        $this->add_control(
            'products',
            [
                'label' => __( 'Products', 'spirit' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'default' => '',
                'multiple' => true,
                'options' => sf_get_option_posts( 'product' ),
            ]
        );

        $this->add_control(
            'categories',
            [
                'label' => esc_attr__( 'Categories', 'spirit' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'default' => '',
                'multiple' => true,
                'options' => sf_get_option_terms( 'product_cat' ),
            ]
        );

        $this->add_control(
            'tags',
            [
                'label' => esc_attr__( 'Tags', 'spirit' ),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'default' => '',
                'multiple' => true,
                'options' => sf_get_option_terms( 'product_tag' ),
            ]
        );

        $this->add_control(
            'show',
            [
                'label' => __( 'Show', 'spirit' ),
                'type' => Controls_Manager::SELECT,
                'options' => array(
                    '' => esc_html__( 'All products', 'spirit' ),
                    'featured' => esc_html__( 'Featured products', 'spirit' ),
                    'onsale' => esc_html__( 'On-sale products', 'spirit' ),
                ),
            ]
        );

        $this->add_control(
            'number',
            [
                'label' => __( 'Number of products to show', 'spirit' ),
                'type' => Controls_Manager::NUMBER,
                'default' => 6,
            ]
        );

        $this->add_control(
            'offset',
            [
                'label' => __( 'Offset', 'spirit' ),
                'type' => Controls_Manager::NUMBER,
                'default' => 0,
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label' => __( 'Order By', 'spirit' ),
                'type' => Controls_Manager::SELECT,
                'options' => array(
                    'date' => esc_html__( 'Date', 'spirit' ),
                    'title' => esc_html__( 'Title', 'spirit' ),
                    'price' => esc_html__( 'Price', 'spirit' ),
                    'rating' => esc_html__( 'Rating', 'spirit' ),
                    'rand' => esc_html__( 'Random', 'spirit' ),
                    'sales' => esc_html__( 'Sales', 'spirit' ),
                    'menu_order' => esc_html__( 'Menu Order', 'spirit' ),
                ),
                'default' => 'date',
            ]
        );

        $this->add_control(
            'order',
            [
                'label' => __( 'Order', 'spirit' ),
                'type' => Controls_Manager::SELECT,
                'options' => array(
                    'desc' => esc_html__( 'DESC', 'spirit' ),
                    'asc' => esc_html__( 'ASC', 'spirit' ),
                ),
                'default' => 'desc',
            ]
        );

        $this->add_control(
            'hide_free',
            [
                'label' => __( 'Hide free products', 'spirit' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
                'separator' => 'before',
            ]
        );

        $this->end_controls_section();

        $this->register_slider_controls_section( [ 'layout' => 'carousel'] );

        $this->start_controls_section(
            'section_layout_style',
            [
                'label' => esc_attr__( 'Layout', 'spirit' ),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $this->add_responsive_control(
            'alignment',
            [
                'label' => __( 'Alignment', 'spirit' ),
                'type' => Controls_Manager::CHOOSE,
                'default' => 'center',
                'options' => [
                    'left' => [
                        'title' => __( 'Left', 'spirit' ),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => __( 'Center', 'spirit' ),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => __( 'Right', 'spirit' ),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .sf-products .product-info' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'spacing',
            [
                'label' => __( 'Space Between', 'spirit' ),
                'type' => Controls_Manager::SLIDER,
                'default' => [ 'size' => 30 ],
                'tablet_default' => [ 'size' => 20 ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .row' => 'margin-left: calc( -{{SIZE}}{{UNIT}}/2 ); margin-right: calc( -{{SIZE}}{{UNIT}}/2 );',
                    '{{WRAPPER}} .row [class^=col-]' => 'padding-left: calc( {{SIZE}}{{UNIT}}/2 ); padding-right: calc( {{SIZE}}{{UNIT}}/2 ); margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail',
                'exclude' => [ 'custom' ],
                'default' => 'shop_catalog',
            ]
        );

        $this->end_controls_section();

        $this->register_navigation_style_controls_section();

    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $products = $this->get_products( $settings );
        if ( $products && $products->have_posts() ) :
        ?>
        <div class="woocommerce">
            <div class="sf-products">
                <div class="products clearfix">
                <?php if ( 'carousel' == $settings['layout'] ) :
                    // use unique id selector
                    $prev_btn_id = uniqid();
                    $next_btn_id = uniqid();
                    $pagination_id = uniqid();
                    $settings['prev_btn_selector'] = '#sf-swiper-btn-'. $prev_btn_id;
                    $settings['next_btn_selector'] = '#sf-swiper-btn-'. $next_btn_id;
                    $settings['pagination_selector'] = '#sf-swiper-pagination-'. $pagination_id;
                    ?>
                    <div class="sf-swiper-container">
                        <div class="swiper" data-settings='<?php echo wp_json_encode( $this->get_slider_settings( $settings ) ); ?>'>
                            <div class="swiper-wrapper">
                                <?php while ( $products->have_posts() ) : $products->the_post(); ?>
                                <div class="swiper-slide"><?php $this->loop_product( $settings['thumbnail_size'] ); ?></div>
                                <?php endwhile; ?>
                            </div>
                        </div>
                        <?php $this->slider_navigation( $settings ); ?>
                    </div>
                <?php else: ?>
                    <div class="row">
                    <?php while ( $products->have_posts() ) : $products->the_post(); ?>
                        <div class="<?php echo esc_attr( sf_get_column_class( $settings ) ); ?>">
                        <?php $this->loop_product( $settings['thumbnail_size'] ); ?>
                        </div>
                    <?php endwhile; ?>
                    </div>
                <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
        endif;
        wp_reset_postdata();
    }

    /**
     * Get products query
     * @param  array $settings  widget settings
     * @return array            products query
     */
    private function get_products( $settings ) {
        $number  = ! empty( $settings['number'] ) ? absint( $settings['number'] ) : 4;
        $show    = ! empty( $settings['show'] ) ? sanitize_title( $settings['show'] ) : '';
        $orderby = ! empty( $settings['orderby'] ) ? sanitize_title( $settings['orderby'] ) : 'date';
        $order   = ! empty( $settings['order'] ) ? sanitize_title( $settings['order'] ) : 'desc';
        $product_visibility_term_ids = wc_get_product_visibility_term_ids();

        $query_args = array(
            'posts_per_page' => $number,
            'post_status'    => 'publish',
            'post_type'      => 'product',
            'no_found_rows'  => 1,
            'order'          => $order,
            'meta_query'     => array(),
            'tax_query'      => array(
                'relation' => 'AND',
            ),
        );

        if ( ! empty( $settings['products'] ) ) {
            $query_args['post__in'] = $settings['products'];
        }

        if ( ! empty( $settings['offset'] ) ) {
            $query_args['offset'] = $settings['offset'];
        }

        if ( ! empty( $settings['hide_free'] ) ) {
            $query_args['meta_query'][] = array(
                'key'     => '_price',
                'value'   => 0,
                'compare' => '>',
                'type'    => 'DECIMAL',
            );
        }

        if ( 'yes' === get_option( 'woocommerce_hide_out_of_stock_items' ) ) {
            $query_args['tax_query'][] = array(
                array(
                    'taxonomy' => 'product_visibility',
                    'field'    => 'term_taxonomy_id',
                    'terms'    => $product_visibility_term_ids['outofstock'],
                    'operator' => 'NOT IN',
                ),
            );
        }

        if ( ! empty( $settings['categories'] ) ) {
            $query_args['tax_query'][] = array(
                'taxonomy' => 'product_cat',
                'field'    => 'term_id',
                'terms'    => $settings['categories'],
                'operator' => 'IN'
            );
        }

        switch ( $show ) {
            case 'featured':
                $query_args['tax_query'][] = array(
                    'taxonomy' => 'product_visibility',
                    'field'    => 'term_taxonomy_id',
                    'terms'    => $product_visibility_term_ids['featured'],
                );
                break;
            case 'onsale':
                $product_ids_on_sale    = wc_get_product_ids_on_sale();
                $product_ids_on_sale[]  = 0;
                $query_args['post__in'] = $product_ids_on_sale;
                break;
        }

        switch ( $orderby ) {
            case 'price':
                $query_args['meta_key'] = '_price'; // WPCS: slow query ok.
                $query_args['orderby']  = 'meta_value_num';
                break;
            case 'rand':
                $query_args['orderby'] = 'rand';
                break;
            case 'sales':
                $query_args['meta_key'] = 'total_sales'; // WPCS: slow query ok.
                $query_args['orderby']  = 'meta_value_num';
                break;
            case 'rating':
                $query_args['meta_key'] = '_wc_average_rating'; // WPCS: slow query ok.
                $query_args['orderby']  = 'meta_value_num';
                break;
            case 'title':
            case 'menu_order':
                $query_args['orderby'] = 'menu_order title';
                break;
            default:
                $query_args['orderby'] = 'date';
        }

        return new \WP_Query( $query_args );
    }

    /**
     * Print loop product
     * @param  string $thumbnail_size product thumbnail size
     */
    public function loop_product( $thumbnail_size = 'shop_catalog' ) {
        ?>
        <div <?php wc_product_class(); ?>>
            <div class="product-body">
                <div class="product-thumb">
                    <?php woocommerce_template_loop_product_link_open(); ?>
                    <?php woocommerce_show_product_loop_sale_flash(); ?>
                    <?php echo woocommerce_get_product_thumbnail( $thumbnail_size ); ?>
                    </a>
                </div>
                <div class="product-info">
                    <h3 class="woocommerce-loop-product__title">
                        <a href="<?php echo esc_url_raw( get_the_permalink() ); ?>">
                            <?php echo get_the_title(); ?>
                        </a>
                    </h3>
                    <div class="product-meta">
                        <?php woocommerce_template_loop_price(); ?>
                        <?php woocommerce_template_loop_rating(); ?>
                    </div>
                    <div class="product-buttons">
                        <?php if ( function_exists( 'tinv_get_option' ) ) : ?>
                        <?php echo do_shortcode("[ti_wishlists_addtowishlist loop=yes]"); ?>
                        <?php endif; ?>
                        <?php woocommerce_template_loop_add_to_cart(); ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register( new SF_Products() );