<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class SF_Image_Grid extends Widget_Base {

	public function get_name() {
		return 'sf-image-grid';
	}

	public function get_title() {
		return __( 'Image Grid', 'spirit' );
	}

	public function get_icon() {
		return 'eicon-icon-box sf-addons-label';
	}

    public function get_categories() {
        return [ 'sf-addons' ];
    }

	public function get_keywords() {
		return [ 'sf' ];
	}

	protected function register_controls() {

		$this->start_controls_section(
			'section_items',
			[
				'label' => __( 'Items', 'spirit' ),
			]
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'item_image',
			[
				'label' => __( 'Image', 'spirit' ),
				'type' => Controls_Manager::MEDIA,
				'label_block' => true
			]
		);

		$repeater->add_control(
			'item_title',
			[
				'label' => __( 'Title', 'spirit' ),
				'type' => Controls_Manager::TEXT,
				'default' => __( 'Title', 'spirit' ),
				'label_block' => true
			]
		);

		$repeater->add_control(
			'item_link',
			[
				'label' => __( 'URL', 'spirit' ),
				'type' => Controls_Manager::URL,
				'label_block' => true,
			]
        );

		$this->add_control(
			'items',
			[
				'label' => __( 'Items', 'spirit' ),
				'type' => Controls_Manager::REPEATER,
				'separator' => 'before',
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ item_title }}}'
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_layout',
			[
				'label' => __( 'Layout', 'spirit' ),
				'tab'   => Controls_Manager::TAB_STYLE
			]
		);

		$this->add_responsive_control(
			'columns',
			[
				'label' => __( 'Columns', 'spirit' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					1 => __( '1', 'spirit' ),
					2 => __( '2', 'spirit' ),
					3 => __( '3', 'spirit' ),
					4 => __( '4', 'spirit' ),
					6 => __( '6', 'spirit' ),
					'' => __( 'Default', 'spirit' ),
				],
				'devices' => [ 'desktop', 'tablet', 'mobile' ],
				'desktop_default' => 4,
				'tablet_default' => 2,
				'mobile_default' => 1
			]
		);

		$this->add_responsive_control(
			'h_spacing',
			[
				'label' => __( 'Horizontal Spacing', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [ 'size' => 30 ],
				'tablet_default' => [ 'size' => 20 ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .row' => 'margin-left: calc( -{{SIZE}}{{UNIT}}/2 ); margin-right: calc( -{{SIZE}}{{UNIT}}/2 );',
					'{{WRAPPER}} .row .col' => 'padding-left: calc( {{SIZE}}{{UNIT}}/2 ); padding-right: calc( {{SIZE}}{{UNIT}}/2 ); margin-bottom: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_responsive_control(
			'v_spacing',
			[
				'label' => __( 'Vertical Spacing', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [ 'size' => 30 ],
				'tablet_default' => [ 'size' => 20 ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100
					],
				],
				'selectors' => [
					'{{WRAPPER}} .row .col' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->end_controls_section();

        $this->start_controls_section(
            'section_images_styling',
            [
                'label' => __( 'Images', 'spirit' ),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $this->start_controls_tabs( 'tab_image' );

		$this->start_controls_tab(
			'normal_image',
			[
				'label' => __( 'Normal', 'spirit' )
			]
		);

        $this->add_control(
            'image_overlay_color',
            [
                'label' => __( 'Overlay Color', 'spirit' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .sf-image-grid__image:before' => 'background-color: {{VALUE}};',
                ]
            ]
        );

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'box_shadow',
				'selector' => '{{WRAPPER}} .sf-image-grid__image',
				'separator' => 'before'
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'hover_image',
			[
				'label' => __( 'Hover', 'spirit' ),
			]
		);

        $this->add_control(
            'hover_image_overlay_color',
            [
                'label' => __( 'Overlay Color', 'spirit' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .sf-image-grid__item:hover .sf-image-grid__image:before' => 'background-color: {{VALUE}};',
                ]
            ]
        );

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'hover_box_shadow',
				'selector' => '{{WRAPPER}} .sf-image-grid__item:hover .sf-image-grid__image',
				'separator' => 'before'
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_control(
			'border_radius',
			[
				'label' => __( 'Border Radius', 'spirit' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .sf-image-grid__image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator' => 'before'
			]
		);

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail',
                'label' => __( 'Image Size', 'spirit' ),
                'default' => 'large',
            ]
		);

        $this->end_controls_section();

		$this->start_controls_section(
			'section_style_title',
			[
				'label' => __( 'Title', 'spirit' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => __( 'Title Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .sf-image-grid__title' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'selector' => '{{WRAPPER}} .sf-image-grid__title',
			]
		);

		$this->end_controls_section();

	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		if ( empty( $settings['items'] ) ) {
			return;
		}

		$data_animation_settings = sf_get_data_animation_settings( $settings );
		$this->add_render_attribute( 'grid-items', 'class', [ 'sf-image-grid__container', sf_get_row_class( $settings ) ] );
		$this->add_render_attribute( 'grid-column', 'class', 'col' );
		
		if ( $data_animation_settings ) { 
			$this->add_render_attribute(
				'grid-items',
				'data-animation-settings',
				$data_animation_settings
			);
			$this->add_render_attribute( 'grid-column', 'class', 'element-invisible' );
		}

		echo '<div class="sf-image-grid">';
		echo '<div '. $this->get_render_attribute_string( 'grid-items' ) . '>';

		foreach ( $settings['items'] as $index => $item ) {
			
			echo '<div '. $this->get_render_attribute_string( 'grid-column' ) . '>';

			$render_name = 'gird-item-' . $index;
			$this->add_render_attribute( $render_name, 'class', 'sf-image-grid__item' );

			$has_link = ! empty( $item['item_link']['url'] );

			if ( $has_link ) {
				$this->add_render_attribute( $render_name, 'href', $item['item_link']['url'] );
	
				if ( $item['item_link']['is_external'] ) {
					$this->add_render_attribute( $render_name, 'target', '_blank' );
				}
	
				if ( $item['item_link']['nofollow'] ) {
					$this->add_render_attribute( $render_name, 'rel', 'nofollow' );
				}
			}

			if ( $has_link ) {
				echo '<a ' . $this->get_render_attribute_string( $render_name ) . '>';
			} else {
				echo '<div ' . $this->get_render_attribute_string( $render_name ) . '>';
			}

			if ( ! empty( $item['item_image']['url'] ) ) {
				echo '<div class="sf-image-grid__image">';
				echo wp_get_attachment_image( $item['item_image']['id'], $settings['thumbnail_size'] );
				echo '</div>';
			} 

			if ( ! empty( $item['item_title'] ) ) {
				echo sprintf( '<h3 class="sf-image-grid__title">%s</h3>', $item['item_title'] );
			}

			if ( $has_link ) {
				echo '</a>';
			} else {
				echo '</div>';
			}

			echo '</div>';
		}

		echo '</div></div>';
	}
}

Plugin::instance()->widgets_manager->register( new SF_Image_Grid() );