<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class SF_Icon_Boxes extends Widget_Base {

	public function get_name() {
		return 'sf-icon-boxes';
	}

	public function get_title() {
		return __( 'Icon Boxes', 'spirit' );
	}

	public function get_icon() {
		return 'eicon-icon-box sf-addons-label';
	}

    public function get_categories() {
        return [ 'sf-addons' ];
    }

    public function get_keywords() {
        return [ 'sf' ];
    }

	/**
	 * Register icon box widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'section_general',
			[
				'label' => __( 'General', 'spirit' ),
			]
		);

  		$this->add_control(
		  	'icon_position',
		  	[
		   	'label' => __( 'Icon Position', 'spirit' ),
		     	'type' => Controls_Manager::SELECT,
		     	'default' => 'top',
		     	'label_block' => false,
		     	'options' => [
		     		'top' => __( 'Top', 'spirit' ),
		     		'left' => __( 'Left', 'spirit' ),
		     		'right' => __( 'Right', 'spirit' )
		     	],
		     	'prefix_class' => 'sf-position-'
		  	]
		);

		$this->add_control(
			'title_tag',
			[
				'label' => __( 'Title HTML Tag', 'spirit' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h3'
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_items',
			[
				'label' => __( 'Items', 'spirit' ),
			]
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'item_icon',
			[
				'label' => __( 'Icon', 'spirit' ),
				'type' => Controls_Manager::ICONS,
				'default' => [
					'value' => 'fas fa-plus-circle',
					'library' => 'solid'
				],
				'separator' => 'before',
			]
		);

		$repeater->add_control(
			'item_title',
			[
				'label' => __( 'Title & Content', 'spirit' ),
				'type' => Controls_Manager::TEXT,
				'default' => __( 'Title', 'spirit' ),
				'label_block' => true
			]
		);

		$repeater->add_control(
			'item_description',
			[
				'label' => __( 'Content', 'spirit' ),
				'type' => Controls_Manager::WYSIWYG,
				'default' => __( 'Content', 'spirit' ),
				'show_label' => false
			]
        );

		$repeater->add_control(
			'item_link',
			[
				'label' => __( 'Link to', 'spirit' ),
				'type' => Controls_Manager::URL,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => __( 'https://your-link.com', 'spirit' ),
				'separator' => 'before',
			]
		);

		$this->add_control(
			'items',
			[
				'label' => __( 'Items', 'spirit' ),
				'type' => Controls_Manager::REPEATER,
				'separator' => 'before',
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'item_icon' => [
							'value' => 'fas fa-dove',
							'library' => 'fa-solid',
						],
						'item_title' => 'Heading text',
						'item_description' => 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.',
						'item_link' => [
							'url' => '#'
						],
					],
					[
						'item_icon' => [
							'value' => 'fas fa-dove',
							'library' => 'fa-solid',
						],
						'item_title' => 'Heading text',
						'item_description' => 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.',
						'item_link' => [
							'url' => '#'
						],
					],
					[
						'item_icon' => [
							'value' => 'fas fa-dove',
							'library' => 'fa-solid',
						],
						'item_title' => 'Heading text',
						'item_description' => 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.',
						'item_link' => [
							'url' => '#'
						],
					]
				],
				'title_field' => '{{{ item_title }}}'
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_layout',
			[
				'label' => __( 'Layout', 'spirit' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		
        $this->add_responsive_control(
            'columns',
            [
                'label' => __( 'Columns', 'spirit' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    1 => __( '1', 'spirit' ),
                    2 => __( '2', 'spirit' ),
                    3 => __( '3', 'spirit' ),
                    4 => __( '4', 'spirit' ),
                    6 => __( '6', 'spirit' ),
                    '' => __( 'Default', 'spirit' ),
                ],
                'devices' => [ 'desktop', 'tablet', 'mobile' ],
                'desktop_default' => 4,
                'tablet_default' => 2,
                'mobile_default' => 1
            ]
        );
		
        $this->add_responsive_control(
            'h_spacing',
            [
                'label' => __( 'Horizontal Spacing', 'spirit' ),
                'type' => Controls_Manager::SLIDER,
                'default' => [ 'size' => 30 ],
                'tablet_default' => [ 'size' => 20 ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .row' => 'margin-left: calc( -{{SIZE}}{{UNIT}}/2 ); margin-right: calc( -{{SIZE}}{{UNIT}}/2 );',
                    '{{WRAPPER}} .row .col' => 'padding-left: calc( {{SIZE}}{{UNIT}}/2 ); padding-right: calc( {{SIZE}}{{UNIT}}/2 ); margin-bottom: {{SIZE}}{{UNIT}};'
                ]
            ]
        );

        $this->add_responsive_control(
            'v_spacing',
            [
                'label' => __( 'Vertical Spacing', 'spirit' ),
                'type' => Controls_Manager::SLIDER,
                'default' => [ 'size' => 30 ],
                'tablet_default' => [ 'size' => 20 ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .row .col' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_box',
			[
				'label' => __( 'Box', 'spirit' ),
				'tab' => Controls_Manager::TAB_STYLE
			]
		);

		$this->add_responsive_control(
			'h_alignment',
			[
				'label' => __( 'Horiziontal Align', 'spirit' ),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'center',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'spirit' ),
						'icon' => 'fa fa-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'spirit' ),
						'icon' => 'fa fa-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'spirit' ),
						'icon' => 'fa fa-align-right',
					]
				],
				'selectors' => [
					'{{WRAPPER}} .sf-icon-box' => 'text-align: {{VALUE}};'
				],
				'label_block' => false
			]
		);

		$this->add_control(
			'v_alignment',
			[
				'label' => __( 'Vertical Align', 'spirit' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => __( 'Top', 'spirit' ),
						'icon' => 'eicon-v-align-top',
					],
					'middle' => [
						'title' => __( 'Middle', 'spirit' ),
						'icon' => 'eicon-v-align-middle',
					],
					'bottom' => [
						'title' => __( 'Bottom', 'spirit' ),
						'icon' => 'eicon-v-align-bottom',
					]
				],
				'default' => 'top',
				'prefix_class' => 'sf-vertical-align-',
				'condition' => [
					'icon_position!' => 'top'
				]
			]
		);

		$this->add_responsive_control(
			'content_padding',
			[
				'label' => __( 'Padding', 'spirit' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .sf-icon-box' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator' => 'before'
			]
		);

		$this->start_controls_tabs( 'tab_colors' );

		$this->start_controls_tab(
			'normal_colors',
			[
				'label' => __( 'Normal', 'spirit' )
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => __( 'Title Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .sf-icon-box__content .sf-icon-box__title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'description_color',
			[
				'label' => __( 'Description Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .sf-icon-box .sf-icon-box__desc' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'bg_color',
			[
				'label' => __( 'Background Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .sf-icon-box' => 'background-color: {{VALUE}};'
				]
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'border',
				'label' => esc_html__( 'Border', 'spirit' ),
				'selector' => '{{WRAPPER}} .sf-icon-box',
				'separator' => 'before'
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'box_shadow',
				'selector' => '{{WRAPPER}} .sf-icon-box',
				'separator' => 'before'
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'hover_colors',
			[
				'label' => __( 'Hover', 'spirit' ),
			]
		);

		$this->add_control(
			'hover_icon_primary_color',
			[
				'label' => __( 'Icon Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}}.sf-view-stacked .sf-icon-box:hover .sf-icon' => 'background-color: {{VALUE}};',
					'{{WRAPPER}}.sf-view-framed .sf-icon-box:hover .sf-icon, {{WRAPPER}}.sf-view-default .sf-icon-box:hover .sf-icon' => 'color: {{VALUE}};border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'hover_icon_secondary_color',
			[
				'label' => __( 'Icon Secondary Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'condition' => [
					'view!' => 'default',
				],
				'selectors' => [
					'{{WRAPPER}}.sf-view-framed .sf-icon-box:hover .sf-icon' => 'background-color: {{VALUE}};',
					'{{WRAPPER}}.sf-view-stacked .sf-icon-box:hover .sf-icon' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'hover_title_color',
			[
				'label' => __( 'Title Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .sf-icon-box:hover .sf-icon-box__title' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'hover_description_color',
			[
				'label' => __( 'Description Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .sf-icon-box:hover .sf-icon-box__desc' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'hover_bg_color',
			[
				'label' => __( 'Background Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .sf-icon-box:hover' => 'background-color: {{VALUE}};'
				]
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'hover_border',
				'label' => esc_html__( 'Border', 'spirit' ),
				'selector' => '{{WRAPPER}} .sf-icon-box:hover',
				'separator' => 'before'
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'hover_box_shadow',
				'selector' => '{{WRAPPER}} .sf-icon-box:hover',
				'separator' => 'before'
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_control(
			'border_radius',
			[
				'label' => __( 'Border Radius', 'spirit' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .sf-icon-box' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'separator' => 'before'
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_icon',
			[
				'label' => __( 'Icon', 'spirit' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->start_controls_tabs( 'icon_colors' );

		$this->add_control(
			'view',
			[
				'label' => __( 'View', 'spirit' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'default' => __( 'Default', 'spirit' ),
					'stacked' => __( 'Stacked', 'spirit' ),
					'framed' => __( 'Framed', 'spirit' ),
				],
				'default' => 'default',
				'prefix_class' => 'sf-view-'
			]
		);

		$this->add_control(
			'shape',
			[
				'label' => __( 'Shape', 'spirit' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'circle' => __( 'Circle', 'spirit' ),
					'square' => __( 'Square', 'spirit' ),
				],
				'default' => 'circle',
				'condition' => [
					'view!' => 'default'
				],
				'prefix_class' => 'sf-shape-',
			]
		);

		$this->start_controls_tab(
			'icon_colors_normal',
			[
				'label' => __( 'Normal', 'spirit' ),
			]
		);

		$this->add_control(
			'primary_color',
			[
				'label' => __( 'Primary Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}}.sf-view-stacked .sf-icon' => 'background-color: {{VALUE}};',
					'{{WRAPPER}}.sf-view-framed .sf-icon, {{WRAPPER}}.sf-view-default .sf-icon' => 'color: {{VALUE}}; border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'secondary_color',
			[
				'label' => __( 'Secondary Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'condition' => [
					'view!' => 'default',
				],
				'selectors' => [
					'{{WRAPPER}}.sf-view-framed .sf-icon' => 'background-color: {{VALUE}};',
					'{{WRAPPER}}.sf-view-stacked .sf-icon' => 'color: {{VALUE}};',
				]
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'icon_colors_hover',
			[
				'label' => __( 'Hover', 'spirit' ),
			]
		);

		$this->add_control(
			'hover_primary_color',
			[
				'label' => __( 'Primary Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'selectors' => [
					'{{WRAPPER}}.sf-view-stacked .sf-icon-box__icon:hover .sf-icon' => 'background-color: {{VALUE}};',
					'{{WRAPPER}}.sf-view-framed .sf-icon-box__icon:hover .sf-icon, {{WRAPPER}}.sf-view-default .sf-icon-box__icon:hover .sf-icon' => 'color: {{VALUE}};border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'hover_secondary_color',
			[
				'label' => __( 'Secondary Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'default' => '',
				'condition' => [
					'view!' => 'default',
				],
				'selectors' => [
					'{{WRAPPER}}.sf-view-framed .sf-icon-box__icon:hover .sf-icon' => 'background-color: {{VALUE}};',
					'{{WRAPPER}}.sf-view-stacked .sf-icon-box__icon:hover .sf-icon' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'hover_animation',
			[
				'label' => __( 'Hover Animation', 'spirit' ),
				'type' => Controls_Manager::HOVER_ANIMATION,
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'icon_spacing',
			[
				'label' => __( 'Spacing', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 15,
				],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}}.sf-position-right .sf-icon-box__icon' => 'margin-left: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}}.sf-position-left .sf-icon-box__icon' => 'margin-right: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}}.sf-position-top .sf-icon-box__icon' => 'margin-bottom: {{SIZE}}{{UNIT}};',
					'(mobile){{WRAPPER}} .sf-icon-box__icon' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'icon_size',
			[
				'label' => __( 'Size', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 6,
						'max' => 300,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .sf-icon' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'icon_padding',
			[
				'label' => __( 'Padding', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'selectors' => [
					'{{WRAPPER}} .sf-icon' => 'padding: {{SIZE}}{{UNIT}};',
				],
				'range' => [
					'em' => [
						'min' => 0,
						'max' => 5,
					],
				],
				'condition' => [
					'view!' => 'default',
				],
			]
		);

		$this->add_control(
			'icon_border_width',
			[
				'label' => __( 'Border Width', 'spirit' ),
				'type' => Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .sf-icon' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'view' => 'framed',
				],
			]
		);

		$this->add_control(
			'icon_border_radius',
			[
				'label' => __( 'Border Radius', 'spirit' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .sf-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition' => [
					'view!' => 'default',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_content',
			[
				'label' => __( 'Content', 'spirit' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'heading_title',
			[
				'label' => __( 'Title', 'spirit' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'selector' => '{{WRAPPER}} .sf-icon-box__content .sf-icon-box__title',
			]
		);

		$this->add_control(
			'heading_description',
			[
				'label' => __( 'Description', 'spirit' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'description_typography',
				'selector' => '{{WRAPPER}} .sf-icon-box__content .sf-icon-box__desc',
			]
		);

		$this->add_responsive_control(
			'description_spacing',
			[
				'label' => __( 'Spacing', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .sf-icon-box__desc' => 'margin-top: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render icon box widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		
        if ( empty( $settings['items'] ) ) {
            return;
        }

		$data_animation      = sf_get_data_animation_settings( $settings );
		$html_data_animation = '';

		if ( $data_animation ) {
			$html_data_animation = " data-animation-settings='" . $data_animation . "'";
		}

		echo '<div class="sf-icon-boxes">';
		echo '<div class="' . esc_attr( sf_get_row_class( $settings ) ) . '"' . $html_data_animation . '>';

		foreach ( $settings['items'] as $index => $item ) {

			if ( $data_animation ) {
				echo '<div class="col element-invisible">';
			} else {
				echo '<div class="col">';
			}
			
			$render_name = 'icon_box_' . $index;
			$this->add_render_attribute( $render_name, 'class', 'sf-icon-box' );

			$has_link = ! empty( $item['item_link']['url'] );

			if ( $has_link ) {
				$this->add_render_attribute( $render_name, 'href', $item['item_link']['url'] );
	
				if ( $item['item_link']['is_external'] ) {
					$this->add_render_attribute( $render_name, 'target', '_blank' );
				}
	
				if ( $item['item_link']['nofollow'] ) {
					$this->add_render_attribute( $render_name, 'rel', 'nofollow' );
				}
			}

			if ( $has_link ) {
				echo '<a ' . $this->get_render_attribute_string( $render_name ) . '>';
			} else {
				echo '<div ' . $this->get_render_attribute_string( $render_name ) . '>';
			}

			if ( ! empty( $item['item_icon']['value'] ) ) {
				
				$this->add_render_attribute( 'icon_' . $index, 'class', [ 'sf-icon', 'elementor-animation-' . $settings['hover_animation'] ] );

				echo '<div class="sf-icon-box__icon">';
				echo '<span ' . $this->get_render_attribute_string( 'icon_' .$index ) . '>'; 

				icons_manager::render_icon( $item['item_icon'], [ 'aria-hidden' => 'true' ] );

				echo '</span>';
				echo '</div>';
			}

			echo '<div class="sf-icon-box__content">';

			if ( ! empty( $item['item_title'] ) ) {
				echo sprintf( '<%1$s class="sf-icon-box__title">%2$s</%1$s>', $settings['title_tag'], $item['item_title'] );
			}

			if ( ! empty( $item['item_description'] ) ) {
				echo '<div class="sf-icon-box__desc">' . $item['item_description'] . '</div>';	
			}

			echo '</div>';

			if ( $has_link ) {
				echo '</a>';
			} else {
				echo '</div>';
			}

			echo '</div>';
		}

		echo '</div>';
		echo '</div>';
	}
}

Plugin::instance()->widgets_manager->register( new SF_Icon_Boxes() );