<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class SF_Content_Block extends Widget_Base {

	public function get_name() {
		return 'sf-content-block';
	}

	public function get_title() {
		return esc_html__( 'Content Block', 'spirit' );
	}

	public function get_icon() {
		return 'eicon-single-page sf-addons-label';
	}

    public function get_categories() {
        return [ 'sf-addons' ];
    }

    public function get_keywords() {
        return [ 'sf' ];
    }

	protected function register_controls() {

        $this->start_controls_section(
            'section_general',
            [
                'label' => esc_html__( 'General', 'spirit' ),
            ]
        );

		$this->add_control(
			'layout',
			[
				'label' => esc_html__( 'Display', 'spirit' ),
				'type' => Controls_Manager::SELECT,
                'default' => 'tabs',
                'options' => [
                    'tabs' => esc_html__( 'Tabs', 'spirit' ),
                    'list' => esc_html__( 'List', 'spirit' )
                ],
                'prefix_class' => 'sf-content-block--',
                'render_type' => 'template'
			]
        );

		$this->add_control(
			'reverse_order',
			[
				'label' => esc_html__( 'Reverse Order', 'spirit' ),
				'type' => Controls_Manager::SWITCHER,
                'default' => 'no'
			]
        );

		$this->add_control(
			'equal_height',
			[
				'label' => esc_html__( 'Equal Height', 'spirit' ),
				'type' => Controls_Manager::SWITCHER,
                'default' => 'no'
			]
        );

        $this->add_control(
			'button_text',
			[
				'label' => esc_html__( 'Button Text', 'spirit' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Read More', 'spirit' ),
				'placeholder' => esc_html__( 'Button Text', 'spirit' ),
			]
		);

        $this->end_controls_section();

		$this->start_controls_section(
			'section_items',
			[
				'label' => esc_html__( 'Items', 'spirit' )
			]
        );

		$repeater = new Repeater();

		$repeater->add_control(
			'icon',
			[
                'label' => esc_html__( 'Icon', 'spirit' ),
                'type' => Controls_Manager::ICONS,
                'label_block' => true,
			]
		);

		$repeater->add_control(
			'title',
			[
                'label' => esc_html__( 'Title', 'spirit' ),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__( 'Title', 'spirit' ),
                'label_block' => true,
                'dynamic' => [
                    'active' => true
                ]
			]
        );

		$repeater->add_control(
			'content',
			[
                'label' => esc_html__( 'Content', 'spirit' ),
                'type' => Controls_Manager::WYSIWYG,
                'label_block' => true,
                'dynamic' => [
                    'active' => true
                ]
			]
        );
        
		$repeater->add_control(
			'image',
			[
                'label' => esc_html__( 'Image', 'spirit' ),
                'type' => Controls_Manager::MEDIA,
                'dynamic' => [
                    'active' => true,
                ]
			]
        );
        
		$repeater->add_control(
			'tab_icon',
			[
                'label' => esc_html__( 'Tab Icon', 'spirit' ),
                'type' => Controls_Manager::ICONS
			]
        );

		$repeater->add_control(
			'link',
			[
				'label' => __( 'Button Link', 'spirit' ),
				'type' => Controls_Manager::URL,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => __( 'https://your-link.com', 'spirit' ),
			]
        );

        $this->add_control(
            'items',
            [
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'title' => 'Title',
                        'content' => 'Lorem ipsum dolor sit amet consectetur adipisicing elit. Suscipit alias, dolorum impedit reiciendis sed corrupti libero illo. Inventore assumenda numquam iste, consequatur enim dicta repellat nulla dolore tempora eaque veritatis.',
                        'image' => [
                            'url' => Utils::get_placeholder_image_src()
                        ]
                    ],
                    [
                        'title' => 'Title',
                        'content' => 'Lorem ipsum dolor sit amet consectetur adipisicing elit. Suscipit alias, dolorum impedit reiciendis sed corrupti libero illo. Inventore assumenda numquam iste, consequatur enim dicta repellat nulla dolore tempora eaque veritatis.',
                        'image' => [
                            'url' => Utils::get_placeholder_image_src()
                        ]
                    ],
                    [
                        'title' => 'Title',
                        'content' => 'Lorem ipsum dolor sit amet consectetur adipisicing elit. Suscipit alias, dolorum impedit reiciendis sed corrupti libero illo. Inventore assumenda numquam iste, consequatur enim dicta repellat nulla dolore tempora eaque veritatis.',
                        'image' => [
                            'url' => Utils::get_placeholder_image_src()
                        ]
                    ],
                ],
                'title_field' => '{{{ title }}}',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_general_style',
            [
                'label' => esc_html__( 'General', 'spirit' ),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

		$this->add_responsive_control(
			'item_spacing',
			[
				'label' => __( 'Spacing', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000
					],
				],
				'selectors' => [
					'{{WRAPPER}} .sf-content-blocks .sf-content-block:not(:last-child)' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'layout' => 'list'
                ]
			]
		);

        $this->add_control(
            'image_width',
            [
                'label' => esc_html__( 'Image Width', 'spirit' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%', 'vw' ],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} .sf-content-block__image' => 'width: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .sf-content-block__main' => 'width: calc(100% - {{SIZE}}{{UNIT}})',
                ]
            ]
        );

		$this->add_responsive_control(
			'content_padding',
			[
				'label' => esc_html__( 'Padding', 'spirit' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .sf-content-block:nth-child(2n+1) .sf-content-block__main' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .sf-content-block:nth-child(2n) .sf-content-block__main' => 'padding: {{TOP}}{{UNIT}} {{LEFT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{RIGHT}}{{UNIT}};',
                ]
			]
        );
        
        $this->add_control(
            'content_bg_color',
            [
                'label' => esc_html__( 'Background Color', 'spirit' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .sf-content-block__main' => 'background-color: {{VALUE}};',
                ]
            ]
        );

		$this->add_responsive_control(
			'border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'spirit' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .sf-content-block' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);

        $this->end_controls_section();
        
        $this->start_controls_section(
			'section_icon_style',
			[
				'label' => __( 'Icon', 'spirit' ),
				'tab' => Controls_Manager::TAB_STYLE
			]
		);

        $this->add_control(
            'icon_size',
            [
                'label' => esc_html__( 'Size', 'spirit' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'em'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 100
                    ]
                ],
                'selectors' => [ 
                    '{{WRAPPER}} .sf-content-block__icon' => 'font-size: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

		$this->add_control(
			'icon_color',
			[
				'label' => __( 'Icon Color', 'spirit' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .sf-content-block__icon' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_responsive_control(
			'icon_spacing',
			[
				'label' => __( 'Spacing', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .sf-content-block__icon' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				]
			]
		);

        $this->end_controls_section();
        
        $this->start_controls_section(
            'section_content_style',
            [
                'label' => esc_html__( 'Content', 'spirit' ),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label' => esc_html__( 'Title', 'spirit' ),
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .sf-content-block__title'
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label' => esc_html__( 'Content', 'spirit' ),
                'name' => 'content_typography',
                'selector' => '{{WRAPPER}} .sf-content-block__text',
            ]
        );

        $this->add_responsive_control(
            'content_spacing',
            [
                'label' => esc_html__( 'Spacing', 'spirit' ),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 300,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .sf-content-block__content' => 'margin-top: {{SIZE}}{{UNIT}}',
                ],
                'separator' => 'after'
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'label' => esc_html__( 'Tabs', 'spirit' ),
                'name' => 'tabs_typography',
                'selector' => '{{WRAPPER}} .sf-content-block__tab',
                
            ]
        );

        $this->add_responsive_control(
            'tabs_spacing',
            [
                'label' => esc_html__( 'Spacing', 'spirit' ),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 300
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} .sf-content-block__tabs' => 'margin-top: {{SIZE}}{{UNIT}}',
                ]
            ]
        );

		$this->add_responsive_control(
			'tabs_icon_size',
			[
				'label' => esc_html__( 'Tab Icon Size', 'spirit' ),
				'type' => Controls_Manager::SLIDER,
				'selectors' => [
					'{{WRAPPER}} .sf-content-block__tab-icon' => 'font-size: {{SIZE}}{{UNIT}};',
				]
			]
        );

        $this->end_controls_section();

		$this->start_controls_section(
			'section_button_style',
			[
                'label' => esc_html__( 'Button', 'spirit' ),
                'tab' => Controls_Manager::TAB_STYLE
			]
        );

		$this->add_control(
			'button_style',
			[
				'label' => __( 'Style', 'spirit' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'primary',
				'options' => sf_get_option_button_styles()
			]
        );

		$this->add_control(
			'button_size',
			[
				'label' => __( 'Size', 'spirit' ),
				'type' => Controls_Manager::SELECT,
				'default' => '',
				'options' => sf_get_option_button_sizes(),
				'style_transfer' => true
			]
        );
        
        $this->add_responsive_control(
            'button_spacing',
            [
                'label' => esc_html__( 'Spacing', 'spirit' ),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 300
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} .sf-content-block__button' => 'margin-top: {{SIZE}}{{UNIT}}',
                ]
            ]
        );

        $this->end_controls_section();
	}

	protected function render() {
        $settings = $this->get_settings();

        if ( empty( $settings['items'] ) ) {
            return;
        }

        
        $this->add_render_attribute( 'content-block', 'class', 'sf-content-block' );
        $layout         = ! empty( $settings['layout'] ) ? $settings['layout'] : 'tabs';
        $button_classes = 'btn';
        $button_text    = esc_html__( 'Read More', 'spirit' );
        $data_animation_settings = sf_get_data_animation_settings( $settings );

        if ( ! empty( $settings['button_style'] ) ) {
            $button_classes .= ' btn-' . $settings['button_style'];
        }

        if ( ! empty( $settings['button_size'] ) ) {
            $button_classes .= ' btn-' . $settings['button_size'];
        }

        $button_attr = [ 'class' => $button_classes ];
        
        if ( 'yes' == $settings['equal_height'] ) {
            $this->add_render_attribute( 'content-block', 'class', 'sf-content-block--equal-height' );
        }

        if ( 'list' === $layout ) {

            $this->add_render_attribute( 'content-blocks', 'class', 'sf-content-blocks' );

            if ( $data_animation_settings ) {
                $this->add_render_attribute(
                    'content-blocks',
                    'data-animation-settings',
                    $data_animation_settings
                );
            }

            echo '<div ' . $this->get_render_attribute_string( 'content-blocks' ) . '>';

            if ( ! empty( $data_animation_settings ) ) {
                $this->add_render_attribute( 'content-block', 'class', 'element-invisible' );
            }

            $order_class = ( 'yes' == $settings['reverse_order'] ) ? 'even' : 'odd';

            foreach ( $settings['items'] as $index => $item ) {
                
                $block_image_class = 'sf-content-block__image';
                $odd_or_even       = ( $index === 0 || $index % 2 === 0 ) ? 'even' : 'odd';
                
                if ( $odd_or_even === $order_class ) {
                    $block_image_class .= ' order-lg-1';
                }

                echo '<div ' . $this->get_render_attribute_string( 'content-block' ) . '>';
                
                if ( ! empty( $item['image']['id'] ) ) {
                    echo '<div class="' . esc_attr( $block_image_class ) . '">';
                    echo wp_get_attachment_image( $item['image']['id'], 'full' );
                    echo '</div>';
                } else if ( ! empty( $item['image']['url'] ) ) {
                    echo '<div class="' . esc_attr( $block_image_class ) . '">';
                    echo '<img src="'. esc_url( $item['image']['url'] ) . '">';
                    echo '</div>';
                }

                echo '<div class="sf-content-block__main">';
                echo '<div class="sf-content-block__content">';

                if ( ! empty( $item['icon']['value'] ) ) {
                    echo '<div class="sf-content-block__icon">';
                    Icons_Manager::render_icon( $item['icon'], [ 'aria-hidden' => 'true' ] );
                    echo '</div>';
                }
                
                if ( ! empty( $item['title'] ) ) {
                    echo '<h3 class="sf-content-block__title">' . esc_html( $item['title'] ) . '</h3>';
                }

                if ( ! empty( $item['content'] ) ) {
                    echo '<div class="sf-content-block__text">' . wp_kses_post( $item['content'] ) . '</div>';
                }

                if ( ! empty( $item['link']['url'] ) ) {
                    $button_attr['href'] = esc_url( $item['link']['url'] );
        
                    if ( $item['link']['is_external'] ) {
                        $button_attr['target'] = '_blank';
                    }
        
                    if ( $item['link']['nofollow'] ) {
                        $button_attr['rel'] = 'nofollow';
                    }

                    if ( empty( $settings['button_text'] ) ) {
                        $button_text = $settings['button_text'];
                    }

                    $button_attr_string = '';
                    foreach ( $button_attr as $name => $value ) {
                        $button_attr_string .= ' ' . $name . '="' . $value . '"';
                    }

                    echo '<div class="sf-content-block__button">';
                    echo '<a' . $button_attr_string . '>' . esc_html( $button_text ) . '</a>';
                    echo '</div>';
                }

                echo '</div>';
                echo '</div>';
                echo '</div>';
            }

            echo '</div>';

        } else {

            $prefix = $this->get_id();
            $this->add_render_attribute( 'content-block-image', 'class', 'sf-content-block__image' );
            $this->add_render_attribute( 'content-block-main', 'class', 'sf-content-block__main' );

            if ( 'yes' == $settings['reverse_order'] ) {
                $this->add_render_attribute( 'content-block-image', 'class', 'order-lg-1' );
            }

            if ( ! empty( $data_animation_settings ) ) {
                $this->add_render_attribute( 'content-block-image', 'class', 'element-invisible' );
                $this->add_render_attribute( 'content-block-main', 'class', 'element-invisible' );
            }

            if ( $data_animation_settings ) {
                $this->add_render_attribute(
                    'content-block',
                    'data-animation-settings',
                    $data_animation_settings
                );
            }

            echo '<div ' . $this->get_render_attribute_string( 'content-block' ) . '>';
            echo '<div ' . $this->get_render_attribute_string( 'content-block-image' ) . '>';

            foreach ( $settings['items'] as $index => $item ) {
                if ( ! empty( $item['image']['id'] ) ) {
                    echo wp_get_attachment_image( $item['image']['id'], 'full', false, array( 'data-index' => $prefix . $index ) );
                } else if ( ! empty( $item['image']['url'] ) ) {
                    echo '<img src="'. esc_url( $item['image']['url'] ) . '" data-index="' . $prefix . $index . '">';
                }
            }

            echo '</div>';
            echo '<div ' . $this->get_render_attribute_string( 'content-block-main' ) . '>';

            foreach ( $settings['items'] as $index => $item ) {
                
                echo '<div class="sf-content-block__content" data-index="' . esc_attr( $prefix . $index ) . '">';
                
                if ( ! empty( $item['icon']['value'] ) ) {
                    echo '<div class="sf-content-block__icon">';
                    Icons_Manager::render_icon( $item['icon'], [ 'aria-hidden' => 'true' ] );
                    echo '</div>';
                }
                
                if ( ! empty( $item['title'] ) ) {
                    echo '<h3 class="sf-content-block__title">' . esc_html( $item['title'] ) . '</h3>';
                }

                if ( ! empty( $item['content'] ) ) {
                    echo '<div class="sf-content-block__text">' . wp_kses_post( $item['content'] ) . '</div>';
                }

                echo '</div>';
            }

            echo '<ul class="sf-content-block__tabs">';

            foreach ( $settings['items'] as $index => $item ) {

                if ( ! empty( $item['title'] ) ) {
                    echo '<li class="sf-content-block__tab" data-tab="' . esc_attr( $prefix . $index ) . '">';
                    if ( ! empty( $item['tab_icon']['value'] ) ) {
                        Icons_Manager::render_icon( $item['tab_icon'], [ 'aria-hidden' => 'true', 'class' => 'sf-content-block__tab-icon' ] );
                    }
                    echo '<span>' . esc_html( $item['title'] ) . '</span>';
                    echo '</li>';
                }
            }
            echo '</ul>'; 

            echo '</div>';
            echo '</div>';
        }
    }

}

Plugin::instance()->widgets_manager->register( new SF_Content_Block() );