<?php
/**
 * SF Product importer
 *
 * @package Spirit_Framework
 * @version 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Include dependencies.
 */
if ( ! class_exists( 'WC_Product_CSV_Importer', false ) ) {
	include_once WC_ABSPATH . 'includes/import/class-wc-product-csv-importer.php';;
}
if ( ! class_exists( 'WC_Product_Importer', false ) ) {
    return;
}

class SF_Product_Importer extends WC_Product_Importer {
	
    /**
	 * Parsed data.
	 *
	 * @var array
	 */
    protected $parsed_data = [];

	/**
	 * Tracks current data being parsed.
	 *
	 * @var integer
	 */
	protected $parsing_data_index = 0;

	/**
	 * Tracks current data being parsed.
	 *
	 * @var bool
	 */
	protected $update_existing = false;

	/**
	 * Initialize importer.
	 *
	 * @param string $file   File to read.
	 */
    public function __construct( $file, $params = [] ) {
        if ( file_exists( $file ) ) {
            $file_content = file_get_contents( $file );
            $this->parsed_data = json_decode( $file_content, true );

            if ( ! empty( $this->parsed_data ) ) {
				$mappings = [];
                foreach ( $this->parsed_data as $key => $parsed_data ) {
                    $this->parsing_data_index = $key;
					
                    if ( isset( $parsed_data['id'] ) ) {
                        $this->parsed_data[ $key ]['id'] = $this->parse_id_field( $parsed_data['id'] );
                    }
                    if ( ! empty( $parsed_data['category_ids'] ) ) {
                        $this->parsed_data[ $key ]['category_ids'] = $this->parse_categories_field( $parsed_data['category_ids'] );
                    }
                    if ( ! empty( $parsed_data['parent_id'] ) ) {
                        $this->parsed_data[ $key ]['parent_id'] = $this->parse_relative_field( $parsed_data['parent_id'] );
                    }
                }
            }
        }
    }

	/**
	 * Parse relative field and return product ID.
	 *
	 * Handles `id:xx` and SKUs.
	 *
	 * If mapping to an id: and the product ID does not exist, this link is not
	 * valid.
	 *
	 * If mapping to a SKU and the product ID does not exist, a temporary object
	 * will be created so it can be updated later.
	 *
	 * @param string $value Field value.
	 *
	 * @return int|string
	 */
	public function parse_relative_field( $value ) {
		global $wpdb;

		if ( empty( $value ) ) {
			return '';
		}

		// IDs are prefixed with id:.
		if ( is_int( $value ) ) {
			$id = $value;

			// If original_id is found, use that instead of the given ID since a new placeholder must have been created already.
			$original_id = $wpdb->get_var( $wpdb->prepare( "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_original_id' AND meta_value = %s;", $id ) ); // WPCS: db call ok, cache ok.

			if ( $original_id ) {
				return absint( $original_id );
			}

			// See if the given ID maps to a valid product allready.
			$existing_id = $wpdb->get_var( $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_type IN ( 'product', 'product_variation' ) AND ID = %d;", $id ) ); // WPCS: db call ok, cache ok.

			if ( $existing_id ) {
				return absint( $existing_id );
			}

			// If we're not updating existing posts, we may need a placeholder product to map to.
			if ( ! $this->update_existing ) {
				$product = wc_get_product_object( 'simple' );
				$product->set_name( 'Import placeholder for ' . $id );
				$product->set_status( 'importing' );
				$product->add_meta_data( '_original_id', $id, true );
				$id = $product->save();
			}

			return $id;
		}

		$id = wc_get_product_id_by_sku( $value );

		if ( $id ) {
			return $id;
		}

		try {
			$product = wc_get_product_object( 'simple' );
			$product->set_name( 'Import placeholder for ' . $value );
			$product->set_status( 'importing' );
			$product->set_sku( $value );
			$id = $product->save();

			if ( $id && ! is_wp_error( $id ) ) {
				return $id;
			}
		} catch ( Exception $e ) {
			return '';
		}

		return '';
	}

	/**
	 * Parse the ID field.
	 *
	 * If we're not doing an update, create a placeholder product so mapping works
	 * for rows following this one.
	 *
	 * @param string $value Field value.
	 *
	 * @return int
	 */
    public function parse_id_field( $value ) {
		global $wpdb;

		$id = absint( $value );

		if ( ! $id ) {
			return 0;
		}

		// See if this maps to an ID placeholder already.
		$original_id = $wpdb->get_var( $wpdb->prepare( "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_original_id' AND meta_value = %s;", $id ) ); // WPCS: db call ok, cache ok.

		if ( $original_id ) {
			return absint( $original_id );
		}

		// Not updating? Make sure we have a new placeholder for this ID.
		if ( ! $this->update_existing ) {
			$row_sku      = isset( $this->parsed_data[ $this->parsing_data_index ]['sku'] ) ? $this->parsed_data[ $this->parsing_data_index ]['sku'] : '';
			$id_from_sku  = $row_sku ? wc_get_product_id_by_sku( $row_sku ) : '';

			// If row has a SKU, make sure placeholder was not made already.
			if ( $id_from_sku ) {
				return $id_from_sku;
			}

			$product = wc_get_product_object( 'simple' );
			$product->set_name( 'Import placeholder for ' . $id );
			$product->set_status( 'importing' );
			$product->add_meta_data( '_original_id', $id, true );

			// If row has a SKU, make sure placeholder has it too.
			if ( $row_sku ) {
				$product->set_sku( $row_sku );
			}
			$id = $product->save();
		}

		return $id && ! is_wp_error( $id ) ? $id : 0;
    }

	/**
	 * Parse relative comma-delineated field and return product ID.
	 *
	 * @param string $value Field value.
	 *
	 * @return array
	 */
	public function parse_relative_comma_field( $value ) {
		if ( empty( $value ) ) {
			return array();
		}

		return array_filter( array_map( array( $this, 'parse_relative_field' ), $this->explode_values( $value ) ) );
	}

	/**
	 * Parse a comma-delineated field from a CSV.
	 *
	 * @param string $value Field value.
	 *
	 * @return array
	 */
	public function parse_comma_field( $value ) {
		if ( empty( $value ) && '0' !== $value ) {
			return array();
		}

		$value = $this->unescape_data( $value );
		return array_map( 'wc_clean', $this->explode_values( $value ) );
	}

	/**
	 * Parse a category field from a CSV.
	 * Categories are separated by commas and subcategories are "parent > subcategory".
	 *
	 * @param string $value Field value.
	 *
	 * @return array of arrays with "parent" and "name" keys.
	 */
	public function parse_categories_field( $value ) {
		if ( empty( $value ) ) {
			return array();
		}

		$row_terms  = $this->explode_values( $value );
		$categories = array();

		foreach ( $row_terms as $row_term ) {
			$parent = null;
			$_terms = array_map( 'trim', explode( '>', $row_term ) );
			$total  = count( $_terms );

			foreach ( $_terms as $index => $_term ) {
				// Don't allow users without capabilities to create new categories.
				if ( ! current_user_can( 'manage_product_terms' ) ) {
					break;
				}

				$term = wp_insert_term( $_term, 'product_cat', array( 'parent' => intval( $parent ) ) );

				if ( is_wp_error( $term ) ) {
					if ( $term->get_error_code() === 'term_exists' ) {
						// When term exists, error data should contain existing term id.
						$term_id = $term->get_error_data();
					} else {
						break; // We cannot continue on any other error.
					}
				} else {
					// New term.
					$term_id = $term['term_id'];
				}

				// Only requires assign the last category.
				if ( ( 1 + $index ) === $total ) {
					$categories[] = $term_id;
				} else {
					// Store parent to be able to insert or query categories based in parent ID.
					$parent = $term_id;
				}
			}
		}

		return $categories;
	}

	/**
	 * Process importer.
	 *
	 * Do not import products with IDs or SKUs that already exist if option
	 * update existing is false, and likewise, if updating products, do not
	 * process rows which do not exist if an ID/SKU is provided.
	 *
	 * @return array
	 */
    public function import() {
        if ( empty( $this->parsed_data ) ) {
            return;
        }

        foreach ( $this->parsed_data as $parsed_data_key => $parsed_data ) {
			$id         = isset( $parsed_data['id'] ) ? absint( $parsed_data['id'] ) : 0;
			$sku        = isset( $parsed_data['sku'] ) ? $parsed_data['sku'] : '';
			$id_exists  = false;
			$sku_exists = false;

			if ( $id ) {
				$product   = wc_get_product( $id );
				$id_exists = $product && 'importing' !== $product->get_status();
			}

			if ( $sku ) {
				$id_from_sku = wc_get_product_id_by_sku( $sku );
				$product     = $id_from_sku ? wc_get_product( $id_from_sku ) : false;
				$sku_exists  = $product && 'importing' !== $product->get_status();
			}

			if ( $id_exists || $sku_exists ) {
				continue;
			}

			$result = $this->process_item( $parsed_data );
        }
    }

	/**
	 * Get imported data 
	 *
	 * @return array
	 */
	public function get_import_data() {
		return $this->parsed_data;
	}
}