<?php
/**
 * SF Demo Installer
 *
 * Demo data installer class
 */

class SF_Demo_Installer {

	private static $demo_key = '_sf_demo_content';
    private static $data_option_key = 'sf_demo_data';

    /**
     * post date - timestamp
     * @var integer
     */
    private static $post_date = 0;

    /**
     * start widget instance from 100
     * @var integer
     */
    private static $widget_instance = 100;

    /**
     * Uninstall demo
     * remove everything
     */
    public static function uninstall_demo() {
        // read demo data
        self::delete_content();
        self::restore_backup();
        delete_option( 'sf_demo_current' );
    }

    /**
     * Delete demo contents
     */
    public static function delete_content() {
        self::delete_posts();
        self::delete_media();
        self::delete_menus();
        self::delete_terms();
        self::delete_sidebars();
        self::delete_cpt_posts();
        self::delete_rev_sliders();
        delete_option( 'sf_demo_data' );
    }

    /**
     * Clear sidebar widgets and nav menu locations
     */
    public static function clear_content() {
        $active_sidebars = get_option( 'sidebars_widgets' );
        $theme_sidebars = apply_filters( 'sf_config_sidebars', array() );
        if ( ! empty( $theme_sidebars ) && is_array( $theme_sidebars ) ) {
            foreach ( $theme_sidebars as $slug ) {
                if ( isset( $active_sidebars[ $slug ] ) ) {
                    $active_sidebars[ $slug ] = array();
                }
            }
        }
        update_option( 'sidebars_widgets', $active_sidebars );

        set_theme_mod( 'nav_menu_locations', [] );
    }

    /**
     * Create backup
     */
    public static function create_backup() {
        $current = get_option( 'sf_demo_current' );

        // backup theme settings before installing the demo
        if ( empty( $current ) ) {
            $backup = [];

            // backup widgets
            $active_sidebars = get_option( 'sidebars_widgets' );
            $backup['sidebars_widgets'] = !empty( $active_sidebars ) ? $active_sidebars : '';
            $theme_sidebars = apply_filters( 'sf_config_sidebars', array() );
            if ( ! empty( $theme_sidebars ) && is_array( $theme_sidebars ) ) {
                foreach ( $theme_sidebars as $slug ) {
                    if ( isset( $active_sidebars[ $slug ] ) ) {
                        $active_sidebars[ $slug ] = array();
                    }
                }
            }
            update_option( 'sidebars_widgets', $active_sidebars );

            // backup menus
            $menus = get_theme_mod( 'nav_menu_locations' );
            $backup['nav_menu_locations'] = !empty( $menus ) ? $menus : '';
            set_theme_mod( 'nav_menu_locations', [] );

            // backup options
            $options = get_theme_mods();
            $backup['mods'] = !empty( $options ) ? $options : '';
            
            // create backup
            update_option( 'sf_demo_backup', $backup );
        }
    }

    /**
     * Restore backup
     */
    public static function restore_backup() {
        $backup = get_option( 'sf_demo_backup' );

        if ( ! empty( $backup ) ) {
            // restore nav menu locations
            if ( isset( $backup['nav_menu_locations'] ) ) {
                set_theme_mod( 'nav_menu_locations', $backup['nav_menu_locations'] );
            }
            // restore widgets
            if ( isset( $backup['sidebars_widgets'] ) ) {
                update_option( 'sidebars_widgets', $backup['sidebars_widgets'] );
            }
            // restore options
            if ( isset( $backup['options'] ) ) {
                $theme_slug = get_option( 'stylesheet' );
                update_option( "theme_mods_$theme_slug", $backup['options'] );
            }
            delete_option( 'sf_demo_backup' );
        }
    }

    /**
     * Set static homepages
     * @param array args 
     */
    public static function set_homepage( $args ) {
        if ( ! empty( $args['show_on_front'] ) ) {
            update_option( 'show_on_front', $args['show_on_front'] );
        }
        if ( ! empty( $args['page_on_front'] ) ) {
            update_option( 'page_on_front', $args['page_on_front'] );
        }
        if ( ! empty( $args['page_for_posts'] ) ) {
            update_option( 'page_for_posts', $args['page_for_posts'] );
        }
    }

    /**
     * Get post date - minus one day
     * @return post date
     */
    public static function get_post_date() {
        if ( 0 !== self::$post_date ) {
            self::$post_date = current_time( 'timestamp' );
        }
        self::$post_date -= 86400;
        return date( 'Y-m-d H:i:s', self::$post_date );
    }

    /**
     * Get post content
     * @return string lorem ipsum
     */
    public static function get_post_content() {
        return 'Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, imperdiet a, venenatis vitae, justo.
<blockquote>Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus.</blockquote>
Vivamus elementum semper nisi. Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue.

Curabitur ullamcorper ultricies nisi. Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing sem neque sed ipsum. Nam quam nunc, blandit vel, luctus pulvinar, hendrerit id, lorem. Maecenas nec odio et ante tincidunt tempus. Donec vitae sapien ut libero venenatis faucibus. Nullam quis ante. Etiam sit amet orci eget eros faucibus tincidunt. Duis leo. Sed fringilla mauris sit amet nibh. Donec sodales sagittis magna. Sed consequat, leo eget bibendum sodales, augue velit cursus nunc, quis gravida magna mi a libero. Fusce vulputate eleifend sapien. Vestibulum purus quam, scelerisque ut, mollis sed, nonummy id, metus. Nullam accumsan lorem in dui. Cras ultricies mi eu turpis.';
    }

    /**
     * Check if data key exists
     *
     * @param string $key
     * @return bool
     */
    public static function record_exists( $key = '' ) {
        if ( ! empty( $key ) ) {
            $demo_data = get_option( self::$data_option_key );
            if ( ! empty( $demo_data[ $key ] ) ) {
                return true;
            }
        }
        return false;
    }

    /**
     * Add data record
     *
     * @param string $key
     * @param mixed $value
     * @return void
     */
    public static function add_record( $key, $value = '', $override = false ) {
        if ( ! empty( $key ) ) {
            $demo_data = get_option( self::$data_option_key );
            if ( ! empty( $demo_data ) && is_array( $demo_data ) ) {
                if ( !isset( $demo_data[ $key ] ) || $override ) {
                    $demo_data[ $key ] = $value;
                }
            } else {
                $demo_data = [ $key => $value ];
            }
            update_option( self::$data_option_key, $demo_data );
        }
    }

    /**
     * Delete a record
     *
     * @param string $key
     * @return void
     */
    public static function delete_record( $key ) {
        if ( ! empty( $key ) ) {
            $demo_data = get_option( self::$data_option_key );
            if ( ! empty( $demo_data ) && is_array( $demo_data ) && isset( $demo_data[ $key ] ) ) {
                unset( $demo_data[ $key ] );
            }
            update_option( self::$data_option_key, $demo_data );
        }
    }

    public static function get_data_value( $key ) {
        $value = '';
        $demo_data = get_option( 'sf_demo_data' );
        if ( ! empty( $demo_data[ $key ] ) ) {
            $value = $demo_data[ $key ];
        }
        return $value;
    }

    public static function get_image_url( $id, $size = 'full' ) {
        $image_url = '';
        if ( ! empty( $id ) ) {
            $image     = wp_get_attachment_image_src( $id, $size );
            $image_url = !empty( $image ) ? $image[0] : '';
        }
        return $image_url;
    }

    /**
     * Add post
     * 
     * @param array $data_key data key
     * @param array $args arguments
     * @return int
     */
    public static function add_post( $data_key, $args ) {
        if ( self::record_exists( $data_key ) ) {
            return;
        }

        $new_post = array( 
            'post_title' => $args['title'],
            'post_status' => 'publish',
            'post_type' => 'post',
            'comment_status' => 'open',
            'tags_input' => isset( $args['tags'] ) ? $args['tags'] : array(),
            'post_date' => self::get_post_date(),
        );

        if ( ! empty( $args['post_id'] ) ) {
            $new_post['import_id'] = absint( $args['post_id'] );
        }

        if ( ! empty( $args['post_content'] ) ) {
            $new_post['post_content'] = $args['post_content'];
        } else {
            $new_post['post_content'] = self::get_post_content();
        }

        if ( ! empty( $args['post_excerpt'] ) ) {
            $new_post['post_excerpt'] = $args['post_excerpt'];
        }

        $post_id = wp_insert_post( $new_post );

        if ( is_wp_error( $post_id ) || $post_id === 0 ) {
            return;
        }

        if ( ! empty( $args['categories'] ) ) {
            wp_set_object_terms( $post_id, $args['categories'], 'category' );
        }

        if ( ! empty( $args['post_format'] ) ) {
            set_post_format( $post_id, $args['post_format'] );
        }

        if ( ! empty( $args['featured'] ) ) {
            set_post_thumbnail( $post_id, $args['featured'] );
        }

        if ( isset( $args['sticky'] ) && $args['sticky'] ) {
            stick_post( $post_id );
        }
        
        if ( ! empty( $args['post_meta'] ) ) {
            foreach ( $args['post_meta'] as $key => $value ) {
                update_post_meta( $post_id, $key, $value );
            }
        }

        // add our demo custom meta field and use it for deletion
        update_post_meta( $post_id, self::$demo_key, true );
        
        // add record
        self::add_record( $data_key, $post_id );

        return $post_id;
    }

    /**
     * Add page
     * 
     * @param array $data_key data key
     * @param array $args arguments
     * @return int
     */
    public static function add_page( $data_key, $args ) {
        if ( self::record_exists( $data_key ) ) {
            return;
        }

        $new_page = array( 
            'post_title' => isset( $args['title'] ) ? $args['title'] : 'Page',
            'post_status' => 'publish',
            'post_type' => 'page',
            'ping_status' => 'closed',
            'comment_status' => 'closed',
        );

        if ( ! empty( $args['post_id'] ) ) {
            $new_page['import_id'] = absint( $args['post_id'] );
        }

        if ( ! empty( $args['post_content'] ) ) {
            $new_page['post_content'] = $args['post_content'];
        } else {
            $new_page['post_content'] = '';
        }

        $post_id = wp_insert_post( $new_page );

        if ( is_wp_error( $post_id ) || $post_id === 0 ) {
            return;
        }

        if ( ! empty( $args['post_meta'] ) ) {
            foreach ( $args['post_meta'] as $key => $value ) {
                update_post_meta( $post_id, $key, $value );
            }
        }

        update_post_meta( $post_id, self::$demo_key, true );

        // add record
        self::add_record( $data_key, $post_id );

        return $post_id;
    }

    /**
     * Add custom post type
     * 
     * @param array $data_key data key
     * @param array $args arguments
     * @return int
     */
    public static function add_custom_post_type( $data_key, $args ) {
        if ( self::record_exists( $data_key ) ) {
            return;
        }

        $new_post = array(
            'post_title' => $args['title'],
            'post_type' => $args['post_type'],
            'post_status' => 'publish',
            'ping_status' => 'closed',
            'comment_status' => 'closed',
            'post_date' => self::get_post_date(),
        );

        if ( ! empty( $args['post_content'] ) ) {
            $new_post['post_content'] = $args['post_content'];
        }

        if ( ! empty( $args['post_excerpt'] ) ) {
            $new_post['post_excerpt'] = $args['post_excerpt'];
        }

        if ( ! empty( $args['post_id'] ) ) {
            $new_post['import_id'] = absint( $args['post_id'] );
        }

        $post_id = wp_insert_post( $new_post );

        if ( is_wp_error( $post_id ) || $post_id === 0 ) {
            return;
        }

        if ( ! empty( $args['featured'] ) ) {
            set_post_thumbnail( $post_id, $args['featured'] );
        }

        if ( ! empty( $args['taxonomy'] ) ) {
            wp_set_object_terms( $post_id, $args['taxonomy'][0], $args['taxonomy'][1] );
        }

        if ( ! empty( $args['post_meta'] ) ) {
            foreach ( $args['post_meta'] as $key => $value ) {
                update_post_meta( $post_id, $key, $value );
            }
        }

        // add our demo custom meta field and use it for deletion
        update_post_meta( $post_id, self::$demo_key, true );

        // add record
        self::add_record( $data_key, $post_id );

        return $post_id;
    }

    /**
     * Add elementor page
     * 
     * @param array $data_key data key
     * @param array $args arguments
     * @return int
     */
    public static function add_elementor_page( $data_key, $args ) {
        if ( self::record_exists( $data_key ) || ! defined( 'ELEMENTOR_VERSION' ) || empty( $args['file'] ) || ! file_exists( $args['file'] ) ) {
            return;
        }

        // Elementor import removes the file after import, so we need to use a copy instead of the original file
        $file_name = basename( $args['file'] );
        $uploads = wp_upload_dir();
        $temp_dir = trailingslashit( $uploads['basedir'] ) . 'sf_demo_temp';

        if ( ! file_exists( $temp_dir ) ) {
            wp_mkdir_p( $temp_dir );
        }

        $path = $temp_dir . $file_name;
        copy( $args['file'], $path );

        if ( ! file_exists( $path ) ) {
            return;
        }

        $source = \Elementor\Plugin::$instance->templates_manager->get_source( 'local' );
        $item = $source->import_template( $file_name, $path );

        if ( is_wp_error( $item ) ) {
            return;
        }
        $post_id = isset( $item[0]['template_id'] ) ? $item[0]['template_id'] : 0;
        $post_type = ! empty( $args['post_type'] ) ? $args['post_type'] : 'page';
        $post_data = array(
            'ID' => $post_id,
            'post_type' => $post_type
        );

        if ( ! empty( $args['title'] ) ) {
            $post_data['post_title'] = $args['title'];
        }

        if ( ! empty( $args['slug'] ) ) {
            $post_data['post_name'] = $args['slug'];
        }

        if ( ! empty( $args['post_excerpt'] ) ) {
            $post_data['post_excerpt'] = $args['post_excerpt'];
        }

        if ( ! empty( $post_id ) ) {
            wp_update_post( $post_data );
        } else {
            return 0;
        }

        if ( ! empty( $args['featured'] ) ) {
            set_post_thumbnail( $post_id, $args['featured'] );
        }

        if ( ! empty( $args['taxonomy'] ) ) {
            wp_set_object_terms( $post_id, $args['taxonomy'][0], $args['taxonomy'][1] );
        }

        if ( ! empty( $args['post_meta'] ) ) {
            foreach ( $args['post_meta'] as $key => $value ) {
                update_post_meta( $post_id, $key, $value );
            }
        }

        // add our demo custom meta field and use it for deletion
        update_post_meta( $post_id, self::$demo_key, true );
        
        // add record
        self::add_record( $data_key, $post_id );

        return $post_id;
    }

    public static function update_elementor_page_settings( $settings ) {
        if ( ! defined( 'ELEMENTOR_VERSION' ) || empty( $settings ) ) {
            return;
        }

        $kit_id = get_option( 'elementor_active_kit' );
        
        if ( ! empty( $kit_id ) ) {
            $page_settings = get_post_meta( $kit_id, '_elementor_page_settings', true );

            if ( ! empty( $page_settings ) && is_array( $page_settings ) ) {
                $page_settings = array_merge( $page_settings, $settings );
            } else {
                $page_settings = $settings;
            }
    
            update_post_meta( $kit_id, '_elementor_page_settings', $page_settings );

            \Elementor\Plugin::instance()->files_manager->clear_cache();
        }
    }

    /**
     * Add term
     * @param string $term_name term name
     * @param string $taxonomy  taxonomy name
     * @param array  $args
     * @param array  $term_meta term meta
     * @return int
     */
    public static function add_term( $term_name, $taxonomy, $args = array(), $term_meta = array() ) {
        $term = term_exists( $term_name, $taxonomy );
        if ( $term !== 0 && $term !==  null ) {
            if ( is_array( $term ) ) {
                return $term['term_id'];
            } else {
                wp_update_term( $term, $taxonomy, $args );
                return $term;
            }
        }

        // set parent id
        if ( ! empty( $args['parent'] ) ) {
            $term = get_term( $args['parent'], $taxonomy );
            if ( ! is_wp_error( $term ) && $term ) {
                $args['parent'] = $term->term_id;
            } else {
                $args['parent'] = 0;
            }
        }

        $term = wp_insert_term( $term_name, $taxonomy, $args );
        $term_id = 0;

        if ( ! is_wp_error( $term ) ) {
            $term_id = $term['term_id'];

            if ( ! empty( $term_meta ) && is_array( $term_meta ) ) {
                foreach ( $term_meta as $key => $value ) {
                    add_term_meta( $term_id, $key, $value, true );
                }
            }

            // save term
            $data_terms = self::get_data_value( 'terms' );
            if ( ! empty( $data_terms ) && is_array( $data_terms ) ) {
                $data_terms[ $taxonomy ][] = $term_id;
            } else {
                $data_terms = [];
                $data_terms[ $taxonomy ] = [ $term_id ];
            }
            self::add_record( 'terms', $data_terms, true );
        }

        return $term_id;
    }

    /**
     * Add a category
     * 
     * @param array $args
     * @return int
     */
    public static function add_category( $args ) {
        // check if the category exist
        $term = term_exists( $args['name'], 'category' );
        if ( $term !== 0 && $term !==  null ) {
            if ( is_array( $term ) ) {
                return $term['term_id'];
            } else {
                wp_update_term( $term, 'category', $args );
                return $term;
            }
        }

        // get parent id
        if ( isset( $args['parent_id'] ) && is_string( $args['parent_id'] ) ) {
            $term = get_category_by_slug( $args['parent_id'] );
            if ( $term ) {
                $args['parent_id'] = $term->term_id;
            } else {
                $args['parent_id'] = 0;
            }
        }

        // create new category
        $cat_id = wp_create_category( $args['name'], $args['parent_id'] );
        
        if ( ! is_wp_error( $cat_id ) && $cat_id !== 0 ) {

            if ( ! empty( $args['description'] ) ) {
                wp_update_term( $cat_id, 'category', array( 'description' => $args['description'] ) );
            }

            // save term
            $data_terms = self::get_data_value( 'terms' );
            if ( ! empty( $data_terms ) && is_array( $data_terms ) ) {
                $data_terms['category'][] = $cat_id;
            } else {
                $data_terms['category'] = [ $cat_id ];
            }
            self::add_record( 'terms', $data_terms, true );
        } else {
            $cat_id = 0;
        }

        return $cat_id;
    }

    /**
     * Add a sidebar
     * 
     * @param string $name sidebar name
     * @param string $desc sidebar description
     * @return int
     */
    public static function add_sidebar( $name, $desc = '' ) {

        if ( empty( $name ) ) {
            return;
        }

        if ( !isset( $desc ) ) {
            $desc = '';
        }

        $sidebar_id = sanitize_title_with_dashes( $name );
        $sidebars = get_option( 'sf_custom_sidebars' );

        // if a sidebar exist return the sidebar id
        if ( ! empty( $sidebars ) && array_key_exists( $sidebar_id, $sidebars ) ) {
            return $sidebar_id;
        }

        // update custom sidebar option
        $sidebars[$sidebar_id] = array( 'name' => $name, 'desc' => $desc );
        update_option( 'sf_custom_sidebars', $sidebars );

        // update sidebar to active sidebar list
        $active_sidebars = get_option( 'sidebars_widgets' );
        $active_sidebars[$sidebar_id] = array();
        update_option( 'sidebars_widgets', $active_sidebars );

        // save sidebar
        $data_sidebars = self::get_data_value( 'sidebars' );
        if ( ! empty( $data_sidebars ) && is_array( $data_sidebars ) ) {
            $data_sidebars[] = $sidebar_id;
        } else {
            $data_sidebars = [ $sidebar_id ];
        }
        self::add_record( 'sidebars', $data_sidebars, true );

        return $sidebar_id;
    }

    /**
     * Add widget to sidebar
     * 
     * @param string $sidebar_id   sidebar id
     * @param string $widget_name  widget name
     * @param array $args          widget args
     */
    public static function add_widget_to_sidebar( $sidebar_id, $widget_name = '', $args = array() ) {

        if ( empty( $sidebar_id ) || empty( $widget_name ) ) {
            return;
        }

        // update widget to active sidebars list
        $active_sidebars = get_option( 'sidebars_widgets' );
        $active_sidebars[ $sidebar_id ][] = $widget_name . '-' . self::$widget_instance;
        update_option( 'sidebars_widgets', $active_sidebars );

        // update widget instance
        $widget = get_option( 'widget_' . $widget_name );
        $widget[ self::$widget_instance ] = $args;
        update_option( "widget_$widget_name", $widget );

        self::$widget_instance ++;
    }
 

    /**
     * Add menu
     * 
     * @param string $data_key data key
     * @param string $menu_name menu name
     * @param array  $locations location data
     * @return int
     */
    public static function add_menu( $menu_name = '', $locations = array() ) {
        $menu_obj = wp_get_nav_menu_object( $menu_name );
        $menu_id = 0;

        // check if a menu exists return menu id or create a new one
        if ( $menu_obj === false ) {
            $menu_id = wp_create_nav_menu( $menu_name );
            if ( is_wp_error( $menu_id ) ) {
                return false;
            }
        } else {
            $menu_id = $menu_obj->term_id;
        }

        // set menu locations
        $menus = get_theme_mod( 'nav_menu_locations' );
        if ( ! empty( $locations ) && is_array( $locations ) ) {
            foreach ( $locations as $location ) {
                $menus[ $location ] = $menu_id;
            }
            set_theme_mod( 'nav_menu_locations', $menus );
        }

        // save menu
        $data_menus = self::get_data_value( 'menus' );
        if ( ! empty( $data_menus ) && is_array( $data_menus ) ) {
            $data_menus[] = $menu_id;
        } else {
            $data_menus = [ $menu_id ];
        }
        self::add_record( 'menus', $data_menus, true );

        return $menu_id;
    }

    /**
     * Add menu item link
     * 
     * @param int $menu_id  menu id
     * @param array $args   menu meta
     */
    public static function add_menu_item_link( $menu_id, $args ) {
        $menu_item_data = array(
            'menu-item-object' => '',
            'menu-item-parent-id' => $args['parent_id'],
            'menu-item-type' => 'custom',
            'menu-item-title' => isset( $args['title'] ) ? $args['title'] : 'Link',
            'menu-item-url' => isset( $args['url'] ) ? $args['url'] : '#',
            'menu-item-classes' => isset( $args['classes'] ) ? $args['classes'] : '',
            'menu-item-status' => 'publish',
        );
        $menu_item_id = wp_update_nav_menu_item( $menu_id, 0, $menu_item_data );

        self::add_custom_nav_fields( $menu_item_id, $args );

        return $menu_item_id;
    }

    /**
     * Add menu item category
     * 
     * @param int $menu_id  menu id
     * @param array $args   menu meta
     */
    public static function add_menu_item_category( $menu_id, $args ) {
        $menu_item_data = array(
            'menu-item-parent-id' => $args['parent_id'],
            'menu-item-title' => isset( $args['title'] ) ? $args['title'] : get_cat_name( $args['category'] ),
            'menu-item-object-id' => $args['category'],
            'menu-item-url' => get_category_link( $args['category'] ),
            'menu-item-object' => 'category',
            'menu-item-type' => 'taxonomy',
            'menu-item-status' => 'publish'
        );
        $menu_item_id = wp_update_nav_menu_item( $menu_id, 0, $menu_item_data );

        self::add_custom_nav_fields( $menu_item_id, $args );

        return $menu_item_id;
    }

    /**
     * Add menu item post type archive
     * 
     * @param int $menu_id  menu id
     * @param array $args   menu meta
     */
    public static function add_menu_item_post_type_archive( $menu_id, $args ) {
        if ( empty( $args['post_type'] ) ) {
            return 0;
        }

        if ( empty( $args['title'] ) ) {
            $title = get_post_type_object( $args['post_type'] );
            $title = $title->labels->archives;
        } else {
            $title = $args['title'];
        }

        $menu_item_data = array(
            'menu-item-parent-id' => $args['parent_id'],
            'menu-item-title' => $title,
            'menu-item-object' => $args['post_type'],
            'menu-item-type' => 'post_type_archive',
            'menu-item-status' => 'publish'
        );
        $menu_item_id = wp_update_nav_menu_item( $menu_id, 0, $menu_item_data );

        self::add_custom_nav_fields( $menu_item_id, $args );

        return $menu_item_id;
    }

    /**
     * Add menu item page
     * 
     * @param int $menu_id  menu id
     * @param array $args   menu meta
     */
    public static function add_menu_item_page( $menu_id, $args ) {
        if ( !isset( $args['post_id'] ) || FALSE === get_post_status( $args['post_id'] ) ) {
            return 0;
        }

        $menu_item_data = array(
            'menu-item-parent-id' => isset( $args['parent_id'] ) ? $args['parent_id'] : 0,
            'menu-item-title' => isset( $args['title'] ) ? $args['title'] : get_the_title( $args['post_id'] ),
            'menu-item-object-id' => $args['post_id'],
            'menu-item-object' => ! empty( $args['post_type'] ) ? $args['post_type'] : 'page',
            'menu-item-type' => 'post_type',
            'menu-item-status' => 'publish'
        );
        $menu_item_id = wp_update_nav_menu_item( $menu_id, 0, $menu_item_data );

        self::add_custom_nav_fields( $menu_item_id, $args );

        return $menu_item_id;
    }

    /**
     * add menu item post
     * 
     * @param int $menu_id   menu id
     * @param array $args    menu meta
     */
    public static function add_menu_item_post( $menu_id, $args ) {
        $menu_item_data = array(
            'menu-item-parent-id' => $args['parent_id'],
            'menu-item-title' => isset( $args['title'] ) ? $args['title'] : get_the_title( $args['post_id'] ),
            'menu-item-object-id' => $args['post_id'],
            'menu-item-object' => 'post',
            'menu-item-type' => 'post_type',
            'menu-item-status' => 'publish'
        );
        $menu_item_id = wp_update_nav_menu_item( $menu_id, 0, $menu_item_data );

        self::add_custom_nav_fields( $menu_item_id, $args );

        return $menu_item_id;
    }

    /**
     * add menu item post_format
     * 
     * @param int $menu_id   menu id
     * @param array $args    menu meta
     */
    public static function add_menu_item_post_format( $menu_id, $args ) {
        $term = get_term_by( 'slug', 'post-format-' . $args['post_format'], 'post_format' );

        if ( !$term ) {
            return;
        }

        $menu_item_data = array(
            'menu-item-parent-id' => $args['parent_id'],
            'menu-item-title' => isset( $args['title'] ) ? $args['title'] : $term->name,
            'menu-item-object-id' => $term->term_id,
            'menu-item-object' => 'post_format',
            'menu-item-type' => 'taxonomy',
            'menu-item-status' => 'publish'
        );
        $menu_item_id = wp_update_nav_menu_item( $menu_id, 0, $menu_item_data );

        self::add_custom_nav_fields( $menu_item_id, $args );

        return $menu_item_id;
    }

    public static function add_custom_nav_fields( $menu_item_id, $args = array() ) {
        if ( isset( $args['_sf_megamenu'] ) ) {
            update_post_meta( $menu_item_id, '_menu_item_sf_megamenu', $args['_sf_megamenu'] );
        }
        // custom fields for megamenu
        update_post_meta( $menu_item_id, self::$demo_key, true );
    }

    /**
     * Add image to library
     * @param string $url image url
     */
    public static function add_media_image( $data_key, $url ) {
        if ( self::record_exists( $data_key ) ) {
            return;
        }

        if ( !function_exists('media_handle_upload') ) {
            require_once( ABSPATH . 'wp-admin/includes/image.php' );
            require_once( ABSPATH . 'wp-admin/includes/file.php' );
            require_once( ABSPATH . 'wp-admin/includes/media.php' );
        }

        $tmp = download_url( $url );
        $desc = 'Demo Image';
        $file_array = array();

        // Set variables for storage
        // fix file filename for query strings
        preg_match( '/[^\?]+\.(jpg|jpe|jpeg|gif|png)/i', $url, $matches );
        $file_array['name'] = basename( $matches[0] );
        $file_array['tmp_name'] = $tmp;

        // If error storing temporarily, unlink
        if ( is_wp_error( $tmp ) ) {
            @unlink( $file_array['tmp_name'] );
            $file_array['tmp_name'] = array();
            echo $tmp->get_error_message();
            return;
        }

        // do the validation and storage stuff
        $id = media_handle_sideload( $file_array, 0, $desc );

        // If error storing permanently, unlink
        if ( is_wp_error( $id ) ) {
            @unlink( $file_array['tmp_name'] );
            echo $id->get_error_message();
            return;
        } else {
            update_post_meta( $id, self::$demo_key, true );
            self::add_record( $data_key, $id );
        }

        return $id;
    }

    /**
     * Import Woocommerce products from json
     *
     * @param string $file
     * @return void
     */
    public static function import_wc_products( $file ) {
        $importer = new SF_Product_Importer( $file );
        $importer->import();
        $imported_data = $importer->get_import_data();

        foreach ( $imported_data as $index => $product_data ) {
            $data_index = $index ++;
            // record products
            if ( isset( $product_data['id'] ) ) {
                update_post_meta( $product_data['id'], self::$demo_key, true );
                self::add_record( 'product_' . $data_index, $product_data['id'] );
            }
            // record product categories
            if ( ! empty( $product_data['category_ids'] ) ) {
                foreach ( $product_data['category_ids'] as $term_id ) {
                    $data_terms = self::get_data_value( 'terms' );
                    if ( ! empty( $data_terms ) && is_array( $data_terms ) ) {
                        $data_terms[ 'product_cat' ][] = $term_id;
                    } else {
                        $data_terms = [];
                        $data_terms[ 'product_cat' ] = [ $term_id ];
                    }
                    self::add_record( 'terms', $data_terms, true );
                }
            }
        }
    }

    /**
     * Import Revolution slider
     * 
     * @param array $file  file path
     * @param string $source local/remote
     */
    public static function import_rev_slider( $file, $source = 'local' ) {
        if ( ! empty( $file ) && class_exists( 'RevSliderFront' ) ) {
            $absolute_path = __FILE__;
            $path_to_file = explode( 'wp-content', $absolute_path );
            $path_to_wp = $path_to_file[0];

            require_once( $path_to_wp . '/wp-load.php' );
            require_once( $path_to_wp . '/wp-includes/functions.php' );

            if ( 'remote' == $source ) {
                $sliders_folder = wp_normalize_path( self::$demo_dir .'/sliders' );
                // create sliders folder if not exist
                if ( ! file_exists( $sliders_folder ) ) {
                    wp_mkdir_p( $sliders_folder );
                }
            }

            $revslider = new RevSlider();

            if ( 'remote' == $source ) {
                $path = parse_url( $file, PHP_URL_PATH );
                $file_name = basename( $path );
                $local_file = wp_normalize_path( $sliders_folder .'/'. $file_name );

                if ( file_exists( $local_file ) ) {
                    $file = $local_file;
                } else {
                    $response = sf_wp_get_http( $file, $local_file );
                    if ( !$response ) {
                        return 0;
                    }
                    $file = $local_file;
                }
            }
            
            if ( file_exists( $file ) ) {
                $result = $revslider->importSliderFromPost( true, true, $file );
                
                if ( true === $result['success'] ) {
                    // save slider
                    $data_sliders = self::get_data_value( 'sliders' );
                    if ( ! empty( $data_sliders ) && is_array( $data_sliders ) ) {
                        $data_sliders[] = $result['sliderID'];
                    } else {
                        $data_sliders = [ $result['sliderID'] ];
                    }
                    self::add_record( 'sliders', $data_sliders, true );

                    return $result['sliderID'];
                }
            }
        }
    }

    /**
     * Removes Revolution sliders.
     *
     * @access private
     */
    public static function delete_rev_sliders() {
        $sliders = self::get_data_value( 'sliders' );

        if ( empty( $sliders ) || !is_array( $sliders ) ) {
            return;
        }

        if ( class_exists( 'RevSliderFront' ) ) {
            $revslider = new RevSlider();
            foreach ( $sliders as $slider_id ) {
                $revslider->initByID( $slider_id );
                $revslider->deleteSlider();
            }
        }
    }

    /**
     * Delete posts
     */
    public static function delete_posts() {
        $query = new WP_Query( array(
            'post_type' => 'any',
            'meta_key'  => self::$demo_key,
            'posts_per_page' => '-1'
        ));
        
        if ( ! empty( $query->posts ) ) {
            foreach ( $query->posts as $post ) {
                $response = wp_delete_post( $post->ID, true );
                if ( $response === false ) {
                    echo 'Fail to delete post: ' . $post->ID;
                }
            }
        }
        wp_reset_postdata();
    }

    /**
     * Delete custom post type posts
     */
    public static function delete_cpt_posts() {
        $query = new WP_Query( array(
            'post_type' => array(
                'wpcf7_contact_form',
                'mc4wp-form',
                'tribe_events',
                'tribe_organizer',
                'tribe_venue'
            ),
            'meta_key'  => self::$demo_key,
            'posts_per_page' => '-1'
        ));
        
        if ( ! empty( $query->posts ) ) {
            foreach ( $query->posts as $post ) {
                $response = wp_delete_post( $post->ID, true );
                if ( $response === false ) {
                    echo 'Fail to delete post: ' . $post->ID;
                }
            }
        }
        wp_reset_postdata();
    }

    /**
     * Delete media files
     */
    public static function delete_media() {
        $query = new WP_Query(
            array(
            'post_type' => array( 'attachment' ),
            'post_status' => 'inherit',
            'meta_key'  => self::$demo_key,
            'posts_per_page' => '-1'
        ));
        if ( ! empty( $query->posts ) ) {
            foreach ( $query->posts as $post ) {
                $response = wp_delete_attachment( $post->ID, true );
                if ( $response === false ) {
                    echo 'Fail to delete attachment: ' . $post->ID;
                }
            }
        }
        wp_reset_postdata();
    }

    /**
     * Delete menus
     */
    public static function delete_menus() {
        $menus = self::get_data_value( 'menus' );
        
        if ( ! empty( $menus ) ) {
            foreach ( $menus as $menu ) {
                wp_delete_nav_menu( $menu );
            }
        }
    }

    /**
     * Delete terms
     */
    public static function delete_terms() {
        $terms = self::get_data_value( 'terms' );

        if ( ! empty( $terms ) ) {
            foreach ( $terms as $taxonomy => $term_ids ) {
                if ( ! empty( $term_ids ) ) {
                    foreach ( $term_ids as $term_id ) {
                        $response = wp_delete_term( $term_id, $taxonomy );
                        if ( $response === false ) {
                            echo 'Fail to delete '. $taxonomy .': ' . $term_id;
                        }
                    }
                }
            }
        }
    }

    /**
     * Delete sidebars
     */
    public static function delete_sidebars() {
        $sidebars = self::get_data_value( 'sidebars' );
        $custom_sidebars = get_option( 'sf_custom_sidebars' );

        if ( ! empty( $sidebars ) && ! empty( $custom_sidebars ) ) {
            foreach ( $sidebars as $sidebar_id ) {
                if ( isset( $custom_sidebars[ $sidebar_id ] ) ) {
                    unset( $custom_sidebars[ $sidebar_id ] );
                }
            }
            update_option( 'sf_custom_sidebars', $custom_sidebars );
        }
    }

    /**
     * Install theme_mods
     * @param string $path  theme option path
     */
    public static function install_theme_options( $path ) {
        $options_json = file_get_contents( $path );
        $mods = json_decode( $options_json, true );
        $theme_slug = get_option( 'stylesheet' );
        update_option( "theme_mods_$theme_slug", $mods );
    }

    /**
     * Update theme_mods
     * @param array $options new options array
     */
    public static function update_theme_options( $options = array() ) {
        $mods = get_theme_mods();
        if ( ! empty( $options ) ) {
            foreach ( $options as $option => $value ) {
                $mods[ $option ] = $value;
            }
            $theme_slug = get_option( 'stylesheet' );
            update_option( "theme_mods_$theme_slug", $mods );
        }
    }

    /**
     * Update wp_options
     * @param array $options new options array
     */
    public static function update_options( $options = array() ) {
        foreach ( $options as $key => $value ) {
            update_option( $key, $value );
        }
    }
}