<?php
/**
* Spirit Framework
* 
*/

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

if ( ! class_exists( 'Spirit_Framework' ) ) {

	final class Spirit_Framework {

		/**
		 * Static property to hold our singleton instance
		 */
		private static $instance = null;

		/**
		 * Settings
		 */
		private static $settings = null;

		/**
		 * If an instance exists, this returns it.  If not, it creates one and
		 * retuns it.
		 *
		 * @return Spirit_Framework
		 */
		public static function instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self;
				self::$instance->init_settings();
				self::$instance->includes();
				self::$instance->hooks();
			}
			return self::$instance;
		}

		/**
		 * Throw error on object clone.
		 *
		 * The whole idea of the singleton design pattern is that there is a single
		 * object therefore, we don't want the object to be cloned.
		 *
		 * @access protected
		 * @return void
		 */
		public function __clone() {
			// Cloning instances of the class is forbidden.
			_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'spirit' ), '1.0' );
		}

		/**
		 * Disable unserializing of the class.
		 *
		 * @access protected
		 * @return void
		 */
		public function __wakeup() {
			// Unserializing instances of the class is forbidden.
			_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'spirit' ), '1.0' );
		}

		/**
		 * Constants
		 */
		private function init_settings() {
			if ( is_null( self::$settings ) ) {
				self::$settings = wp_parse_args(
					get_option( 'sf_settings' ),
					self::get_default_settings()
				);
			}
		}

		/**
		 * Include files
		 */
		private function includes() {
			// load required files
			require_once SF_FRAMEWORK_DIR . 'includes/libs/kirki/kirki.php';
			require_once SF_FRAMEWORK_DIR . 'includes/customizer/customizer.php';
			require_once SF_FRAMEWORK_DIR . 'includes/formatting.php';
			require_once SF_FRAMEWORK_DIR . 'includes/core-functions.php';
			require_once SF_FRAMEWORK_DIR . 'includes/helpers.php';
			require_once SF_FRAMEWORK_DIR . 'includes/template-hooks.php';
			require_once SF_FRAMEWORK_DIR . 'includes/custom/post-types.php';
			require_once SF_FRAMEWORK_DIR . 'includes/custom/taxonomies.php';
			require_once SF_FRAMEWORK_DIR . 'includes/integration/custom-fonts.php';
			require_once SF_FRAMEWORK_DIR . 'includes/admin/class-sf-admin.php';

			if ( SF()->get_setting( 'enable_custom_sidebars' ) ) {
				SF_Custom_Sidebars::instance();
			}
			
			if ( SF()->get_setting( 'enable_login_registration' ) ) {
				SF_Login_Registration::instance();
			}
		}

		/**
		 * Hooks
		 */
		private function hooks() {
			register_activation_hook( SF_FRAMEWORK_FILE, array( 'Spirit_Framework', 'install' ) );

			add_action( 'plugins_loaded', array( self::$instance, 'load_plugin_textdomain' ) );
			add_action( 'plugins_loaded', array( self::$instance, 'load_elementor_addon' ) );
			add_action( 'wp_enqueue_scripts', array( $this, 'frontend_scripts' ) );
			add_filter( 'kirki_enqueue_google_fonts', 'sf_custom_google_fonts' );
			add_action( 'widgets_init', array( $this, 'register_widgets' ) );
		}

		/**
		 * Load textdomain
		 */
		public function load_plugin_textdomain() {
			load_plugin_textdomain( 'spirit', false, SF_FRAMEWORK_DIR . 'languages' );
		}

		/**
		 * Load elementor addon
		 */
		public function load_elementor_addon() {
			// if ( defined( 'ELEMENTOR_VERSION' ) && version_compare( ELEMENTOR_VERSION, '3.6', '>=' ) ) {
				SF_Elementor_Addon::instance();
			// }
		}

		/**
		 * Front end scripts
		 */
		public function frontend_scripts() {
			$suffix = defined( 'SF_SCRIPT_DEBUG' ) && SF_SCRIPT_DEBUG ? '' : '.min';

			$adobe_fonts_css = SF_Fonts::get_adobe_fonts_css();
			if ( ! empty( $adobe_fonts_css ) ) {
				wp_enqueue_style( 'sf-adobe-fonts', $adobe_fonts_css, [], SF_FRAMEWORK_VERSION );
			}

			wp_register_style( 'fancybox', SF_FRAMEWORK_URI .'assets/lib/fancybox/css/fancybox.min.css' );
        	
        	wp_register_script( 'fancybox', SF_FRAMEWORK_URI .'assets/lib/fancybox/js/jquery.fancybox.min.js', [ 'jquery' ], SF_FRAMEWORK_VERSION, true );
			
			wp_register_script( 'isotope', SF_FRAMEWORK_URI .'assets/lib/isotope/isotope.pkgd.min.js', [ 'jquery' ], SF_FRAMEWORK_VERSION, true );
			
			wp_register_script( 'jquery-throttle-debounce', SF_FRAMEWORK_URI . 'assets/lib/jquery/jquery.throttle-debounce.min.js', [ 'jquery' ], '1.1', true );

			wp_register_script( 'jquery-animate-number', SF_FRAMEWORK_URI .'assets/lib/jquery/jquery.animateNumber.min.js', [ 'jquery' ], '0.0.14', true );

			wp_register_script( 'swiper', SF_FRAMEWORK_URI .'assets/lib/swiper/js/swiper.min.js', [], '8.4.5', true );

			wp_register_script( 'sf-frontend', SF_FRAMEWORK_URI .'assets/js/frontend/frontend' . $suffix . '.js', array( 'jquery', 'jquery-throttle-debounce', 'imagesloaded' ), SF_FRAMEWORK_VERSION, true );

			wp_register_script( 'sf-share', SF_FRAMEWORK_URI .'assets/js/frontend/share' . $suffix . '.js', array( 'jquery' ), SF_FRAMEWORK_VERSION, true );
			
			wp_register_style( 'font-awesome-5-all', SF_FRAMEWORK_URI . 'assets/lib/font-awesome/css/all.min.css', [], '5.15.3' );
			wp_register_style( 'font-awesome-4-shim', SF_FRAMEWORK_URI . 'assets/lib/font-awesome/css/v4-shims.min.css', [], '5.15.3' );

			if ( ! SF()->get_setting( 'disable_fa5' ) ) {
				wp_enqueue_style( 'font-awesome-5-all' );
			}
			
			if ( apply_filters( 'sf_config_load_fa4_shim', false ) ) {
				wp_enqueue_style( 'font-awesome-4-shim' );
			}
		}

		/**
		 * Register widgets
		 */
		public function register_widgets() {
			register_widget( 'SF_Widget_About' );
			register_widget( 'SF_Widget_About_Site' );
			register_widget( 'SF_Widget_Contact_Info' );
			register_widget( 'SF_Widget_Popular_Categories' );
			register_widget( 'SF_Widget_Popular_Posts' );
			register_widget( 'SF_Widget_Social_Icons' );
		}

		/**
		 * Get setting
		 * 
		 * @return mixed
		 */
		public function get_setting( $key, $default = '' ) {
			$return = $default;
			if ( isset( self::$settings[ $key ] ) ) {
				$return = self::$settings[ $key ];
			}
			return apply_filters( "sf_get_setting_{$key}", $return );
		}

		/**
		 * Get default settings
		 *
		 * @return array
		 */
		public static function get_default_settings() {
			// Default email content
			$email_content_new_account = sprintf( __( 'Hi %s,', 'spirit' ), '{username}' ) . "\n\n";
			$email_content_new_account .= sprintf( __( 'Thanks for creating an account on: %s', 'spirit' ), '{site_title}' ) . "\n\n";
			$email_content_new_account .= sprintf( __( 'Username: %s', 'spirit' ), '{username}' ) . "\n\n";
			
			if ( SF()->get_setting( 'new_account_generate_password' ) ) {
				$email_content_new_account .= sprintf( __( 'Password: %s', 'spirit' ), '{password}' ) . "\n\n";
			} else {
				$email_content_new_account .= __( 'Password: [Password entered at registration]', 'spirit' ) . "\n\n";
			}
	
			return [
				'enable_custom_sidebars' => 1,
				'enable_login_registration' => 1,
				'disable_login_modal' => 0,
				'disable_fa5' => 0,
				'login_redirect_page' => '',
				'registration_redirect_page' => '',
				'verification_page' => '',
				'verification_redirect_page' => '',
				'reset_password_page' => '',
				'new_account_verify_email' => 0,
				'verification_auto_login' => 0,
				'password_strength_meter' => 1,
				'login_remember_me' => 0,
				'new_account_default_roles' => '',
				'new_account_generate_username' => 0,
				'new_account_generate_password' => 0,
				'first_last_name' => 0,
				'new_account_agreement_text' => __( 'I agree to [terms] and [privacy_policy]', 'spirit' ),
				'terms_page' => '',
				'privacy_page' => '',
				'recaptcha' => 0,
				'recaptcha_site_key' => '',
				'recaptcha_secret_key' => '',
				'recaptcha_source' => 'google.com',
				'recaptcha_v3_threshold' => '0.5',
				'recaptcha_version' => '2',
				'email_logo' => '',
				'email_type' => 'plain',
				'email_content' => 'template',
				'from_name' => get_bloginfo( 'name' ),
				'from_email' => get_bloginfo( 'admin_email' ),
				'new_account_subject' => '',
				'new_account_heading' => '',
				'new_account_email_content' => $email_content_new_account,
				'reset_password_subject' => '',
				'reset_password_heading' => '',
				'reset_password_email_content' => __( 'Someone requested that the password be reset for the following account:', 'spirit' ) . "\n\n" . sprintf( __( 'Username: %s', 'spirit' ), '{username}' ) . "\n\n" . __( 'If this was a mistake, just ignore this email and nothing will happen.', 'spirit' ) . "\n\n" . __( 'To reset your password, visit the following address:', 'spirit' ) . "\n\n{reset_password_url}",
				'activate_account_subject' => '',
				'activate_account_heading' => '',
				'activate_account_email_content' => __( "Welcome to", "spirit" ) . " {site_title},\n\n" . __( "Please click the link below to confirm your email address.", "spirit" ) . "\n\n{email_confirmation_link}",
                'desc' => sprintf( __( '%s - Site Title' ), '{site_title}' ) . '<br />' . sprintf( __( '%s - User login' ), '{username}' ) . '<br />' . sprintf( __( '%s - Email confirmation link' ), '{email_confirmation_link}' )
			];
		}

		/**
		 * Get the template path.
		 *
		 * @return string
		 */
		public function template_path() {
			return apply_filters( 'sf_template_path', 'spirit/' );
		}

		/**
		 * Get the plugin path.
		 *
		 * @return string
		 */
		public function plugin_path() {
			return SF_FRAMEWORK_DIR;
		}

		/**
		 * Runs on plugin activation
		 *
		 * @return string
		 */
		public static function install() {
			// install default fonts
			SF_Fonts::add_default_fonts();

			if ( defined( 'ELEMENTOR_VERSION' ) ) {
				update_option( 'elementor_experiment-e_swiper_latest', 'default' );
			}
		}
	}
}