<?php
/**
* SF Widget Helper Class
* 
*/

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

class SF_Widget extends WP_Widget {

    /** 
    * Create Widget 
    * 
    * Creates a new widget and sets it's labels, description, fields and options 
    * 
    * @access   public
    * @param    array
    * @return   void
    * @since    1.0
    */

    function create_widget( $args ) {
        // settings some defaults
        $defaults = array(
            'label'        => '',
            'description'  => '',
            'slug'         => '',
            'fields'       => array(),
            'options'      => array(),
        );

        // parse and merge args with defaults
        $args = wp_parse_args( $args, $defaults );

        // extract each arg to its own variable
        extract( $args, EXTR_SKIP );

        // set the widget vars
        $this->slug    = $slug;
        $this->fields  = $fields;

        // check options
        $this->options = array( 'classname' => $this->slug, 'description' => $description );                        
        if ( ! empty( $options ) ) $this->options = array_merge( $this->options, $options );

        // call WP_Widget to create the widget
        parent::__construct( $this->slug, $label, $this->options );

        // Refreshing the widget's cached output with each new post
        add_action( 'save_post',    array( $this, 'flush_widget_cache' ) );
        add_action( 'deleted_post', array( $this, 'flush_widget_cache' ) );
        add_action( 'switch_theme', array( $this, 'flush_widget_cache' ) );
    }


    /** 
    * Form
    * 
    * Creates the settings form. 
    * 
    * @access   private
    * @param    array
    * @return   void
    * @since    1.0     
    */

    function form( $instance ) {
        $this->instance = $instance;
        $form = $this->create_fields();

        echo $form;
    }


    /** 
    * Update Fields
    *  
    * @access   private
    * @param    array
    * @param    array
    * @return   array
    * @since    1.0
    */

    function update( $new_instance, $old_instance ) {
        $instance = $old_instance;
        
        $this->before_update_fields();

        foreach ( $this->fields as $key ) {
            $slug = $key['id'];

            // if ( isset( $key['validate'] ) ) {
            //     if ( false === $this->validate( $key['validate'], $new_instance[$slug] ) )
            //     return $instance;
            // }

            if ( ! isset( $new_instance[$slug] ) ) {
                $new_instance[$slug] = '';
            }

            if ( isset( $key['filter'] ) ) {
                $instance[$slug] = $this->filter( $key['filter'], $new_instance[$slug] );
            } else {
                $instance[$slug] = strip_tags( $new_instance[$slug] );
            }
        }
        
        return $this->after_validate_fields( $instance );
    }
    

    /**
     * Flush widget cache
     */
    
    public function flush_widget_cache() {
        wp_cache_delete( $this->id_base, 'widget' );
    }


    /** 
    * Before Validate Fields
    *
    * Allows to hook code on the update.
    *  
    * @access   public
    * @param    string
    * @return   string
    * @since    1.6
    */

    function before_update_fields() {
        return;
    }


    /** 
    * After Validate Fields
    * 
    * Allows to modify the output after validating the fields.
    *
    * @access   public
    * @param    string
    * @return   string
    * @since    1.6
    */

    function after_validate_fields( $instance = "" ) {
        return $instance;
    }


    /** 
    * Validate 
    *  
    * @access   private
    * @param    string
    * @param    string
    * @return   boolean
    * @since    1.0
    */

    function validate( $rules, $value ) {
        $rules = explode( '|', $rules );

        if ( empty( $rules ) || count( $rules ) < 1 ) {
            return true;
        }

        foreach ( $rules as $rule ) {
            if ( false === $this->do_validation( $rule, $value ) )
            return false;
        }

        return true;
    }


    /** 
    * Filter 
    *  
    * @access   private
    * @param    string
    * @param    string
    * @return   void
    * @since    1.0
    */

    function filter( $filters, $value = '' ) {
        $filters = explode( '|', $filters ); 

        if ( empty( $filters ) || count( $filters ) < 1 ) {
            return $value;
        }

        foreach ( $filters as $filter ) {
            $value = $this->do_filter( $filter, $value );
        }

        return $value;
    }


    /** 
    * Do Validation Rule
    *  
    * @access   private
    * @param    string
    * @param    string
    * @return   boolean
    * @since    1.0
    */

    function do_validation( $rule, $value = '' ) {

        switch ( $rule ) {

            case 'alpha':
                return ctype_alpha( $value );
            break;

            case 'alpha_numeric':
                return ctype_alnum( $value );
            break;

            case 'alpha_dash':
                return preg_match( '/^[a-z0-9-_]+$/', $value );
            break;

            case 'numeric':
                return ctype_digit( $value );
            break;

            case 'integer':
                return ( bool ) preg_match( '/^[\-+]?[0-9]+$/', $value );
            break;

            case 'boolean':
                return is_bool( $value );
            break;

            case 'email':
                return is_email( $value );
            break;

            case 'decimal':
                return ( bool ) preg_match( '/^[\-+]?[0-9]+\.[0-9]+$/', $value );
            break;

            case 'natural':
                return ( bool ) preg_match( '/^[0-9]+$/', $value );
            return;

            case 'natural_not_zero':
                if ( ! preg_match( '/^[0-9]+$/', $value ) ) return false;
                if ( $value == 0 ) return false;
                return true;
            return;

            default:
                if ( method_exists( $this, $rule ) )
                    return $this->$rule( $value );
                else
                    return false;
            break;

        }
    }       


    /** 
    * Do Filter
    *  
    * @access   private
    * @param    string
    * @param    string
    * @return   boolean
    * @since    1.0
    */

    function do_filter( $filter, $value = '' ) {
        switch ( $filter ) {
            case 'strip_tags':
                $value = strip_tags( $value );
            break;

            case 'wp_strip_all_tags':
                $value = wp_strip_all_tags( $value );
            break;

            case 'esc_attr':
                if ( is_array( $value ) ) {
                    $value = array_map( 'esc_attr', $value );
                } else {
                    $value = esc_attr( $value );
                }
            break;

            case 'esc_url':
                $value = esc_url( $value );
            break;

            case 'esc_textarea':
                $value = esc_textarea( $value );
            break;

            case 'natural':
                $value = absint( $value );
            break;

            default:
                if ( method_exists( $this, $filter ) ) {
                    $value = $this->$filter( $value );
                } else {
                    $value = $value;
                }
            break;
        }
        return $value;
    }


    /** 
    * Create Fields 
    * 
    * Creates each field defined. 
    * 
    * @access   private
    * @param    string
    * @return   string
    * @since    1.0
    */

    function create_fields( $out = "" ) {

        $out = $this->before_create_fields( $out );

        if ( ! empty( $this->fields ) ) {
            foreach ( $this->fields as $key ) 
                $out .= $this->create_field( $key );    
        }

        $out = $this->after_create_fields( $out );

        return $out;
    }


    /** 
    * Before Create Fields
    *
    * Allows to modify code before creating the fields.
    *  
    * @access   public
    * @param    string
    * @return   string
    * @since    1.0
    */

    function before_create_fields( $out = "" ) {
        return $out;
    }


    /** 
    * After Create Fields
    * 
    * Allows to modify code after creating the fields.
    *
    * @access   public
    * @param    string
    * @return   string
    * @since    1.0
    */

    function after_create_fields( $out = "" ) {
        return $out;
    }


    /** 
    * Create Fields
    *  
    * @access   private
    * @param    string
    * @param    string
    * @return   string
    * @since    1.0
    */

    function create_field( $key, $out = "" ) {

        /* Set Defaults */
        $key['std'] = isset( $key['std'] ) ? $key['std'] : "";

        $slug = $key['id'];
                
        if ( isset( $this->instance[$slug] ) ) {
            $key['value'] = $this->instance[$slug];
            if ( ! is_array( $this->instance[$slug] ) ) {
                $key['value'] = strip_tags( $this->instance[$slug] );
            }
        } else {
            unset( $key['value'] );
        }

        /* Set field id and name  */
        $key['_id'] = $this->get_field_id( $slug );
        $key['_name'] = $this->get_field_name( $slug );

        /* Set field type */
        if ( ! isset( $key['type'] ) ) {
            $key['type'] = 'text';
        }

        /* Prefix method */
        $field_method = 'create_field_' . str_replace( '-', '_', $key['type'] );

        /* Run method */
        if ( method_exists( $this, $field_method ) ) {
            $div_start = ( 'image' === $key['type'] ) ? '<div class="sf-widget-row sf-media-upload">' : '<div class="sf-widget-row">';
            $div_end = '</div>';
            
            if ( isset( $key['start'] ) && !$key['start'] ) {
                $div_start = '<br>';
            }

            if ( isset( $key['end'] ) && !$key['end'] ) {
                $div_end = '';
            }

            return $div_start . $this->$field_method( $key ) . $div_end;
        }
    }


    /** 
    * Field Text
    *  
    * @access   private
    * @param    array
    * @param    string
    * @return   string
    * @since    1.0
    */
    
    function create_field_text( $key, $out = "" ) {

        $out .= $this->create_field_label( $key['name'], $key['_id'] );

        $out .= '<input type="text" ';

        if ( isset( $key['class'] ) )
            $out .= 'class="' . esc_attr( $key['class'] ) . '" ';

        $value = isset( $key['value'] ) ? $key['value'] : $key['std'];

        $out .= 'id="' . esc_attr( $key['_id'] ) . '" name="' . esc_attr( $key['_name'] ) . '" value="' . esc_attr__( $value ) . '" ';

        if ( isset( $key['size'] ) )
            $out .= 'size="' . esc_attr( $key['size'] ) . '" ';             

        $out .= ' />';
        
        if ( isset( $key['desc'] ) )
            $out .= '<small class="description">' . esc_html( $key['desc'] ) . '</small>';

        return $out;
    }


    /** 
    * Field Textarea
    *  
    * @access   private
    * @param    array
    * @param    string
    * @return   string
    * @since    1.0
    */
    
    function create_field_textarea( $key, $out = "" ) {

        $out .= $this->create_field_label( $key['name'], $key['_id'] );

        $out .= '<textarea ';

        if ( isset( $key['class'] ) )
            $out .= 'class="' . esc_attr( $key['class'] ) . '" ';

        if ( isset( $key['rows'] ) )
            $out .= 'rows="' . esc_attr( $key['rows'] ) . '" '; 

        if ( isset( $key['cols'] ) )
            $out .= 'cols="' . esc_attr( $key['cols'] ) . '" '; 

        $value = isset( $key['value'] ) ? $key['value'] : $key['std'];

        $out .= 'id="'. esc_attr( $key['_id'] ) .'" name="' . esc_attr( $key['_name'] ) . '">' . esc_html( $value );

        $out .= '</textarea>';
        
        if ( isset( $key['desc'] ) )
            $out .= '<small class="description">' . esc_html( $key['desc'] ) . '</small>';

        return $out;
    }


    /** 
    * Field Checkbox
    *  
    * @access   private
    * @param    array
    * @param    string
    * @return   string
    * @since    1.0
    */
    
    function create_field_checkbox( $key, $out = "" ) {

        $out .= ' <input type="checkbox"';

        if ( isset( $key['class'] ) ) {
            $out .= ' class="' . esc_attr( $key['class'] ) . '"';
        }

        $out .= ' id="' . esc_attr( $key['_id'] ) . '" name="' . esc_attr( $key['_name'] ) . '" value="1"';
        
        if ( ( isset( $key['value'] ) && $key['value'] == 1 ) OR ( ! isset( $key['value'] ) && $key['std'] == 1 ) ) {
            $out .= ' checked="checked"';
        }

        $out .= ' /> ';

        $out .= '<label for="' . esc_attr( $key['_id'] ). '">' . esc_html( $key['name'] ) . '</label>';
        
        if ( isset( $key['desc'] ) ) {
            $out .= '<small class="description">' . esc_html( $key['desc'] ) . '</small>';
        }

        return $out;
    }


    /** 
    * Field Select
    *  
    * @access   private
    * @param    array
    * @param    string
    * @return   string
    * @since    1.0
    */
    
    function create_field_select( $key, $out = "" ) {

        $out .= $this->create_field_label( $key['name'], $key['_id'] );

        $out .= '<select id="' . esc_attr( $key['_id'] ) . '" name="' . esc_attr( $key['_name'] ) . '" ';

        if ( isset( $key['class'] ) )
            $out .= 'class="' . esc_attr( $key['class'] ) . '" ';

        $out .= '> ';

        $selected = isset( $key['value'] ) ? $key['value'] : $key['std'];

            foreach ( $key['fields'] as $label => $value ) {

                $out .= '<option value="' . esc_attr( $value ) . '"';

                if ( $selected == $value ) {
                    $out .= ' selected="selected" ';
                }

                $out .= '> ' . esc_html( $label ) . '</option>';

            }

        $out .= ' </select> ';
        
        if ( isset( $key['desc'] ) ) {
            $out .= '<small class="description">' . esc_html( $key['desc'] ) . '</small>';
        }

        return $out;            
    }


    /** 
    * Field Select Categories
    *  
    * @access   private
    * @param    array
    * @param    string
    * @return   string
    * @since    1.0
    */    
      
    function create_field_select_cats( $key, $out = "" ) {

        $out .= $this->create_field_label( $key['name'], $key['_id'] );

        $selected = isset( $key['value'] ) ? $key['value'] : $key['std'];

        $cats = get_categories( 'hide_empty=0&type=post' );
        $cat_options[] = '<option value="">' . esc_html__( 'Select Categories', 'spirit' ) . '</option>';
        $cats_array = [];
        
        if ( ! empty( $selected ) ) {
            $cats_array = $selected;
            if ( ! is_array( $selected ) ) {
                $cats_array = explode( ',', $selected );
            }
        }

        if ( $cats ) {
            foreach ( $cats as $cat ) {
                $select_attr = in_array( $cat->term_id, $cats_array )  ? ' selected="selected"' : '';
                $cat_options[] = '<option value="' . esc_attr( $cat->term_id ) .'"' . esc_html( $select_attr ) . '>' . esc_attr( $cat->cat_name ) . '</option>';
            }
        }

        $out .= '<div class="sf-widget-control">';
        $out .= '<select id="' . esc_attr( $key['_id'] ) . '" name="' . esc_attr( $key['_name'] ) . '[]" class="sf-js-select';

        if ( isset( $key['class'] ) ) {
            $out .= ' ' . esc_attr( $key['class'] );
        }

        $out .= '" multiple="multiple" style="width:100%;">';

        $out .= implode( '', $cat_options );

        $out .= '</select></div>';
        
        if ( isset( $key['desc'] ) )
            $out .= '<small class="description">' . esc_html( $key['desc'] ) . '</small>';

        return $out;            
    }
    
    
    /** 
    * Field Number
    *  
    * @access   private
    * @param    array
    * @param    string
    * @return   string
    * @since    1.0
    */
    
    function create_field_number( $key, $out = "" ) {

        $out .= $this->create_field_label( $key['name'], $key['_id'] );

        $out .= '<input type="number" ';

        if ( isset( $key['class'] ) )
            $out .= 'class="' . esc_attr( $key['class'] ) . '" ';

        $value = isset( $key['value'] ) ? $key['value'] : $key['std'];

        $out .= 'id="' . esc_attr( $key['_id'] ) . '" name="' . esc_attr( $key['_name'] ) . '" value="' . esc_attr__( $value ) . '" ';

        if ( isset( $key['size'] ) )
            $out .= 'size="' . esc_attr( $key['size'] ) . '" ';             

        $out .= ' />';
        
        if ( isset( $key['desc'] ) )
            $out .= '<small class="description">' . esc_html( $key['desc'] ) . '</small>';

        return $out;
    }

    /** 
    * Field Image
    *  
    * @access   private
    * @param    array
    * @param    string
    * @return   string
    * @since    1.0
    */
    
    function create_field_image( $key, $out = "" ) {

        $out .= $this->create_field_label( $key['name'], $key['_id'] );

        $out .= '<input type="text" ';

        if ( isset( $key['class'] ) )
            $out .= 'class="' . esc_attr( $key['class'] ) . '" ';

        $value = isset( $key['value'] ) ? $key['value'] : $key['std'];

        $out .= 'id="' . esc_attr( $key['_id'] ) . '" name="' . esc_attr( $key['_name'] ) . '" value="' . esc_attr__( $value ) . '" ';

        if ( isset( $key['size'] ) )
            $out .= 'size="' . esc_attr( $key['size'] ) . '" ';             

        $out .= ' />';
        $out .= '<div class="sf-media-btns' . ( empty( $value ) ? '' : ' selected' ) . '"><button type="button" class="sf-btn-upload-image button button-primary">' . esc_attr__( 'Upload Image', 'spirit' ) . '</button><button type="button" class="sf-btn-edit-image button button-primary">' . esc_attr__( 'Edit', 'spirit' ) . '</button><button type="button" class="sf-btn-remove-image button button-primary">' . esc_attr__( 'Remove', 'spirit' ) . '</button>
        </div><div class="sf-media-preview">';

        if ( !empty( $value ) ) {
            $out .= '<img src="' . esc_url( $value ) . '" alt="preview">';
        }

        $out .= '</div>';

        return $out;
    }

    /** 
    * Field Icon
    *  
    * @access   private
    * @param    array
    * @param    string
    * @return   string
    * @since    1.0
    */
    
    function create_field_icon( $key, $out = "" ) {

        $out .= $this->create_field_label( $key['name'], $key['_id'] );

        $out .= '<div style="position:relative;">';
        $out .= '<input type="hidden" ';

        if ( isset( $key['class'] ) )
            $out .= 'class="sf-input-iconpicker ' . esc_attr( $key['class'] ) . '" ';

        $value = isset( $key['value'] ) ? $key['value'] : $key['std'];

        $out .= 'id="' . esc_attr( $key['_id'] ) . '" name="' . esc_attr( $key['_name'] ) . '" value="' . esc_attr__( $value ) . '" ';

        if ( isset( $key['size'] ) ) {
            $out .= 'size="' . esc_attr( $key['size'] ) . '" ';
        }

        $out .= ' />';
        $out .= '</div>';

        return $out;
    }

    /** 
    * Field Label
    *  
    * @access   private
    * @param    string
    * @param    string
    * @return   string
    * @since    1.0
    */

    function create_field_label( $name = "", $id = "" ) {
        return '<label for="' . esc_attr( $id ). '" class="sf-widget-label">' . esc_html( $name ) . ':</label>';
    }
}