<?php
/**
 * SF Theme Manager
 */

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.
class SF_Theme_Manager {

	public static function init() {
		add_action( 'wp_ajax_sf_theme_manager', [__CLASS__, 'handle_ajax_requests'] );
	}

	/**
	 * Get theme info
	 *
	 * @return array
	 */
	public static function get_theme() {
		$active_theme = wp_get_theme();

		if ( is_child_theme() ) {
			$parent_theme = wp_get_theme( $active_theme->Template );
			$theme_data   = [
				'name'          => $parent_theme->Name,
				'version'       => $parent_theme->Version,
				'slug'          => $parent_theme->get_stylesheet(),
				'author'        => $parent_theme->Author,
				'author_uri'    => $parent_theme->AuthorURI,
				'is_child'      => true,
				'child_name'    => $active_theme->Name,
				'child_version' => $active_theme->Version,
				'child_slug'    => $active_theme->get_stylesheet(),
			];
		} else {
			$theme_data = [
				'name'       => $active_theme->Name,
				'version'    => $active_theme->Version,
				'slug'       => $active_theme->get_stylesheet(),
				'author'     => $active_theme->Author,
				'author_uri' => $active_theme->AuthorURI,
				'is_child'   => false,
			];
		}

		return $theme_data;
	}

	/**
	 * Get remote theme data
	 *
	 * @return array
	 */
	public static function get_remote_theme_data() {
		$theme_data = get_transient( 'sf_theme_data' );
		if ( false !== $theme_data ) {
			return $theme_data;
		}

		$url = add_query_arg(
			[
				'action' => 'get_theme_data',
				'theme'  => sf_product_registration()->get_product_id(),
			],
			'https://api.themespirit.com/wp-json/themespirit/v1/product'
		);
		$response = wp_remote_get(
			$url,
			[
				'headers' => [
					'User-Agent' => 'Spirit Framework ' . SF_FRAMEWORK_VERSION,
				],
				'timeout' => 15,
			]
		);

		if ( ! is_wp_error( $response ) ) {
			$response_code = wp_remote_retrieve_response_code( $response );

			if ( 200 === $response_code ) {
				$theme_data = json_decode( wp_remote_retrieve_body( $response ), true );

				if ( empty( $theme_data['error'] ) ) {
					set_transient( 'sf_theme_data', $theme_data, 12 * HOUR_IN_SECONDS );
				}
			}
		}

		if ( ! isset( $theme_data['product_url'] ) ) {
			$theme_data['product_url'] = '#';
		}

		return $theme_data;
	}

	public static function get_child_theme() {
		$active_theme = wp_get_theme();
		if ( is_child_theme() ) {
			return $active_theme;
		}
		$child_theme      = false;
		$installed_themes = wp_get_themes();
		$child_theme_dir  = $active_theme->get_template() . '-child';
		foreach ( $installed_themes as $key => $theme_obj ) {
			if ( $theme_obj->get_stylesheet() === $child_theme_dir ) {
				$child_theme = $theme_obj;
				break;
			}
		}
		return $child_theme;
	}

	/**
	 * Handle ajax theme update & install
	 *
	 * @return void
	 */
	public static function handle_ajax_requests() {
		if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'sf_theme_manager' ) ) {
			wp_send_json_error( __( 'Token is invalid or expired. Please refresh and try again.', 'spirit' ) );
		}

		if ( ! current_user_can( 'update_themes' ) ) {
			wp_send_json_error( __( 'Sorry, you are not allowed to perform the.', 'spirit' ) );
		}

		$action = ! empty( $_POST['theme_action'] ) ? $_POST['theme_action'] : '';
		$theme  = ! empty( $_POST['theme_slug'] ) ? $_POST['theme_slug'] : '';

		if ( ! class_exists( 'Theme_Upgrader', false ) ) {
			require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		}

		if ( 'update' === $action ) {
			$skin     = new WP_Ajax_Upgrader_Skin();
			$upgrader = new Theme_Upgrader( $skin );
			$result   = $upgrader->upgrade( $theme );

			if ( is_wp_error( $result ) ) {
				wp_send_json_error( $result->get_error_message() );
			} elseif ( is_wp_error( $skin->result ) ) {
				wp_send_json_error( $skin->result->get_error_message() );
			} elseif ( $skin->get_errors()->has_errors() ) {
				wp_send_json_error( $skin->get_error_messages() );
			} elseif ( is_null( $result ) ) {
				global $wp_filesystem;
				$message = __( 'Unable to connect to the filesystem. Please confirm your credentials.', 'spirit' );
				// Pass through the error from WP_Filesystem if one was raised.
				if ( $wp_filesystem instanceof WP_Filesystem_Base && is_wp_error( $wp_filesystem->errors ) && $wp_filesystem->errors->has_errors() ) {
					$message .= ' ' . esc_html( $wp_filesystem->errors->get_error_message() );
				}
				wp_send_json_error( $skin->get_error_messages() );
			}

			if ( $result ) {
				wp_send_json_success( SF_Admin::l10n()['theme_update_success'] );
			}

		} else if ( 'install-child' === $action ) {
			$remote_data = self::get_remote_theme_data();
			$skin        = new Automatic_Upgrader_Skin();
			$upgrader    = new Theme_Upgrader( $skin, ['clear_destination' => true] );
			$url         = isset( $remote_data['child'] ) ? $remote_data['child'] : '';

			if ( empty( $url ) ) {
				wp_send_json_error( __( 'Theme package not available', 'spirit' ) );
			}

			$result = $upgrader->install( $url );

			if ( null === $result && ! empty( $skin->result ) ) {
				$result = $skin->result;
			}

			if ( is_wp_error( $result ) ) {
				wp_send_json_error( SF_Admin::l10n()['theme_install_fail'] );
			}

			if ( $result ) {
				$child_theme = self::get_child_theme();
				if ( false !== $child_theme ) {
					switch_theme( $child_theme->get_stylesheet() );
					self::migrate_theme_options( $child_theme->get_template(), $child_theme->get_stylesheet() );
				}
				wp_send_json_success( SF_Admin::l10n()['child_theme_installed'] );
			}
		}
	}

	/**
	 * Migrate theme options
	 *
	 * @param string $from_theme from theme slug
	 * @param string $to_theme to theme slug
	 * @return bool
	 */
	private static function migrate_theme_options( $from_theme, $to_theme ) {
		$from_theme_option = get_option( "theme_mods_{$from_theme}" );
		return update_option( "theme_mods_{$to_theme}", $from_theme_option );
	}
}

SF_Theme_Manager::init();