<?php

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

if ( ! class_exists( 'SF_Product_Updater' ) ) {
	/**
	 * SF Product Updater
	 *
	 * @class ç
	 * @version 1.0.0
	 */
	class SF_Product_Updater {

		/**
		 * Token input field name
		 *
		 * @access private
		 * @since 1.0.0
		 * @var string
		 */
		private static $updates_url = 'https://api.themespirit.com';

		/**
		 *
		 * @param string $slug
		 * @param string $purchase_code
		 * @param string $access_token
		 * @param boolean $debug
		 */
		function __construct() {
			// Check for theme & plugin updates.
			add_filter( 'http_request_args', [$this, 'update_check'], 5, 2 );
			// Inject theme updates into the response array.
			add_filter( 'pre_set_site_transient_update_themes', [$this, 'update_themes'] );
			add_filter( 'pre_set_transient_update_themes', [$this, 'update_themes'] );
			// Inject plugin updates into the response array.
			add_filter( 'pre_set_site_transient_update_plugins', [$this, 'update_plugins'] );
			add_filter( 'pre_set_transient_update_plugins', [$this, 'update_plugins'] );
			// Inject plugin information into the API calls.
			add_filter( 'plugins_api', [$this, 'plugins_api'], 10, 3 );
			// Deferred download.
			add_action( 'upgrader_package_options', [$this, 'deferred_download'], 9 );
			// Requests to update server args.
			add_filter( 'http_request_args', [$this, 'request_headers'], 10, 2 );
			// add counter notification
			add_action( 'sf_admin_menu_title', [ $this, 'add_update_count_notification_to_menu_title' ], 10, 2 );
		}

		/**
		 * Disables requests to the wp.org repository for premium themes.
		 *
		 * @since 1.0.0
		 *
		 * @param array  $request An array of HTTP request arguments.
		 * @param string $url The request URL.
		 * @return array
		 */
		public function update_check( $request, $url ) {
			// Theme update request.
			if ( false !== strpos( $url, '//api.wordpress.org/themes/update-check/1.1/' ) ) {
				/**
				 * Excluded theme slugs that should never ping the WordPress API.
				 * We don't need the extra http requests for themes we know are premium.
				 */
				$slug = get_template();

				// Decode JSON so we can manipulate the array.
				$data = json_decode( $request['body']['themes'] );

				// Remove the excluded themes.
				unset( $data->themes->$slug );

				// Encode back into JSON and update the response.
				$request['body']['themes'] = wp_json_encode( $data );
			}

			// Plugin update request
			if ( false !== strpos( $url, '//api.wordpress.org/plugins/update-check/1.1/' ) ) {
				$data    = json_decode( $request['body']['plugins'] );
				$plugins = sf_plugins_manager()->get_external_plugins();

				foreach ( $plugins as $plugin => $plugin_data ) {
					if ( isset( $data->plugins->$plugin ) ) {
						unset( $data->plugins->$plugin );
					}
				}
				$request['body']['plugins'] = wp_json_encode( $data );
			}

			return $request;
		}

		/**
		 * Add referrer to headers.
		 *
		 * @param array  $parsed_args Parsed request args.
		 * @param string $url         Request URL.
		 * @return array
		 */
		public function request_headers( $parsed_args = [], $url = '' ) {
			// If its not requesting the updates server.
			if ( false === strpos( $url, self::$updates_url ) ) {
				return $parsed_args;
			}

			if ( ! isset( $parsed_args['headers'] ) || ! is_array( $parsed_args['headers'] ) ) {
				$parsed_args['headers'] = [];
			}

			$parsed_args['headers']['Referer'] = site_url();

			return $parsed_args;
		}

		/**
		 * Update themes
		 *
		 * @param object $transient The pre-saved value of the `update_themes` site transient.
		 * @return object
		 */
		public function update_themes( $transient ) {
			// if ( empty( $transient->checked ) || ! sf_product_registration()->is_registered() ) {
			if ( empty( $transient->checked ) ) {
				return $transient;
			}

			$current_theme_name    = '';
			$current_theme_version = '';
			$theme                 = wp_get_theme();
			$slug                  = $theme->get_template();

			if ( $theme->parent() ) {
				$current_theme_name    = $theme->parent()->get( 'Name' );
				$current_theme_version = $theme->parent()->get( 'Version' );
			} else {
				$current_theme_name    = $theme->get( 'Name' );
				$current_theme_version = $theme->get( 'Version' );
			}

			$changelog_url = get_template_directory() . '/changelog.txt';
			
			$theme_data = SF_Theme_Manager::get_remote_theme_data();

			if ( isset( $theme_data['name'] ) && $current_theme_name == $theme_data['name'] ) {
				if ( version_compare( $current_theme_version, $theme_data['version'], '<' ) ) {
					$transient->response[$slug] = [
						'theme'       => $slug,
						'new_version' => $theme_data['version'],
						'url'         => $changelog_url,
						'package'     => $this->get_deferred_download( strtolower( $theme_data['name'] ) ),
					];
				}
			}

			return $transient;
		}

		/**
		 * Inject update data for premium plugins.
		 *
		 * @param object $transient The pre-saved value of the `update_plugins` site transient.
		 * @return object
		 */
		public function update_plugins( $transient ) {
			if ( ! sf_product_registration()->is_registered() ) {
				return $transient;
			}

			// Fetch plugins data
			$plugins_data      = $this->get_plugins_data();
			$installed_plugins = sf_plugins_manager()->get_installed_plugins();

			foreach ( $plugins_data as $plugin_data ) {
				if ( isset( $plugin_data['slug'] ) ) {
					$plugin_slug = $plugin_data['slug'];
					// skip not installed plugins
					$plugin_file = sf_plugins_manager()->get_plugin_file( $plugin_slug );
					if ( $plugin_file === $plugin_slug ) {
						continue;
					}
					// compared versions
					if ( version_compare( $installed_plugins[$plugin_file]['Version'], $plugin_data['version'], '<' ) ) {
						$_plugin = [
							'slug'        => $plugin_slug,
							'plugin'      => $plugin_file,
							'new_version' => $plugin_data['version'],
							'url'         => $plugin_data['homepage'],
							'package'     => $this->get_deferred_download( $plugin_slug ),
						];
						$transient->response[$plugin_file] = (object) $_plugin;
					}
				}
			}
			return $transient;
		}

		/**
		 * Inject API data for premium plugins.
		 *
		 * @since 1.0.0
		 *
		 * @param bool   $response Always false.
		 * @param string $action The API action being performed.
		 * @param object $args Plugin arguments.
		 * @return bool|object $response The plugin info or false.
		 */
		public function plugins_api( $response, $action, $args ) {
			// Process premium theme updates.
			if ( 'plugin_information' === $action && isset( $args->slug ) ) {
				$plugins = $this->get_plugins_data();

				foreach ( $plugins as $plugin ) {
					if ( isset( $plugin['slug'] ) && $args->slug === $plugin['slug'] ) {
						$response                 = new stdClass();
						$response->slug           = $args->slug;
						$response->plugin         = $plugin['slug'];
						$response->plugin_name    = $plugin['name'];
						$response->name           = $plugin['name'];
						$response->version        = $plugin['version'];
						$response->author         = $plugin['author'];
						$response->homepage       = $plugin['homepage'];
						$response->requires       = $plugin['requires'];
						$response->tested         = $plugin['tested'];
						$response->downloaded     = $plugin['downloaded'];
						$response->last_updated   = ! empty( $plugin['last_updated'] ) ? $plugin['last_updated'] : '';
						$response->sections       = ['description' => $plugin['short_description']];
						$response->banners['low'] = ! empty( $plugin['banners']['low'] ) ? $plugin['banners']['low'] : '';
						$response->rating         = ! empty( $plugin['rating'] ) && ! empty( $plugin['rating']['rating'] ) && $plugin['rating']['rating'] > 0 ? $plugin['rating']['rating'] / 5 * 100 : 0;
						$response->num_ratings    = ! empty( $plugin['rating'] ) && ! empty( $plugin['rating']['count'] ) ? $plugin['rating']['count'] : 0;
						$response->download_link  = $this->get_deferred_download( $plugin['slug'] );
						break;
					}
				}
			}

			return $response;
		}

		/**
		 * Get remote plugins data
		 *
		 * @return array
		 */
		public function get_plugins_data() {
			$plugins_data = get_transient( 'sf_plugins_data' );
			if ( false !== $plugins_data ) {
				return $plugins_data;
			}

			$external_plugins = sf_plugins_manager()->get_external_plugins();
			if ( empty( $external_plugins ) ) {
				return [];
			}

			$plugins_data     = [];
			$external_plugins = array_keys( $external_plugins );
			$url              = add_query_arg(
				[
					'action' => 'get_plugin_data',
					'plugin' => $external_plugins,
				],
				self::$updates_url . '/wp-json/themespirit/v1/product'
			);
			$response = wp_remote_get(
				$url,
				[
					'headers' => [
						'User-Agent' => 'Spirit Framework ' . SF_FRAMEWORK_VERSION,
					],
					'timeout' => 15,
				]
			);

			if ( ! is_wp_error( $response ) ) {
				$response_code = wp_remote_retrieve_response_code( $response );

				if ( 200 === $response_code ) {
					$plugins_data = json_decode( wp_remote_retrieve_body( $response ), true );

					if ( empty( $plugins_data['error'] ) ) {
						set_transient( 'sf_plugins_data', $plugins_data, 12 * HOUR_IN_SECONDS );
					}
				}
			}

			return $plugins_data;
		}

		/**
		 * Get deferred download url
		 *
		 * @param string $slug plugin slug
		 * @return string
		 */
		public function get_deferred_download( $slug ) {
			$args = [
				'deferred_download' => true,
				'sf_item'         => $slug,
			];
			return add_query_arg( $args, admin_url() );
		}

		/**
		 * Filters the package options before running an update.
		 *
		 * @since 1.0.0
		 *
		 * @param array $options {
		 *     Options used by the upgrader.
		 *
		 *     @type string $package                     Package for update.
		 *     @type string $destination                 Update location.
		 *     @type bool   $clear_destination           Clear the destination resource.
		 *     @type bool   $clear_working               Clear the working resource.
		 *     @type bool   $abort_if_destination_exists Abort if the Destination directory exists.
		 *     @type bool   $is_multi                    Whether the upgrader is running multiple times.
		 *     @type array  $hook_extra {
		 *         Extra hook arguments.
		 *
		 *         @type string $action               Type of action. Default 'update'.
		 *         @type string $type                 Type of update process. Accepts 'plugin', 'theme', or 'core'.
		 *         @type bool   $bulk                 Whether the update process is a bulk update. Default true.
		 *         @type string $plugin               Path to the plugin file relative to the plugins directory.
		 *         @type string $theme                The stylesheet or template name of the theme.
		 *         @type string $language_update_type The language pack update type. Accepts 'plugin', 'theme',
		 *                                            or 'core'.
		 *         @type object $language_update      The language pack update offer.
		 *     }
		 */
		public function deferred_download( $options ) {
			$package = $options['package'];
			if ( false !== strrpos( $package, 'deferred_download' ) && false !== strrpos( $package, 'sf_item' )) {
				parse_str( wp_parse_url( $package, PHP_URL_QUERY ), $args );
				if ( ! empty( $args['sf_item'] ) ) {
					$options['package'] = sf_plugins_manager()->get_download_link( $args['sf_item'] );
				}
			}
			return $options;
		}

		/**
		 * Add update counter to menu title
		 *
		 * @param string $menu_slug
		 * @param string $menu_title
		 * @return string Menu Title
		 */
		public function add_update_count_notification_to_menu_title( $menu_title, $menu_slug ) {

			if ( 'plugins' === $menu_slug ) {
				$counter = 0;
				$installed_plugins = sf_plugins_manager()->get_installed_plugins();
				$plugins_data  = $this->get_plugins_data();

				foreach ( $plugins_data as $plugin_data ) {
					if ( isset( $plugin_data['slug'] ) ) {
						$plugin_slug = $plugin_data['slug'];
						// skip not installed plugins
						$plugin_file = sf_plugins_manager()->get_plugin_file( $plugin_slug );
						if ( $plugin_file === $plugin_slug ) {
							continue;
						}
						// compared versions
						if ( version_compare( $installed_plugins[ $plugin_file ]['Version'], $plugin_data['version'], '<' ) ) {
							$counter ++;
						}
					}
				}

				$menu_title .= ' <span class="sf-update-products-count update-plugins count-' . $counter . '"><span class="plugin-count">' . $counter . '</span></span>';
			}
			return $menu_title;
		}
	}
}

new SF_Product_Updater();