<?php 

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

if ( ! class_exists( 'SF_Product_Registration' ) ) {
	/**
	 * SF Product Registration
	 *
	 * @class SF_Product_Registration
	 * @version 1.2.0
	 */
	class SF_Product_Registration {

		/**
		 * The single class instance.
		 *
		 * @since 1.0.0
		 * @access private
		 *
		 * @var object
		 */
		private static $_instance = null;

		/**
		 * The setting option name.
		 *
		 * @access private
		 * @var string
		 */
		private $option_name = 'sf_product_data';

		/**
		 * The Envato product ID
		 *
		 * @access private
		 * @since 1.0.0
		 * @var string
		 */
		private $product_id = '';

		/**
		 * Envato API response as WP_Error object.
		 *
		 * @access private
		 * @since 1.0.0
		 * @var null|object WP_Error.
		 */
		private $error_log = null;

		/**
		 * Token input field name
		 *
		 * @access private
		 * @since 1.0.0
		 * @var string
		 */
		public $code_field_name = 'sf_input_code';

		/**
		 * Main SF_Product_Registration Instance
		 *
		 * Ensures only one instance of this class exists in memory at any one time.
		 *
		 * @see SF_Product_Registration()
		 * @uses SF_Product_Registration::init_globals() Setup class globals.
		 * @uses SF_Product_Registration::init_includes() Include required files.
		 *
		 * @since 1.0.0
		 * @static
		 * @return SF_Product_Registration.
		 */
		public static function instance() {
			if ( is_null( self::$_instance ) ) {
				self::$_instance = new self();
			}
			return self::$_instance;
		}

		/**
		 * Constructor
		 */
		function __construct() {
			// add_action( 'wp_ajax_sf_product_registration', [ $this, 'handle_product_registration' ] );
		}

		/**
		 * You cannot clone this class.
		 *
		 * @since 1.0.0
		 */
		public function __clone() {
			_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'spirit' ), '1.0.0' );
		}

		/**
		 * You cannot unserialize instances of this class.
		 *
		 * @since 1.0.0
		 */
		public function __wakeup() {
			_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'spirit' ), '1.0.0' );
		}

		/**
		 * Get messages
		 */
		public function get_message( $key ) {
			$message = '';

			switch( $key ) {
				case 'registered':
					$message = esc_html__( 'Congratulations! Your product is registered now.', 'spirit' );
					break;
				case 'unregistered':
					$message = esc_html__( 'Please enter your theme purchase code and get access to our prebuilt websites, auto-updates, and premium plugins. The purchase code and site URL will be sent to a ThemeSpirit server located in the U.S. to verify the purchase.', 'spirit' );
					break;
				case 'invalid_code':
					$message = esc_html__( 'Invalid purchase code', 'spirit' );
					break;
			}

			return $message;
		}

		/**
		 * Set product registration data.
		 *
		 * @access public
		 * @return void
		 */
		public function set_product_data( $product_data = array() ) {
			$product_id = $this->get_product_id();

			if ( empty( $product_id ) ) {
				return;
			}

			$option_value = get_option( $this->option_name, [] );

			if ( isset( $option_value[ $product_id ] ) || is_array( $option_value ) ) {
				$option_value[ $product_id ] = $product_data;
			} else {
				$option_value = [ $product_id => $product_data ];
			}

			update_option( $this->option_name, $option_value );
		}

		/**
		 * Set product registration data.
		 *
		 * @access public
		 * @return void
		 */
		public function set_error_log( $error = '' ) {
			$this->error_log = $error;
		}

		/**
		 * Check if product is registered
		 *
		 * @param string $product_id
		 * @return boolean
		 */
		public function is_registered() {
			return ( 'registered' == $this->get_option( 'status' ) ) ? true : false;
		}

		/**
		 * Returns the stored purchase code for the product.
		 *
		 * @access public
		 * @param string $product_id The product ID
		 * @return string purchase code.
		 */
		public function get_option( $key = '' ) {
			$data = get_option( $this->option_name, [] );
			$product_id = $this->get_product_id();
			return isset( $data[ $product_id ][ $key ] ) ? $data[ $product_id ][ $key ] : '';
		}

		/**
		 * Returns the product id.
		 *
		 * @access public
		 * @return string $product_id The product ID
		 */	
		public function get_product_id( ) {
			if ( empty( $this->product_id ) ) {
				$this->product_id = apply_filters( 'sf_envato_item_id', '' );
			}
			return $this->product_id;
		}

		/**
		 * Return the page URL.
		 *
		 * @since 1.0.0
		 *
		 * @return string
		 */
		public function get_page_url() {
			return $this->page_url;
		}

		/**
		 * Return masked code
		 */
		public function get_masked_code( $code = '' ) {
			if ( ! empty( $code ) ) {
				$code = substr( $code, 0, 16 );
				$code .= '****************';
			}
			return $code;
		}

		/**
		 * Print product registration form
		 *
		 * @return void
		 */
		public function form() {
			$is_product_registered = $this->is_registered();
			?>
			<form id="sf-form-registration" class="<?php echo esc_attr( $is_product_registered ? 'is-registered' : '' ); ?>" method="post">
				<div class="sf-top-messages">
					<?php
					if ( $is_product_registered ) {
						echo $this->get_message( 'registered' );
					} else {
						echo $this->get_message( 'unregistered' );
					}
					?>
				</div>
				<div class="sf-form-main">
					<div class="sf-input-container">
						<input type="text" name="<?php echo esc_attr( $this->code_field_name ); ?>" class="sf-input-code" value="<?php echo esc_attr( $this->get_masked_code( $this->get_option( 'code' ) ) ); ?>"<?php echo esc_attr( $is_product_registered ? ' disabled' : '' ); ?> />
						<span class="dashicons dashicons-yes"></span>
						<span class="dashicons dashicons-admin-network"></span>
						<input type="hidden" name="sf_registration_action" value="<?php echo esc_attr(  $is_product_registered ? 'unregister' : 'register' ); ?>" />
						<span class="spinner"></span>
					</div>
					<div class="sf-buttons-wrapper">
						<input type="submit" name="submit" class="button button-primary button-large sf-button-register" value="<?php esc_html_e( 'Register', 'spirit' ); ?>">
						<input type="submit" name="submit" class="button button-primary button-large sf-button-unregister" value="<?php esc_html_e( 'Unregister', 'spirit' ); ?>">
						<?php wp_nonce_field( 'sf_product_reg', 'sf_product_reg_nonce' ); ?>
					</div>
				</div>
				<div class="sf-form-messages"></div>
				<?php if ( ! $is_product_registered ) : ?>
					<div class="sf-bottom-messages">
						<a href="https://help.market.envato.com/hc/en-us/articles/202822600-Where-Is-My-Purchase-Code-" rel="nofollow" target="_blank"><?php esc_html_e( 'Where Is My Purchase Code?', 'spirit' ); ?></a>
					</div>
				<?php endif; ?>
			</form><?php
		}

		/**
		 * Handle AJAX registration
		 */
		public function handle_product_registration() {
			if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'sf_product_reg' ) ) {
				wp_send_json_error( [ 'message' => '<span class="error">' . __( 'Token is invalid or expired. Please refresh and try again.', 'spirit' ) . '</span>' ] );
			}

			$code   = isset( $_POST['code'] ) ? sanitize_text_field( $_POST['code'] ) : '';
			$action = isset( $_POST['_action'] ) ? $_POST['_action'] : '';

			if ( 'unregister' === $action ) {
				$code = $this->get_option( 'code' );
			} 

			if ( empty( $code ) ) {
				if ( 'unregister' === $action ) {
					$this->set_product_data( [
						'code'   => '',
						'status' => '',
					] );
					wp_send_json_success( [
						'message' => $this->get_message( 'unregistered' )
					] );
				} else if ( 'register' === $action ) {
					wp_send_json_error( [
						'message' => '<span class="error">' . $this->get_message( 'invalid_code' ) . '</span>'
					] );
				}
			} else if ( 36 !== strlen( $code ) ) {
				wp_send_json_error( [
					'message' => '<span class="error">' . $this->get_message( 'invalid_code' ) . '</span>'
				] );
			}

			$url      = 'https://api.themespirit.com/wp-json/themespirit/v1/buyer';
			$url      = add_query_arg( ['action' => $action], $url );
			$response = wp_remote_get( $url, [
				'headers' => [
					'User-Agent'    => 'Spirit Framework ' . SF_FRAMEWORK_VERSION,
					'Authorization' => 'Bearer ' . $code,
				],
				'timeout' => 15,
			] );

			if ( ! is_wp_error( $response ) ) {
				$response_code = wp_remote_retrieve_response_code( $response );

				if ( 200 === $response_code ) {
					$body = json_decode( wp_remote_retrieve_body( $response ), true );

					if ( ! empty( $body['error'] ) ) {
						wp_send_json_error( [
							'message' => '<span class="error">' . $body['error'] . '</span>'
						] );
					}

					if ( true === $body ) {
						if ( 'register' === $action ) {
							$this->set_product_data( [
								'code'   => $code,
								'status' => 'registered',
							] );
							$message = $this->get_message( 'registered' );
						} else if ( 'unregister' === $action ) {
							$this->set_product_data( [
								'code'   => '',
								'status' => '',
							] );
							$message = $this->get_message( 'unregistered' );
						}
						wp_send_json_success( [
							'code'    => $this->get_masked_code( $code ),
							'message' => $message,
						] );
					}
				} else {
					wp_send_json_error( [
						'message' => '<span class="error">' . wp_remote_retrieve_response_message( $response ) . '</span>'
					] );
				}
			}

			exit;
		}

	}
}

if ( ! function_exists( 'sf_product_registration' ) ) {
	/**
	 * The main function responsible for returning the one true
	 * SF_Product_Registration Instance to functions everywhere.
	 *
	 * Use this function like you would a global variable, except
	 * without needing to declare the global.
	 *
	 * Example: <?php $sf_product = sf_product_registration(); ?>
	 *
	 * @since 1.0.0
	 * @return SF_Product_Registration The one true SF_Product_Registration Instance
	 */
	function sf_product_registration() {
		return SF_Product_Registration::instance();
	}
}