<?php
/**
 * SF Plugins Manager
 *
 * @class SF_Plugins_Manager
 * @version 1.0.0
 */
class SF_Plugins_Manager {

	/**
	 * Static property to hold our singleton instance
	 */
	protected static $_instance = null;

    /**
     * Holds arrays of plugin details.
     *
     * @var array
     */
    private $plugins = null;

    /**
     * Regular expression to test if a URL is a WP plugin repo URL.
     *
     * @const string Regex.
     *
     */
    const WP_REPO_REGEX = '|^http[s]?://wordpress\.org/(?:extend/)?plugins/|';

    /**
     * Arbitrary regular expression to test if a string starts with a URL.
     *
     * @const string Regex.
     */
    const IS_URL_REGEX = '|^http[s]?://|';

	/**
	 * Main instance
	 * Ensures only one instance of this class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @static
	 * @return SF_Plugins_Manager - Main instance
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

    function __construct() {
		add_action( 'wp_ajax_sf_plugins_manager', [ $this, 'handle_ajax_requests' ] );
	}

    /**
     * Register plugins from the configuration
     *
     * @return void
     */
    public function register_plugins() {
        $plugins = apply_filters( 'sf_config_plugins', [] );
        foreach( $plugins as $plugin ) {
            $this->register( $plugin );
        }
    }

    /**
     * Get all installed plugins
     *
     * @param string $plugin_folder
     * @return array
     */
	public function get_installed_plugins( $plugin_folder = '' ) {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		return get_plugins( $plugin_folder );
	}

    /**
     * Get plugin file
     * e.g. sample-plugin/sample-plugin.php
     *
     * @param [type] $slug
     * @return void
     */
	public function get_plugin_file( $slug ) {
        $keys = array_keys( $this->get_installed_plugins() );
        foreach ( $keys as $key ) {
            if ( preg_match( '|^' . $slug . '/|', $key ) ) {
                return $key;
            }
        }
        return $slug;
	}

    /**
     * Get plugins from configuration
     *
     * @return array
     */
    public function get_all_plugins() {
        if ( null === $this->plugins ) {
            $plugins = apply_filters( 'sf_config_plugins', [] );
            foreach( $plugins as $plugin ) {
                $this->register( $plugin );
            }
        }
        return $this->plugins;
    }

    /**
     * Get plugins hosted externally
     *
     * @return array
     */
    public function get_external_plugins() {
        $external_plugins = [];
        $all_plugins = $this->get_all_plugins();
        foreach ( $all_plugins as $key => $data ) {
            if ( isset( $data['source_type'] ) && $data['source_type'] === 'external' ) {
                $external_plugins[ $key ] = $data;
            }
        }
        return $external_plugins;
    }

    /**
     * Get plugin source
     *
     * @param string $slug plugin slug
     * @return void
     */
    public function get_plugin_source( $slug ) {
        if ( ! empty( $this->plugins[ $slug ]['source'] ) ) {
            return $this->plugins[ $slug ]['source'];
        }
        return '';
    }

    /**
     * Determine what type of source the plugin comes from.
     *
     * @param string $source The source of the plugin as provided, either empty (= WP repo), a file path
     *                       (= bundled) or an external URL.
     * @return string 'repo', 'external', or 'bundled'
     */
    public function get_plugin_source_type( $source ) {
        if ( 'repo' === $source || preg_match( self::WP_REPO_REGEX, $source ) ) {
            return 'repo';
        } elseif ( preg_match( self::IS_URL_REGEX, $source ) || 'premium' === $source ) {
            return 'external';
        } else {
            return 'bundled';
        }
    }

	public function get_label( $key ) {
		$label = [
			'active'     => __( 'Active', 'spirit' ),
			'inactive'   => __( 'Inactive', 'spirit' ),
			'activate'   => __( 'Activate', 'spirit' ),
			'deactivate' => __( 'Deactivate', 'spirit' ),
			'install'    => __( 'Install', 'spirit' ),
			'update'     => __( 'Update', 'spirit' ),
		];
		if ( isset( $label[$key] ) ) {
			return $label[$key];
		}
		return '';
	}

    /**
     * Check if there's an update for the plugin
     *
     * @param string $slug
     * @return bool
     */
    public function does_plugin_have_update( $slug ) {
        // Presume bundled and external plugins will point to a package which meets the minimum required version.
        if ( 'repo' !== $this->plugins[ $slug ]['source_type'] ) {
            if ( $this->does_plugin_require_update( $slug ) ) {
                return $this->plugins[ $slug ]['version'];
            }
            return false;
        }

        $repo_updates = get_site_transient( 'update_plugins' );

        if ( isset( $repo_updates->response[ $this->plugins[ $slug ]['file_path'] ]->new_version ) ) {
            return $repo_updates->response[ $this->plugins[ $slug ]['file_path'] ]->new_version;
        }

        return false;
    }

    /**
     * Check whether a plugin complies with the minimum version requirements.
     *
     * @param string $slug Plugin slug.
     * @return bool True when a plugin needs to be updated, otherwise false.
     */
    public function does_plugin_require_update( $slug ) {
        $installed_version = $this->get_installed_version( $slug );
        $minimum_version   = $this->plugins[ $slug ]['version'];

        return version_compare( $minimum_version, $installed_version, '>' );
    }

    /**
     * Retrieve the version number of an installed plugin.
     *
     * @param string $slug Plugin slug.
     * @return string Version number as string or an empty string if the plugin is not installed
     *                or version unknown (plugins which don't comply with the plugin header standard).
     */
    public function get_installed_version( $slug ) {
        $installed_plugins = $this->get_installed_plugins(); // Retrieve a list of all installed plugins (WP cached).

        if ( ! empty( $installed_plugins[ $this->plugins[ $slug ]['file_path'] ]['Version'] ) ) {
            return $installed_plugins[ $this->plugins[ $slug ]['file_path'] ]['Version'];
        }

        return '';
    }

    /**
     * Sanitizes a string key.
     *
     * Near duplicate of WP Core `sanitize_key()`. The difference is that uppercase characters *are*
     * allowed, so as not to break upgrade paths from non-standard bundled plugins using uppercase
     * characters in the plugin directory path/slug. Silly them.
     *
     * @see https://developer.wordpress.org/reference/hooks/sanitize_key/
     *
     * @param string $key String key.
     * @return string Sanitized key
     */
    public function sanitize_key( $key ) {
        $raw_key = $key;
        $key     = preg_replace( '`[^A-Za-z0-9_-]`', '', $key );

        /**
         * Filter a sanitized key string.
         *
         * @param string $key     Sanitized key.
         * @param string $raw_key The key prior to sanitization.
         */
        return apply_filters( 'tgmpa_sanitize_key', $key, $raw_key );
    }

    /**
     * Add individual plugin to our collection of plugins.
     *
     * If the required keys are not set or the plugin has already
     * been registered, the plugin is not added.
     *
     * @param array|null $plugin Array of plugin arguments or null if invalid argument.
     * @return null Return early if incorrect argument.
     */
    public function register( $plugin ) {
        if ( empty( $plugin['slug'] ) || empty( $plugin['name'] ) ) {
            return;
        }

        if ( empty( $plugin['slug'] ) || ! is_string( $plugin['slug'] ) || isset( $this->plugins[ $plugin['slug'] ] ) ) {
            return;
        }

        $defaults = array(
            'name'               => '',      // String
            'slug'               => '',      // String
            'source'             => 'repo',  // String
            'required'           => false,   // Boolean
            'version'            => '',      // String
            'force_activation'   => true,   // Boolean
            'force_deactivation' => false,   // Boolean
            'external_url'       => '',      // String
            'is_callable'        => '',      // String|Array.
        );

        // Prepare the received data.
        $plugin = wp_parse_args( $plugin, $defaults );

        // Standardize the received slug.
        $plugin['slug'] = $this->sanitize_key( $plugin['slug'] );

        // Forgive users for using string versions of booleans or floats for version number.
        $plugin['version']            = (string) $plugin['version'];
        $plugin['source']             = empty( $plugin['source'] ) ? 'repo' : $plugin['source'];
        $plugin['required']           = $plugin['required'];
        $plugin['force_activation']   = $plugin['force_activation'];
        $plugin['force_deactivation'] = $plugin['force_deactivation'];

        // Enrich the received data.
        $plugin['file_path']   = $this->get_plugin_file( $plugin['slug'] );
        $plugin['source_type'] = $this->get_plugin_source_type( $plugin['source'] );

        // Set the class properties.
        $this->plugins[ $plugin['slug'] ]    = $plugin;
        $this->sort_order[ $plugin['slug'] ] = $plugin['name'];

        // Should we add the force activation hook ?
        if ( true === $plugin['force_activation'] ) {
            $this->has_forced_activation = true;
        }

        // Should we add the force deactivation hook ?
        if ( true === $plugin['force_deactivation'] ) {
            $this->has_forced_deactivation = true;
        }
    }

    /**
     * Handle plugin install, activate, deactivate, update
     *
     * @return object
     */
	public function handle_ajax_requests() {
        $action = ! empty( $_POST['plugin_action'] ) ? $_POST['plugin_action'] : '';
        $slug   = ! empty( $_POST['plugin_slug'] ) ? $_POST['plugin_slug'] : '';

        // Actions that require product registration
        if ( 'install' === $action || 'update' === $action ) {
            $premium_plugins = $this->get_external_plugins();
            $premium_plugins = array_keys( $premium_plugins );
    
            if ( in_array( $slug, $premium_plugins ) && ! sf_product_registration()->is_registered() ) {
                wp_send_json_error( [
                    'messages' => sprintf( __( 'This plugin can only be installed or updated, after you have successfully completed the product registration on the <a href="%s" target="_blank">Dashboard</a> tab.', 'spirit' ), admin_url( 'admin.php?page=sf_dashboard#sf-product-registration' ) )
                ] );
            }
    
            if ( empty( $slug ) ) {
                wp_send_json_error( [
                    'messages' => esc_html__( 'Plugin slug empty', 'spirit' )
                ] );
            }
        }

		$plugin = new SF_Plugin( $slug );
        $next_action = 'activate';

		switch ( $action ) {
		case 'install':
			$result = $plugin->install();
            if ( $result ) {
                $result = $plugin->activate();
                if ( $result ) {
                    $next_action = 'deactivate';
                }
            }
			break;

		case 'activate':
			$result      = $plugin->activate();
			$next_action = 'deactivate';
			break;

		case 'deactivate':
			$result = $plugin->deactivate();
			break;

		case 'update':
			$result = $plugin->update();
			if ( 'active' == $plugin->get_status() ) {
				$next_action = 'deactivate';
			} else {
				$next_action = 'activate';
			}
			break;
		}

		if ( $result ) {
            $status = $plugin->get_status();
			wp_send_json_success( [
				'messages' => $plugin->get_messages(),
				'action'   => $next_action,
				'action_label' => $this->get_label( $next_action ),
				'slug'     => $plugin->get_slug(),
				'version'  => $plugin->get_current_version(),
				'info'     => $plugin->get_data(),
				'status'   => $status,
				'status_label'   => $this->get_label( $status ),
			] );
		}

		wp_send_json_error( [
			'messages' => $plugin->get_messages(),
		] );
	}

    /**
     * Get download link
     *
     * @param string $slug
     * @param bool $silent silence all error messages
     * @return string|WP_Error
     */
    public function get_download_link( $slug, $silent = true ) {
        if ( ! sf_product_registration()->is_registered() ) {
            if ( $silent ) {
                return '';
            } else {
                return new WP_Error( 'unregistered', __( 'Product is not registered', 'spirit' ) );
            }
        }

        $url = add_query_arg( [
            'item' => $slug,
        ], 'https://api.themespirit.com/wp-json/themespirit/v1/download/' );

        $response = wp_remote_get(
            $url,
            [
                'headers' => [
                    'Authorization' => 'Bearer ' . sf_product_registration()->get_option( 'code' ),
                    'User-Agent'    => 'Spirit Framework ' . SF_FRAMEWORK_VERSION,
                ],
                'timeout' => 30,
            ]
        );
        
        if ( ! is_wp_error( $response ) ) {
            $response_code = wp_remote_retrieve_response_code( $response );
            $response_body = wp_remote_retrieve_body( $response );

            if ( 200 === $response_code && ! empty( $response_body ) ) {
                return json_decode( $response_body );
            }
        }

        return '';
    }
}

/**
 * Main instance of SF_Plugins_Manager.
 *
 * Returns the main instance of SF_Plugins_Manager to prevent the need to use globals.
 *
 * @return SF_Plugins_Manager
 */
function sf_plugins_manager() {
	return SF_Plugins_Manager::instance();
}

sf_plugins_manager();