<?php

include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
/**
 * SF Plugin
 *
 * @class SF_Plugin
 * @version 1.0.0
 */
class SF_Plugin {

    private $slug = '';
    private $file = '';
    private $path = '';
    private $data = null;
    private $source_type = 'repo';
    private $source = '';
    private $messages = [];
    private $is_required = false;
    private $args = [];

    public function __construct( $plugin ) {
        if ( is_array( $plugin ) ) {
            $slug = isset( $plugin['slug'] ) ? $plugin['slug'] : '';
            $this->args = $plugin;
        } else {
            $slug = $plugin;
            $plugins = sf_plugins_manager()->get_all_plugins();
            if ( isset( $plugins[ $slug ] ) ) {
                $this->args = $plugins[ $slug ];
            }
        }
        $this->slug = $slug;
        $this->set_plugin();
    }

    public function set_plugin() {
        if ( ! empty( $this->slug ) ) {
            $this->file = $this->get_plugin_file( $this->slug );
            $this->path = WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . $this->file;

            if ( ! empty( $this->args['source'] ) ) {
                $this->source_type = sf_plugins_manager()->get_plugin_source_type( $this->args['source'] );
                $this->source = $this->args['source'];
            }

            if ( isset( $this->args['required'] ) && $this->args['required'] ) {
                $this->is_required = true;
            }
        }
    }

	public function get_installed_plugins() {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		return get_plugins();
	}

	public function get_plugin_file( $slug ) {
        $keys = array_keys( $this->get_installed_plugins() );
        foreach ( $keys as $key ) {
            if ( preg_match( '|^' . $slug . '/|', $key ) ) {
                return $key;
            }
        }
        return $slug;
	}

    public function get_name() {
        return ! empty( $this->args['name'] ) ? $this->args['name'] : '';
    }
    
    public function get_status() {
        if ( empty( $this->file ) || empty( $this->path ) || ! file_exists( $this->path ) ) {
            return 'not_installed';
        }

        if ( ! is_plugin_active( $this->file ) ) {
            return 'inactive';
        }

        return 'active';
    }

    public function get_data() {
        if ( ! is_null( $this->data ) ) {
            return $this->data;
        }

        if ( empty( $this->path ) || ! file_exists( $this->path ) ) {
            return false;
        }

        $this->data = get_plugin_data( $this->path );
            
        return $this->data;
    }

    public function get_messages() {
        return $this->messages;
    }

    public function get_slug() {
        return $this->slug;
    }
    
    public function get_current_version() {
        $info = $this->get_data();
        return ! empty( $info['Version'] ) ? $info['Version'] : false;
    }

    public function is_required() {
        return $this->is_required;
    }

    public function get_source_type() {
        return $this->source_type;
    }

    public function get_icon_url() {
        $icon_url = '';
        if ( ! empty( $this->args['icon'] ) ) {
            $icon_url = $this->args['icon'];
        } elseif ( 'repo' === $this->source_type ) {
            $icon_url = 'https://ps.w.org/' . $this->slug . '/assets/icon-128x128.png';
        }
        if ( empty( $icon_url ) ) {
            $icon_url = SF_FRAMEWORK_URI . 'assets/images/plugin-icon.png';
        }
        return $icon_url;
    }

    public function install_wporg() {
        include_once( ABSPATH . 'wp-admin/includes/plugin-install.php' );

        $api = plugins_api(
            'plugin_information',
            array(
                'slug' => $this->slug,
            )
        );

        if ( is_wp_error( $api ) ) {
            return false;
        }

        return $this->wp_install( $api->download_link );
    }

    public function install_local() {
        return $this->wp_install( $this->source );
    }

    public function install() {
        if ( 'not_installed' !== $this->get_status() ) {
            return false;
        }
        
        if ( 'repo' === $this->source_type ) {
            $result = $this->install_wporg();
        } elseif ( 'bundled' === $this->source_type ) {
            $result = $this->wp_install( $this->source );
        } elseif ( 'external' === $this->source_type ) {
            $package = sf_plugins_manager()->get_download_link( $this->slug, false );
            if ( ! is_wp_error( $package ) ) {
                $result = $this->wp_install( $package );
            } else {
                $result = false;
                $this->messages = $package->get_error_messages();
            }
        }

        if ( $result ) {
            $this->set_plugin();
        }

        return $result;
    }

    public function wp_install( $package ) {
        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        include_once ABSPATH . 'wp-admin/includes/plugin-install.php';

        $skin            = new WP_Ajax_Upgrader_Skin();
        $plugin_upgrader = new Plugin_Upgrader( $skin );
        $result          = $plugin_upgrader->install( $package );
        $this->messages  = $skin->get_upgrade_messages();

        if ( is_wp_error( $result ) ) {
            return false;
        }

        return (bool) $result;
    }

    public function update() {
        do_action( 'sf_plugin_before_update' );

        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

        $skin     = new WP_Ajax_Upgrader_Skin();
        $upgrader = new Plugin_Upgrader( $skin );
        $result   = $upgrader->bulk_upgrade( [ $this->file ] );

        $this->messages = $skin->get_upgrade_messages();

        if ( is_wp_error( $result ) ) {
            return false;
        }

        return (bool) $result;
    }

    public function activate() {
        $status = $this->get_status();

        if ( 'not_installed' == $status || 'active' == $status || is_plugin_active_for_network( $this->file ) ) {
            return false;
        }

        $result = activate_plugin( $this->file );

        if ( is_wp_error( $result ) ) {
            return false;
        }

        $recent = (array) get_option( 'recently_activated' );
        unset( $recent[ $this->file ] );
        update_option( 'recently_activated', $recent );

        do_action( 'sf_plugin_activate_plugin', $this->file );

        return true;
    }

    public function deactivate() {
        deactivate_plugins( $this->file );

        update_option( 'recently_activated', array( $this->file => time() ) + (array) get_option( 'recently_activated' ) );

        return true;
    }
}