<?php
/**
 * SF Login
 *
 * Login, Register, Lost Password, Email Verification
 *
 * @package  Spirit_Framework
 */

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

class SF_Login_Registration {
	
	/**
	 * Static property to hold our singleton instance
	 */
	protected static $_instance = null;

	/**
	 * Main instance
	 * Ensures only one instance of this class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 * @static
	 * @return SF_Login_Registration - Main instance
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

    /**
     * Constructor
     */
    function __construct() {
        add_action( 'wp_enqueue_scripts', array( $this, 'frontend_scripts' ) );

        add_action( 'wp_ajax_nopriv_sf_ajax_login', array( $this, 'handle_login' ) );

        add_action( 'wp_ajax_nopriv_sf_ajax_register', array( $this, 'handle_registration' ) );
        add_action( 'wp_ajax_sf_ajax_register', array( $this, 'handle_registration' ) );

        add_action( 'wp_ajax_nopriv_sf_ajax_lost_password', array( $this, 'handle_lost_password' ) );
        add_action( 'wp_ajax_sf_ajax_lost_password', array( $this, 'handle_lost_password' ) );

        add_action( 'wp_ajax_nopriv_sf_ajax_resend_link', array( $this, 'handle_resend_link' ) );
        add_action( 'wp_ajax_sf_ajax_resend_link', array( $this, 'handle_resend_link' ) );
        
        add_filter( 'authenticate', array( $this, 'check_email_verification' ), 100, 2 );
        add_action( 'init', array( $this, 'custom_actions' ) );
        
        add_filter( 'sf_login_redirect_page', array( $this, 'reset_password_redirect_page' ) );
        add_filter( 'sf_elementor_widgets', array( $this, 'add_elementor_widgets' ) );

        if ( ! SF()->get_setting( 'disable_login_modal' ) ) {
            add_action( 'wp_footer', array( $this, 'print_login_modal' ), 100 );
        }

        if ( sf_is_captcha_enabled() ) {
            add_action( 'sf_login_form_end', 'sf_output_form_captcha_widget' );
            add_action( 'sf_register_form_end', 'sf_output_form_captcha_widget' );
            add_action( 'sf_lost_password_form_end', 'sf_output_form_captcha_widget' );
        }

        add_action( 'sf_login_form_start', 'sf_output_preloader' );
        add_action( 'sf_register_form_start', 'sf_output_preloader' );
        add_action( 'sf_lost_password_form_start', 'sf_output_preloader' );

        add_shortcode( 'sf_login_box', array( $this, 'print_login_box' ) );
        add_shortcode( 'sf_login', array( $this, 'print_login_form' ) );
        add_shortcode( 'sf_register', array( $this, 'print_register_form' ) );
        add_shortcode( 'sf_email_verification', array( $this, 'print_email_verification' ) );
        add_shortcode( 'sf_reset_password', array( $this, 'print_reset_password_form' ) );
        add_shortcode( 'sf_lost_password', array( $this, 'print_lost_password_form' ) );
        
        if ( is_admin() ) {
            add_filter( 'manage_users_columns', array( $this, 'add_manage_users_columns' ) );
            add_filter( 'manage_users_custom_column', array( $this, 'add_manage_users_custom_column' ), 10, 3 );
            add_action( 'wp_ajax_sf_ajax_approve_user', array( $this, 'approve_user' ) );
            add_filter( 'bulk_actions-users', array( $this, 'bulk_approve' ) );
            add_filter( 'handle_bulk_actions-users', array( $this, 'bulk_approve_handler' ), 10, 3 );
            add_filter( 'removable_query_args', array( $this, 'bulk_action_removable_query_args' ) );
            add_action( 'admin_notices', array( $this, 'bulk_action_admin_notice' ) );
            add_action( 'profile_update', array( $this, 'profile_update' ) );
        }

        add_action( 'sf_register_form', [ $this, 'custom_registration_fields' ] );
        add_action( 'sf_user_registeration', [ $this, 'save_custom_registration_fields' ], 10, 2 );
    }

    /**
     * Enqueue scripts
     */
    public function frontend_scripts() {
        $suffix = defined( 'SF_SCRIPT_DEBUG' ) && SF_SCRIPT_DEBUG ? '' : '.min';
        
        wp_enqueue_style(
            'sf-login',
            SF_FRAMEWORK_URI .'assets/css/login.min.css',
            false,
            SF_FRAMEWORK_VERSION
        );

        wp_style_add_data( 'sf-login', 'rtl', 'replace' );

        $deps = [ 'jquery' ];

        if ( SF()->get_setting( 'password_strength_meter' ) ) {
            $deps[] = 'password-strength-meter';
        }

        if ( sf_is_captcha_enabled() ) {
            $deps[]  = 'google-recaptcha';
            $source  = SF()->get_setting( 'recaptcha_source' );
            $source  = ! empty( $source ) ? $source : 'google.com';
            $source  = apply_filters( 'sf_recaptcha_source', $source );
            $version = SF()->get_setting( 'recaptcha_version' );
            $version = in_array( $version, [ '2', '3' ] ) ? $version : '2';
            $url     = sprintf( 'https://www.%s/recaptcha/api.js', $source );

            if ( '3' == $version ) {
                $url = add_query_arg(
                    array( 'render' => SF()->get_setting( 'recaptcha_site_key' ) ),
                    $url
                );
            }

            wp_register_script( 'google-recaptcha', $url, [], $version . '.0', true );
        }

        wp_register_script(
            'sf-login',
            SF_FRAMEWORK_URI .'assets/js/frontend/login' . $suffix . '.js',
            $deps,
            SF_FRAMEWORK_VERSION,
            true
        );
        
        wp_enqueue_script( 'sf-login' );

        wp_localize_script(
            'sf-login',
            'sf_login_data',
            array(
                'ajax_url' => admin_url( 'admin-ajax.php' ),
                'L10n'   => array(
                    'required_fields' => __( 'Please fill in all required fields.', 'spirit' ),
                    'empty_captcha'   => __( 'Please verify you\'re a human.', 'spirit' ),
                    'show_password'   => __( 'Show Password', 'spirit' ),
                    'hide_password'   => __( 'Hide Password', 'spirit' ),
                ),
                'i18n_password_not_match' => __( 'Confirm password does not match.', 'spirit' ),
                'i18n_password_hint'  => apply_filters( 'sf_password_hint', wp_get_password_hint() )
            )
        );
    }

    /**
     * Custom column title
     * @param array $columns columns data
     */
    public function add_manage_users_columns( $columns ) {
        return array_merge( $columns, array( 'sf_status' => __( 'User Status', 'spirit' ) ) );
    }
    
    /**
     * Custom column content
     * @param string $output
     * @param string $column_name
     * @param int $user_id
     */
    public function add_manage_users_custom_column( $output, $column_name, $user_id ) {
        ob_start();

        if ( $column_name == 'sf_status' ) {
            $activation_status = get_user_meta( $user_id, 'sf_activation_status' );

            if ( isset( $activation_status[0] ) && count( $activation_status ) == 1 ) :
            ?>
            <div class="sf_status">
                <?php echo ( $activation_status[0] == 1 ? esc_html__( 'Verified', 'spirit' ) : esc_html__( 'Pending Verification', 'spirit' ) ); ?>
            </div>
            <div class="row-actions">
                <?php if ( $activation_status[0] == 0 ) : ?>
                    <span class="sf_action sf_approve" user_id="<?php echo esc_attr( $user_id ); ?>" do="approve"><?php esc_html_e( 'Approve now', 'spirit' ); ?></span>
                <?php elseif ( $activation_status[0] == 1 ) : ?>
                    <span class="sf_action sf_remove_approval" user_id="<?php echo esc_attr( $user_id ); ?>" do="remove_approval"><?php esc_html_e( 'Remove approval', 'spirit' ); ?></span>
                <?php endif; ?>
            </div>
            <?php
            endif;
        }
        
        return ob_get_clean();
    }

    /**
     * Approve user
     *
     * @return void
     */
    public function approve_user() {
        $user_id = isset( $_POST['user_id'] ) ? $_POST['user_id'] : '';
        $action = isset( $_POST['do'] ) ? $_POST['do'] : '';
        
        if ( empty( $user_id ) || empty( $action ) ) {
            wp_die();
        }

        if ( $action == 'approve' ) {
            update_user_meta( $user_id, 'sf_activation_status', 1 );
        }

        if ( $action == 'remove_approval' ) {
            update_user_meta( $user_id, 'sf_activation_status', 0 );
        }
        
        $activation_status = get_user_meta( $user_id, 'sf_activation_status', true );
        $activation_status = empty( $activation_status ) ? 0 : $activation_status;
        $user_staus = $activation_status == 1 ? esc_html__( 'Verified', 'spirit' ) : esc_html__( 'Pending Verification', 'spirit' );
        echo $user_staus;

        wp_die();
    }

    public function bulk_approve( $actions ) {
        $actions['sf_bulk_approve'] = __( 'Approve', 'spirit' );
        $actions['sf_bulk_remove_approval'] = __( 'Remove Approval', 'spirit' );
        return $actions;
    }
    
    /**
     * Bulk approve handler
     *
     * @param string $redirect_to
     * @param string $action
     * @param array $items
     * @return $redirect_to
     */
    public function bulk_approve_handler( $redirect_to, $action, $items ) {
        if ( $action == 'sf_bulk_approve' ) {
            $user_status = 1;
            $redirect_to = add_query_arg( 'sf_bulk_approve', count( $items ), $redirect_to );
        } elseif ( $action =='sf_bulk_remove_approval' ) {
            $user_status = 0;
            $redirect_to = add_query_arg( 'sf_bulk_remove_approval', count( $items ), $redirect_to );
        } else {
            return $redirect_to;
        }

        foreach ( $items as $user_id ) {
            $activation_status = get_user_meta( $user_id, 'sf_activation_status' );
            if ( isset( $activation_status[0] ) ) {
                update_user_meta( $user_id, 'sf_activation_status', $user_status );
            }
        }
        return $redirect_to;
    }
    
    /**
     * Bulk action removable query args
     *
     * @param array $args
     * @return $args
     */
    public function bulk_action_removable_query_args( $args ) {
        $args[] = 'sf_bulk_approve';
        $args[] = 'sf_bulk_remove_approval';
        return $args;
    }

    /**
     * Bulk action admin notice
     *
     * @return void
     */
    public function bulk_action_admin_notice() {
        if ( isset( $_REQUEST['sf_bulk_approve'] ) ) {
            $user_count = intval( $_REQUEST['sf_bulk_approve'] );

            echo '<div id="message" class="notice notice-success is-dismissible">';
            echo '<p>'. sprintf( _n( '%s user account marked as approved.', '%s user accounts marked as approved.', $user_count, 'spirit' ), number_format_i18n( $user_count ) ) . '</p>';
            echo '</div>';
    
        } elseif ( isset( $_REQUEST['sf_bulk_remove_approval'] ) ) {
            $user_count = intval( $_REQUEST['sf_bulk_remove_approval'] );

            echo '<div id="message" class="notice notice-success is-dismissible">';
            echo '<p>'. sprintf( _n( '%s user account marked as unverified.', '%s user accounts marked as unverified.', $user_count, 'spirit' ), number_format_i18n( $user_count ) ) . '</p>';
            echo '</div>';
        }
    }

    /**
     * User profile update
     *
     * @param integer $user_id
     * @return void
     */
    public function profile_update( $user_id ) {
        if ( ! current_user_can( 'edit_users' ) ) {
            return -1;
        }

        if ( ! current_user_can( 'edit_user', $user_id ) ) {
            return -1;
        }

        $default_roles = SF()->get_setting( 'new_account_default_roles' );

        if ( empty( $default_roles ) ) {
            return 1;
        }

        $user            = get_userdata( $user_id );
        $editable_roles  = get_editable_roles();
        
        foreach( $default_roles as $role_id ) {

            if ( empty( $role_id ) || ! isset( $editable_roles[ $role_id ] ) ) {
                continue;
            }

            if ( is_array( $user->roles ) && ! in_array( $role_id, $user->roles ) ) {
                $user->add_role( $role_id );
            }
        }

        return 2;
    }

    /**
     * Process login
     */
    public function handle_login() {
        if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'sf_login_nonce' ) ) {
            wp_send_json_error( array( 'message'=>'<span class="error">' . __( 'Token is invalid or expired. Please refresh and try again.', 'spirit' ) . '</span>' ) );
        }
        
        $username = isset( $_POST['username'] ) ? sanitize_user( $_POST['username'] ) : '';
        $password = isset( $_POST['password'] ) ? sanitize_text_field( $_POST['password'] ) : '';
        $remember = ! empty( $_POST['remember'] ) ? true : false;
        $token    = ! empty( $_POST['token'] ) ? $_POST['token'] : '';
        $messages = sf_get_login_form_messages();
        $error    = '';

        if ( sf_is_captcha_enabled() && ! sf_is_human( $token, 'sf_ajax_login' ) ) {
            wp_send_json_error( array( 'message'=>'<span class="error">' . $messages['captcha_failed'] . '</span>' ) );
        }

        $signon = wp_signon( array(
            'user_login'    => $username,
            'user_password' => $password,
            'remember'      => $remember
        ), is_ssl() );

        if ( is_wp_error( $signon ) ) {
            switch ( $signon->get_error_code() ) {
            case 'invalid_username':
                $error = $messages['invalid_username'];
                break;
            case 'incorrect_password':
                $error = $messages['incorrect_password'];
                break;
            default:
                $error = $signon->get_error_message();
                break;
            }
        } else if ( empty( $_COOKIE[ LOGGED_IN_COOKIE ] ) ) {
            if ( headers_sent() ) {
                /* translators: 1: Browser cookie documentation URL, 2: Support forums URL */
                $error = sprintf( __( '<strong>ERROR</strong>: Cookies are blocked due to unexpected output. For help, please see <a href="%1$s" target="_blank">this documentation</a> or try the <a href="%2$s">support forums</a>.', 'spirit' ),
                    __( 'https://codex.wordpress.org/Cookies', 'spirit' ), __( 'https://wordpress.org/support/', 'spirit' ) );
            }
        }

        if ( ! empty( $error ) ) {
            wp_send_json_error( [ 'message' => '<span class="error">' . $error . '</span>' ] );
        }

        if ( $signon && ! is_wp_error( $signon ) ) {
            // wp_set_current_user( $user->ID, $username );

            if ( empty( $_POST['redirect'] ) ) {
                $redirect_url = get_the_permalink( SF()->get_setting( 'login_redirect_page' ) );
                $redirect_url = apply_filters( 'sf_login_redirect_page', $redirect_url, $signon );
            } else {
                $redirect_url = $_POST['redirect'];
            }
        
            wp_send_json_success( [
                'refresh'      => true,
                'message'      => '<span class="success">' . $messages['success'] . '</span>',
                'redirect_url' => esc_url( $redirect_url ),
            ] );
        }

        exit();
    }

    /**
     * Process account registration
     */
    public function handle_registration() {
        if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'sf_register_nonce' ) ) {
            wp_send_json_error( array( 'message'=>'<span class="error">' . __( 'Token is invalid or expired. Please refresh and try again.', 'spirit' ) . '</span>' ) );
        }

        $form       = isset( $_POST['form'] ) ? $_POST['form'] : '';
        $form_data  = [];
        parse_str( $form, $form_data );
        $username   = isset( $form_data['sf-username'] ) ? sanitize_user( $form_data['sf-username'] ) : '';
        $email      = isset( $form_data['sf-email'] ) ? $form_data['sf-email'] : '';
        $password   = ! empty( $form_data['sf-password'] ) ? $form_data['sf-password'] : '';
        $first_name = isset( $form_data['sf-firstname'] ) ? sanitize_text_field( $form_data['sf-firstname'] ) : '';
        $last_name  = isset( $form_data['sf-lastname'] ) ? sanitize_text_field( $form_data['sf-lastname'] ) : '';
        $token      = ! empty( $_POST['token'] ) ? $_POST['token'] : '';
        $messages   = sf_get_new_account_form_messages();
        $errors     = $error_fields = [];
        $success    = '';
        $signon     = false;

        if ( sf_is_captcha_enabled() && ! sf_is_human( $token, 'sf_ajax_register' ) ) {
            wp_send_json_error( array( 'message'=>'<span class="error">' . $messages['captcha_failed'] . '</span>' ) );
        }

        if ( get_option( 'users_can_register' ) ) {
            $generate_username = SF()->get_setting( 'new_account_generate_username' );
            $generate_password = SF()->get_setting( 'new_account_generate_password' );
            $verify_email      = SF()->get_setting( 'new_account_verify_email' );
            
            // Check username
            if ( ! $generate_username ) {
                if ( username_exists( $username ) ) {
                    $errors[] = $messages['username_exists'];
                    $error_fields[] = '.sf-username';
                } elseif ( !validate_username( $username ) ) {
                    $errors[] = $messages['invalid_username'];
                    $error_fields[] = '.sf-username';
                }
            }

            // Check email
            if ( is_email( $email ) ) {
                if ( email_exists( $email ) ) {
                    $errors[] = $messages['existing_user_email'];
                    $error_fields[] = '.sf-email';
                }
            } else {
                $errors[] = $messages['invalid_email'];
                $error_fields[] = '.sf-email';
            }

            if ( ! empty( $form_data ) && is_array( $form_data ) ) {
                $form_data['username']   = $username;
                $form_data['email']      = $email;
                $form_data['first_name'] = $first_name;
                $form_data['last_name']  = $last_name;
            }

            $errors['error_fields'] = $error_fields;
            $errors = apply_filters( 'sf_user_registration_errors', $errors, $form_data );

            if ( isset( $errors['error_fields'] ) ) {
                $error_fields = $errors['error_fields'];
                unset( $errors['error_fields'] );
            }

            // Create user if no errors
            if ( empty( $errors ) ) {

                // Generate unique username from email address
                if ( $generate_username ) {
                    $username   = sanitize_user( current( explode( '@', $email ) ), true );
                    $o_username = $username;
                    $append     = 1;

                    while ( username_exists( $username ) ) {
                        $username = $o_username . $append;
                        $append ++;
                    }
                }

                // Generate random password
                if ( $generate_password ) {
                    $password = wp_generate_password( 12, false );
                }

                // User data
                $new_user_data = apply_filters(
                    'sf_new_user_data', array(
                        'user_login' => $username,
                        'user_pass'  => $password,
                        'user_email' => $email,
                        'first_name' => $first_name,
                        'last_name'  => $last_name
                    )
                );

                $user_id = wp_insert_user( $new_user_data );

                if ( is_wp_error( $user_id ) ) {
                    $errors[] = $user_id->get_error_message();
                } else {
                    // Assign default roles
                    $default_roles = SF()->get_setting( 'new_account_default_roles' );
                    $default_roles = apply_filters( 'sf_new_account_default_roles', $default_roles );

                    if ( ! empty( $default_roles ) && is_array( $default_roles ) ) {
                        $new_user = new WP_User( $user_id );
                        foreach( $default_roles as $role ) {
                            $new_user->add_role( $role );
                        }
                    }

                    // User verification
                    if ( $verify_email ) {
                        $activation_key = MD5( $email . uniqid( time(), true ) );
                        add_user_meta( $user_id, 'sf_activation_key', $activation_key );
                        add_user_meta( $user_id, 'sf_activation_status', 0 );
                        $activation_link = sf_get_email_verification_link( $user_id, $activation_key );
                    } else {
                        // auto login user
                        if ( ! SF()->get_setting( 'new_account_auto_login' ) ) {
                            $signon = wp_signon( array(
                                'user_login' => $username,
                                'user_password' => $password,
                                'remember' => true
                            ), false );
                        }
                    }

                    do_action( 'sf_user_registeration', $user_id, $form_data );

                    $mail = new SF_Email();
                    
                    if ( ! empty( $activation_link ) ) {
                        if ( $mail->send_activate_account_mail( $user_id, $new_user_data, $activation_link ) ) {
                            $success = $messages['success_verify_email'] . '<div class="text-center"><button type="button" class="btn btn-sm btn-outline-dark sf-resend-link" data-nonce="'. wp_create_nonce( 'sf_resend_link_nonce' ) .'" data-user="'. $username .'">'. __( 'Re-send Verfication Email', 'spirit' ) .'</button></div>';
                        } else {
                            $errors[] = $messages['send_mail_error'];
                        }
                    } else {
                        if ( $mail->send_new_account_mail( $user_id, $new_user_data, $generate_password ) ) {
                            $success = $messages['success'];
                        } else {
                            $errors[] = $messages['send_mail_error'];
                        }
                    }
                }
            }

        } else {
            $errors[] = $messages['no_registration'];
        }

        if ( ! empty( $errors ) ) {
            wp_send_json_error( array(
                'message' => '<span class="error">' . implode( '<br/>', $errors ) . '</span>',
                'fields'  => $error_fields
            ) );
        }
        
        if ( ! empty( $success ) ) {

            if ( empty( $_POST['redirect'] ) ) {
                $redirect_url = get_the_permalink( SF()->get_setting( 'registration_redirect_page' ) );
                $redirect_url = apply_filters( 'sf_registration_redirect_page', $redirect_url );
            } else {
                $redirect_url = $_POST['redirect'];
            }

            wp_send_json_success( array(
                'delayed_refresh' => ( $signon && !is_wp_error( $signon ) ? true : false ),
                'message'         =>'<span class="success">' . $success . '</span>',
                'redirect_url'    => esc_url( $redirect_url )
            ) );
        }

        wp_send_json_error( array( 'message'=>'<span class="error">' . esc_html__( 'Unknown error', 'spirit' ). '</span>' ) );
    }

    /**
     * Process lost password
     */
    public function handle_lost_password() {
        if ( empty( $_POST['nonce'] ) || !wp_verify_nonce( $_POST['nonce'], 'sf_lost_password_nonce' ) ) {
            wp_send_json_error( array( 'message'=>'<span class="error">' . __( 'Token is invalid or expired. Please refresh and try again.', 'spirit' ) . '</span>' ) );
        }
        
        $user_login = isset( $_POST['user_login'] ) ? $_POST['user_login'] : '';
        $token      = ! empty( $_POST['token'] ) ? $_POST['token'] : '';
        $messages   = sf_get_lost_password_form_messages();
        $error      = $success = '';

        // if ( empty( $user_login ) ) {
        //     $error = $messages['username_email_empty'];
        // }

        if ( sf_is_captcha_enabled() && ! sf_is_human( $token, 'sf_ajax_lost_password' ) ) {
            wp_send_json_error( array( 'message'=>'<span class="error">' . $messages['captcha_failed'] . '</span>' ) );
        }

        if ( is_email( $user_login ) && email_exists( $user_login ) ) {
            $user = get_user_by( 'email', $user_login );

        } elseif ( validate_username( $user_login ) && username_exists( $user_login ) ) {
            $user = get_user_by( 'login', $user_login );
        }

        if ( ! empty( $user ) && ! is_wp_error( $user ) ) {

            $errors = new \WP_Error();
            // Fires before errors are returned from a password reset request.
            do_action( 'lostpassword_post', $errors, $user );

            if ( $errors->has_errors() ) {
                wp_send_json_error( array( 'message'=>'<span class="error">' . $errors->get_error_message() . '</span>' ) );
            }
            
            // Fires before a new password is retrieved.
            do_action( 'retrieve_password', $user->user_login );

            $allow = true;
            if ( is_multisite() && is_user_spammy( $user ) ) {
                $allow = false;
            }
            
            // Filters whether to allow a password to be reset.
            $allow = apply_filters( 'allow_password_reset', $allow, $user->ID );

            if ( $allow && ! is_wp_error( $allow ) ) {

                // Generate something random for a password reset key.
                $reset_key = wp_generate_password( 20, false );
                
                // Fires when a password reset key is generated.
                do_action( 'retrieve_password_key', $user->user_login, $reset_key );

                global $wpdb, $wp_hasher;
                // Now insert the key, hashed, into the DB.
                if ( empty( $wp_hasher ) ) {
                    require_once ABSPATH . 'wp-includes/class-phpass.php';
                    $wp_hasher = new PasswordHash( 8, true );
                }
                
                $hashed = time() . ':' . $wp_hasher->HashPassword( $reset_key );
                $wpdb->update( $wpdb->users, array( 'user_activation_key' => $hashed ), array( 'user_login' => $user->user_login ) );

                // Reset page URL
                $reset_pass_url = sf_get_reset_password_link( $reset_key, $user->user_login );

                // Send mail
                $mail = new SF_Email();

                if ( $mail->send_reset_password_mail( $user->ID, $user->user_login, $user->user_email, $reset_pass_url ) ) {
                    $success = $messages['success'];
                } else {
                    $error = $messages['send_mail_error'];
                }
            
            } else {
                $error = $messages['no_password_reset'];
            }
        
        } else {
            $error = $messages['user_not_registered'];
        }  
        
        if ( ! empty( $error ) ) {
            wp_send_json_error( array( 'message'=>'<span class="error">' . $error . '</span>' ) );
        }
        
        if ( ! empty( $success ) ) {
            wp_send_json_success( array( 'message'=>'<span class="success">' . $success . '</span>' ) );
        }

        wp_send_json_error( array( 'message'=>'<span class="error">' . esc_html__( 'Unknown error', 'spirit' ). '</span>' ) );
    }

    /**
     * Email verification
     *
     * @param null|WP_User|WP_Error $user
     * @param string $username
     * @return null|WP_User|WP_Error
     */
    public function check_email_verification( $user, $username ) {
        if ( is_wp_error( $user ) ) {
            return $user;
        }

        $activation_status = get_user_meta( $user->ID, 'sf_activation_status' );

        if ( isset( $activation_status[0] ) && count( $activation_status ) == 1 && $activation_status[0] == 0 ) {
            return new WP_Error( 'email_not_verified', sprintf(
                __( 'You have not verified your email address, please check your email and click on verification link we sent you. <a href="javascript:void(0)" data-user="%s">Re-send the link</a>', 'spirit' ),
                $username
            ) );
        }

        return $user;
    }

    /**
     * Process resend verification email
     */
    public function handle_resend_link() {
        if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'sf_resend_link_nonce' ) ) {
            wp_send_json_error( array( 'message'=>'<span class="error">' . __( 'Token is invalid or expired. Please refresh and try again.', 'spirit' ) . '</span>' ) );
        }

        $user_login = isset( $_POST['user_login'] ) ? $_POST['user_login'] : '';
        $user = get_user_by( 'login', $user_login );
        
        if ( $user ) {
            $activation_status = get_user_meta( $user->ID, 'sf_activation_status' );
            if ( isset( $activation_status[0] ) && $activation_status[0] == 1 ) {
                wp_send_json_error( [ 'message' => __( 'Account already activated.', 'spirit' ) ] );
            } else {
                $activation_key = MD5( $user->user_email . uniqid( time(), true ) );
                update_user_meta( $user->ID, 'sf_activation_key', $activation_key );
                $activation_link = sf_get_email_verification_link( $user->ID, $activation_key );
                $new_user_data = array(
                    'user_login' => $user_login,
                    'user_email' => $user->user_email,
                    'first_name' => $user->first_name,
                    'last_name'  => $user->last_name
                );
            }
        }

        if ( ! empty( $activation_link ) ) {
            $mail = new SF_Email();
            
            if ( $mail->send_activate_account_mail( $user->ID, $new_user_data, $activation_link ) ) {
                wp_send_json_success();
            } else {
                wp_send_json_error();
            }
        }

        wp_die();
    }

    /**
     * Print login modal
     *
     * @param array $atts
     * @return string
     */
    function print_login_modal() {
        $args = array(
            'minimum_password_strength' => '3',
            'login_redirect_url' => '',
            'registration_redirect_url' => ''
        );

        add_action( 'sf_login_form_end', 'sf_output_login_form_bottom_message' );
        add_action( 'sf_register_form_end', 'sf_output_register_form_bottom_message' );

        ?>
        <div class="sf-login-overlay">
            <div class="sf-login-modal">
                <?php sf_get_template( 'account/login-box.php', $args ); ?>
                <button class="sf-modal-close"><span class="sf-icon-close"></span></button>
            </div>
        </div>
        <?php
    }

    /**
     * Print login box
     *
     * @param array $atts
     * @return string
     */
    function print_login_box( $atts ) {
        // Defaults
        $args = shortcode_atts( array(
            'minimum_password_strength' => '3',
            'login_redirect_url' => '',
            'registration_redirect_url' => ''
        ), $atts );

        add_action( 'sf_login_form_end', 'sf_output_login_form_bottom_message' );
        add_action( 'sf_register_form_end', 'sf_output_register_form_bottom_message' );

        return sf_get_template_html( 'account/login-box.php', $args );
    }

    /**
     * Print login form
     *
     * @param array $atts
     * @return string
     */
    function print_login_form( $atts ) {
        // Defaults
        $args = shortcode_atts( array(
            'login_redirect_url' => ''
        ), $atts );

        return sf_get_template_html( 'account/form-login.php', $args );
    }

    /**
     * Print registration form
     *
     * @param array $atts
     * @return string
     */
    function print_register_form( $atts ) {
        // Defaults
        $args = shortcode_atts( array(
            'minimum_password_strength' => '3',
            'registration_redirect_url' => ''
        ), $atts );

        return sf_get_template_html( 'account/form-register.php', $args );
    }

    /**
     * Print lost password form
     *
     * @param array $atts
     * @return string
     */
    function print_lost_password_form( $atts ) {
        return sf_get_template_html( 'account/form-lost-password.php', $atts );
    }

    /**
     * Print reset password form
     *
     * @param array $atts
     * @return string
     */
    function print_reset_password_form( $atts ) {
        // Defaults
        $args = shortcode_atts( array(
            'minimum_password_strength' => '3',
            'login_redirect_url' => get_site_url()
        ), $atts );

        $action = isset( $_REQUEST['action'] ) ? $_REQUEST['action'] : 'login';

        if ( $action == 'sf_resetpass' ) {
            list( $rp_path ) = explode( '?', wp_unslash( $_SERVER['REQUEST_URI'] ) );
            $rp_cookie       = 'wp-resetpass-' . COOKIEHASH;
    
            if ( isset( $_COOKIE[ $rp_cookie ] ) && 0 < strpos( $_COOKIE[ $rp_cookie ], ':' ) ) {
                list( $rp_login, $rp_key ) = explode( ':', wp_unslash( $_COOKIE[ $rp_cookie ] ), 2 );

                $args['form_message'] = '<span class="success">' . esc_html__( 'Enter new password', 'spirit' ) . '</span>';
                $args['rp_key']       = $rp_key;
            }

            return sf_get_template_html( 'account/form-reset-password.php', $args );

        } else if ( $action == 'sf_lostpass' ) {
            
            if ( isset( $_REQUEST['error'] ) ) {
                if ( 'expiredkey' == $_REQUEST['error'] ) {
                    $args['form_message'] = '<span class="error">' . esc_html__( 'Your password reset link has expired. Please request a new link below.', 'spirit' ) . '</span>';
                } else {
                    $args['form_message'] = '<span class="error">' . esc_html__( 'Your password reset link appears to be invalid. Please request a new link below.', 'spirit' ) . '</span>';
                }
            }

            return sf_get_template_html( 'account/form-lost-password.php', $args );

        } else if ( $action == 'sf_login' ) {

            if ( isset( $_REQUEST['status'] ) && 'success' == $_REQUEST['status'] ) {
                $args['form_message'] = '<span class="success">' . esc_html__( 'Your password has been reset.', 'spirit' ) . '</span>';
            }
            return sf_get_template_html( 'account/form-login.php', $args );
        }

        if ( is_user_logged_in() ) {
            return sf_get_template_html( 'account/form-lost-password.php', $args );
        } else {
            return sf_get_template_html( 'account/form-login.php', $args );
        }
    }

    /**
     * Print email verification status
     *
     * @param array $atts
     * @return string
     */
    function print_email_verification( $atts ) {
        $html = '';
        $status_code = '';

        if ( ! is_page( SF()->get_setting( 'verification_page' ) ) || ! SF()->get_setting( 'new_account_verify_email' ) || empty( $_GET['activation_key'] ) ) {
            $status_code = 'invalid';
        }

        $users = get_users( array( 'meta_value' => $_GET['activation_key'] ) );
        $user_id = isset( $users[0]->ID ) ? $users[0]->ID : '';
        
        if ( '' == $user_id ) {
            $status_code = 'invalid';
        } else {
            $activation_key = get_user_meta( $user_id, 'sf_activation_key', true );
            $activation_status = get_user_meta( $user_id, 'sf_activation_status', true );

            if ( $activation_status != 0 ) {
                $status_code = 'verified';
            } elseif ( ! empty( $activation_key ) && $activation_key == $_GET['activation_key'] ) {
                $status_code = 'verified';
                update_user_meta( $user_id, 'sf_activation_status', 1 );
                $redirect_page_url = get_permalink( SF()->get_setting( 'verification_redirect_page' ) );

                $generate_password = SF()->get_setting( 'new_account_generate_password' );
                if ( $generate_password ) {
                    // set new password
                    $password = wp_generate_password( 12, false );
                    wp_set_password( $password, $user_id );
                } else {
                    $password = '';
                }
                
                $mail = new SF_Email();
                $mail->send_new_account_mail(
                    $user_id,
                    array(
                        'user_login' => $users[0]->user_login,
                        'user_pass'  => $password,
                        'user_email' => $users[0]->user_email,
                        'first_name' => $users[0]->first_name,
                        'last_name'  => $users[0]->last_name
                    ),
                    $generate_password
                );
                
                if ( SF()->get_setting( 'verification_auto_login' ) ) {
                    wp_set_current_user( $user_id, $users[0]->user_login );
                    
                    // add auto login query args
                    if ( ! empty( $redirect_page_url ) ) {
                        $redirect_page_url = add_query_arg( array( 'sf_autologin' => 'yes', 'key' => $activation_key ), $redirect_page_url );
                    }
                }

                if ( ! empty( $redirect_page_url ) ) {
                    $status_code = 'verified_and_redirect';
                    $html .= '<script>jQuery(document).ready(function(){setTimeout(function(){window.location.href = "'. $redirect_page_url .'";}, 3000);})</script>';
                }

            } else {
                $status_code = 'unverified';
            }
        }

        switch ( $status_code ) {
            case 'invalid': $html .= esc_html__( 'Invalid request', 'spirit' ); break;
            case 'verified': $html .= esc_html__( 'Your email has been verified.', 'spirit' ); break;
            case 'verified_and_redirect': $html .= esc_html__( 'Your email has been verified. Redirecting in 3 seconds..', 'spirit' ); break;
            default: $html .= esc_html__( 'Sorry we could not verify your email address.', 'spirit' ); break;
        }

        return $html;
    }

    /**
     * Auto login & reset password
     */
    function custom_actions() {
        $action = isset( $_REQUEST['action'] ) ? $_REQUEST['action'] : 'login';

        if ( 'sf_resetpass' === $action ) {
            // Reset page URL
            if ( ! empty( $reset_password_page = SF()->get_setting( 'reset_password_page' ) ) ) {
                $reset_pass_url = get_page_link( $reset_password_page );
            } else {
                $reset_pass_url = site_url( "wp-login.php" );
            }

            list( $rp_path ) = explode( '?', wp_unslash( $_SERVER['REQUEST_URI'] ) );
            $rp_cookie       = 'wp-resetpass-' . COOKIEHASH;

            if ( isset( $_GET['key'] ) ) {
                $value = sprintf( '%s:%s', wp_unslash( $_GET['login'] ), wp_unslash( $_GET['key'] ) );
                setcookie( $rp_cookie, $value, 0, $rp_path, COOKIE_DOMAIN, is_ssl(), true );
    
                wp_safe_redirect( remove_query_arg( array( 'key', 'login' ) ) );
                exit;
            }
    
            if ( isset( $_COOKIE[ $rp_cookie ] ) && 0 < strpos( $_COOKIE[ $rp_cookie ], ':' ) ) {
                list( $rp_login, $rp_key ) = explode( ':', wp_unslash( $_COOKIE[ $rp_cookie ] ), 2 );
    
                $user = check_password_reset_key( $rp_key, $rp_login );
    
                if ( isset( $_POST['sf-password'] ) && ! hash_equals( $rp_key, $_POST['rp_key'] ) ) {
                    $user = false;
                }
            } else {
                $user = false;
            }

            if ( ! $user || is_wp_error( $user ) ) {
                setcookie( $rp_cookie, ' ', time() - YEAR_IN_SECONDS, $rp_path, COOKIE_DOMAIN, is_ssl(), true );

                if ( $user && $user->get_error_code() === 'expired_key' ) {
                    wp_redirect( $reset_pass_url . '?action=sf_lostpass&error=expiredkey' );
                } else {
                    wp_redirect( $reset_pass_url . '?action=sf_lostpass&error=invalidkey' );
                }

                exit;
            }

            $errors = new WP_Error();
    
            /**
             * Fires before the password reset procedure is validated.
             *
             * @since 3.5.0
             *
             * @param WP_Error         $errors WP Error object.
             * @param WP_User|WP_Error $user   WP_User object if the login and reset key match. WP_Error object otherwise.
             */
            do_action( 'validate_password_reset', $errors, $user );
    
            if ( ( ! $errors->has_errors() ) && isset( $_POST['sf-password'] ) && ! empty( $_POST['sf-password'] ) ) {
                reset_password( $user, $_POST['sf-password'] );
                setcookie( $rp_cookie, ' ', time() - YEAR_IN_SECONDS, $rp_path, COOKIE_DOMAIN, is_ssl(), true );
                wp_redirect( $reset_pass_url . '?action=sf_login&status=success' );
                exit;
            }

        } else if ( isset( $_GET['sf_autologin'] ) && $_GET['sf_autologin']=='yes' && isset( $_GET['key'] ) ) {
            $activation_key = $_GET['key'];
            $users = get_users( array( 'meta_value' => $_GET['key'] ) );
            $user_id = isset( $users[0]->ID ) ? $users[0]->ID : '';
            
            if ( '' !== $user_id ) {
                $activation_status = get_user_meta( $user_id, 'sf_activation_status', true );
                if ( $activation_status != 0 ) {
                    wp_set_auth_cookie( $user_id );
                    do_action( 'wp_login', $users[0]->user_login, $users[0] );
                    wp_set_current_user( $user_id, $users[0]->user_login );
                }
            }
        }
    }

    public function reset_password_redirect_page( $redirect_url ) {
        global $post;
        if ( $post && $post->ID === SF()->get_setting( 'reset_password_page' ) ) {
            $redirect_url = get_site_url();
        }
        return $redirect_url;
    }

    /**
     * Add elementor widgets
     *
     * @param array $widgets
     * @return array
     */
    public function add_elementor_widgets( $widgets ) {
        $widgets[] = SF_FRAMEWORK_DIR . 'includes/elementor/widgets/login-form.php';
        $widgets[] = SF_FRAMEWORK_DIR . 'includes/elementor/widgets/registration-form.php';
        $widgets[] = SF_FRAMEWORK_DIR . 'includes/elementor/widgets/reset-password-form.php';
        return $widgets;
    }

    public function custom_registration_fields() {
        $custom_fields = SF()->get_setting( 'new_account_fields' );
        $custom_fields = apply_filters( 'sf_custom_registration_fields', $custom_fields );

        if ( empty( $custom_fields ) || ! is_array( $custom_fields ) ) {
            return;
        }

        $defaults = [
            'tel' => [
                'label' => esc_html__( 'Phone', 'spirit' ),
            ],
            'number' => [
                'label' => esc_html__( 'Number', 'spirit' ),
            ],
            'email' => [
                'label' => esc_html__( 'Email', 'spirit' ),
            ],
            'text' => [
                'label' => esc_html__( 'Text', 'spirit' ),
            ],
            'date' => [
                'label' => esc_html__( 'Date', 'spirit' ),
            ],
            'url' => [
                'label' => esc_html__( 'Website', 'spirit' ),
            ],
        ];

        foreach ( $custom_fields as $field ) {

            if ( empty( $field['id'] ) ) {
                continue;
            }

            $attributes= [
                'type' => 'text',
                'name' => '',
                'placeholder' => ''
            ];
            $input_attrs = $icon = '';
            switch ( $field['type'] ) {
                case 'text':
                case 'email':
                case 'number':
                case 'date':
                case 'tel':
                case 'url':
                    $attributes['placeholder'] = ! empty( $field['label'] ) ? $field['label'] : $defaults[$field['type']]['label'];
                    $attributes['type'] = $field['type'];
                    $attributes['name'] = $field['id'];
                    if ( ! empty( $field['required'] ) ) {
                        $attributes['class'] = 'validate';
                    }
                    $icon = ! empty( $field['icon'] ) ? '<i class="'. esc_attr( $field['icon'] ) .'"></i>' : '';
                    break;
            }
            $attributes = apply_filters( 'sf_custom_reg_field_'. $field['id'] .'_attributes', $attributes );
            foreach ( $attributes as $key => $value ) {
                $input_attrs .= " {$key}='{$value}'";
            }
            printf( '<div class="sf-form-row">%s<input %s></div>', $icon, $input_attrs );
        }
    }

    /**
     * Add custom fields to registration form
     */
    function save_custom_registration_fields( $user_id, $form_data ) {
        $custom_fields = SF()->get_setting( 'new_account_fields' );
        $custom_fields = apply_filters( 'sf_custom_registration_fields', $custom_fields );

        if ( empty( $custom_fields ) || ! is_array( $custom_fields ) ) {
            return;
        }
        
        foreach ( $custom_fields as $field ) {
            if ( ! empty( $field['id'] ) && isset( $form_data[ $field['id'] ] ) ) {
                update_user_meta( $user_id, $field['id'], sanitize_text_field( $form_data[ $field['id'] ] ) );
            }
        }
    }
}

/**
 * Login form messages
 * @return array translatable messages
 */
function sf_get_login_form_messages() {
    return apply_filters( 'sf_login_form_messages', array(
        'success' => __( 'Login successful, redirecting..', 'spirit' ),
        'invalid_username' => __( '<strong>ERROR</strong>: Invalid username.', 'spirit' ) .
                ' <a class="sf-lost-password" href="' . wp_lostpassword_url() . '">' .
                __( 'Lost your password?', 'spirit' ) .
                '</a>',
        'incorrect_password' => __( '<strong>ERROR</strong>: Incorrect password.', 'spirit' ) .
            ' <a class="sf-lost-password" href="' . wp_lostpassword_url() . '">' .
            __( 'Lost your password?', 'spirit' ) .
            '</a>',
        'captcha_failed' => __( 'Not a human!', 'spirit' )
    ) );
}

/**
 * New account form messages
 * @return array translatable messages
 */
function sf_get_new_account_form_messages() {
    return apply_filters( 'sf_new_account_form_messages', array(
        'success'              => __( 'Registration complete. Please check your email.', 'spirit' ),
        'success_verify_email' => __( 'Registration complete. Please verify your email address.', 'spirit' ),
        'no_registration'      => __( 'User registration is currently not allowed.', 'spirit' ),
        'username_exists'      => __( 'Sorry, that username already exists! ', 'spirit' ),
        'invalid_username'     => __( 'Please enter a valid username.', 'spirit' ),
        'existing_user_email'  => __( 'That email is already registered, please choose another one.', 'spirit' ),
        'invalid_email'        => __( 'Please enter a valid email address.', 'spirit' ),
        'send_mail_error'      => __( 'Registration complete. System is unable to send you an email.', 'spirit' ),
        'captcha_failed'       => __( 'Not a human!', 'spirit' )
    ) );
}

/**
 * Lost password form messages
 * @return array translatable messages
 */
function sf_get_lost_password_form_messages() {
    return apply_filters( 'sf_lost_password_form_messages', array(
        'success'             => __( 'Link for password reset has been emailed to you. Please check your email.', 'spirit' ),
        'no_password_reset'   => __( 'Password reset is not allowed for this user.', 'spirit' ),
        'user_not_registered' => __( 'There is no user registered with that email.', 'spirit' ),
        'send_mail_error'     => __( 'System is unable to send you an email for password reset.', 'spirit' ),
        'captcha_failed'      => __( 'Not a human!', 'spirit' )
    ) );
}


/**
 * Get new account agreement text
 */
function sf_get_new_account_agreement_text() {
    $text            = SF()->get_setting( 'new_account_agreement_text' );
    $terms_page_id   = SF()->get_setting( 'terms_page_id' );
    $privacy_page_id = SF()->get_setting( 'privacy_page_id' );
    $terms_link      = sf_get_post_title_link( $terms_page_id, array( 'class' => 'sf-agreement-link', 'target' => '_blank' ) );
    $privacy_link    = sf_get_post_title_link( $privacy_page_id, array( 'class' => 'sf-agreement-link', 'target' => '_blank' ) );
    
    $find_replace = array(
        '[terms]'          => $terms_link,
        '[privacy_policy]' => $privacy_link,
    );

    return str_replace( array_keys( $find_replace ), array_values( $find_replace ), $text );
}


/**
 * Get email verification link
 * @param  int    $user_id user id
 * @param  string $key     activation key
 * @return string          activation link
 */
function sf_get_email_verification_link( $user_id, $key = '' ) {
    $link = '';
    
    if ( empty( $key ) ) {
        $key = get_user_meta( $user_id, 'sf_activation_key', true );
    }

    if ( ! empty( $key ) ) {
        $verification_page = get_permalink( SF()->get_setting( 'verification_page' ) );
        
        if ( ! empty( $verification_page ) ) {
            $link = add_query_arg( array( 'activation_key' => $key ), $verification_page );
        }
    }

    return apply_filters( 'sf_email_verification_link', $link );
}

/**
 * Get reset password link
 * @param  string $key          reset key
 * @param  string $user_login   user login
 * @return string reset password link
 */
function sf_get_reset_password_link( $key = '', $user_login = '' ) {
    $link = '';

    // Generate custom url for reset password page or use default wordpress reset password link
    if ( ! empty( $reset_password_page = SF()->get_setting( 'reset_password_page' ) ) ) {
        $link = get_page_link( $reset_password_page );

        if ( ! empty( $key ) && ! empty( $user_login ) ) {
            $link = add_query_arg(
                array(
                    'action' => 'sf_resetpass',
                    'key'    => $key,
                    'login'  => rawurlencode( $user_login )
                ),
                $link
            );
        } else {
            $link = add_query_arg( array( 'action' => 'sf_resetpass' ), $link );
        }

    } else {
        
        if ( ! empty( $key ) && ! empty( $user_login ) ) {
            $link = network_site_url( "wp-login.php?action=rp&key=$key&login=" . rawurlencode( $user_login ), 'login' );
        } else {
            $link = network_site_url( 'wp-login.php?action=resetpass', 'login' );
        }
    }

    return apply_filters( 'sf_reset_password_link', $link );
}

/**
 * Print forget password message
 */
function sf_output_login_form_bottom_message() {
     ?>
    <div class="sf-bottom-message">
        <a href="#lostpassword" class="sf-lost-tab"><?php esc_html_e( 'Lost Password?', 'spirit' ); ?></a>
    </div>
    <?php
}

/**
 * Print login message
 */
function sf_output_register_form_bottom_message() {
    ?>
   <div class="sf-bottom-message">
       <?php esc_html_e( 'Already have an account?', 'spirit' ); ?>
       <a href="#login" class="sf-login-tab"><?php esc_html_e( 'Log In', 'spirit' ); ?></a>
   </div>
   <?php
}