<?php
/**
 * SF Elementor Addon
 * 
 * @package  Spirit_Framework
 */

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

class SF_Elementor_Addon {
	
	/**
	 * Static property to hold our singleton instance
	 */
	protected static $_instance = null;

	/**
	 * Main instance
	 * Ensures only one instance of this class is loaded or can be loaded.
	 *
	 * @static
	 * @return SF_Elementor_Addon - Main instance
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	function __construct() {
		require_once SF_FRAMEWORK_DIR . 'includes/elementor/functions.php';
		require_once SF_FRAMEWORK_DIR . 'includes/elementor/sections/animation.php';
		
		add_action( 'elementor/init', [ $this, 'add_elementor_category' ] );
		add_action( 'elementor/controls/register', [ $this, 'include_controls' ] );
		add_action( 'elementor/widgets/register', [ $this, 'include_widgets' ] );
		add_action( 'elementor/editor/before_enqueue_styles', [ $this, 'editor_before_styles' ] );
		add_action( 'elementor/editor/before_enqueue_scripts', [ $this, 'editor_before_scripts' ] );
		add_action( 'elementor/frontend/after_enqueue_styles', [ $this, 'frontend_after_styles' ] );
		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'frontend_after_scripts' ] );

		add_action( 'wp_ajax_sf_load_gallery_items', [ $this, 'load_gallery_items' ] );
		add_action( 'wp_ajax_nopriv_sf_load_gallery_items', [ $this, 'load_gallery_items' ] );
	}

	/**
	 * Before editor styles
	 */
	public function editor_before_styles() {
		wp_register_style( 'sf-editor', SF_FRAMEWORK_URI . 'assets/css/editor.min.css' );
		wp_enqueue_style( 'sf-editor' );
		wp_style_add_data( 'sf-editor', 'rtl', 'replace' );

		$default_icon_control = apply_filters( 'sf_elementor_default_icon_control', false );

		if ( ! $default_icon_control ) {
			wp_register_style( 'font-awesome-5-all', SF_FRAMEWORK_URI . 'assets/lib/font-awesome/css/all.min.css', [], '5.15.3' );
			wp_register_style( 'font-awesome-4-shim', SF_FRAMEWORK_URI . 'assets/lib/font-awesome/css/v4-shims.min.css', [], '5.15.3' );
			wp_register_style( 'fonticonpicker', SF_FRAMEWORK_URI . 'assets/lib/fonticonpicker/css/iconpicker.min.css' );
			wp_enqueue_style( 'font-awesome-5-all' );
			wp_enqueue_style( 'font-awesome-4-shim' );
			wp_enqueue_style( 'fonticonpicker' );
		}
		
		do_action( 'sf_elementor_editor_before_enqueue_styles' );
	}

	/**
	 * Before editor scripts
	 */
	public function editor_before_scripts() {
        wp_register_script( 'fonticonpicker', SF_FRAMEWORK_URI . 'assets/lib/fonticonpicker/js/jquery.fonticonpicker.min.js', [ 'jquery' ], SF_FRAMEWORK_VERSION, true );
        wp_register_script( 'sf-editor', SF_FRAMEWORK_URI . 'assets/js/admin/editor.min.js', [ 'jquery', 'fonticonpicker' ], SF_FRAMEWORK_VERSION, true );

		$default_icon_control = apply_filters( 'sf_elementor_default_icon_control', false );
		
		if ( ! $default_icon_control ) {
			wp_enqueue_script( 'fonticonpicker' );
			wp_enqueue_script( 'sf-editor' );
			wp_localize_script( 'sf-editor', 'SF_Editor', [ 'fontawesomeicons' => sf_get_font_icons() ] );
		}
	}

	/**
	 * After frontend styles
	 */
	public function frontend_after_styles() {
		wp_dequeue_style( 'font-awesome' );
		wp_enqueue_style( 'sf-frontend', SF_FRAMEWORK_URI .'assets/css/frontend.min.css', false, SF_FRAMEWORK_VERSION );
		wp_style_add_data( 'sf-frontend', 'rtl', 'replace' );
	}

	/**
	 * After frontend scripts
	 */
	public function frontend_after_scripts() {
		wp_enqueue_script( 'sf-frontend' );
		wp_localize_script( 'sf-frontend', 'sf_js_data', array( 'ajax_url' => admin_url( 'admin-ajax.php' ) ) );
	}

	/**
	 * Include custom controls
	 * @param  array $element elements array
	 */
	public function include_controls( $element ) {
		require_once SF_FRAMEWORK_DIR . 'includes/elementor/controls/icon.php';
		$element->register_control( 'sf_icon', new SF_Icon_Control );
	}
	
	/**
	 * Add element category
	 */
	public function add_elementor_category() {
        \Elementor\Plugin::instance()->elements_manager->add_category(
	        'sf-addons',
            array(
	            'title' => esc_attr__( 'Theme', 'spirit' ),
	            'icon' => 'far fa-plug'
            ),
	        0
    	);
	}

	/**
	 * Include custom widgets
	 * @param  object $widgets_manager
	 */
	public function include_widgets( $widgets_manager ) {
		$custom_widgets = [
			'slider-base',
			'accordion',
			'button',
			'buttons',
			'countdown',
			'content-block',
			'heading',
			'icon-box',
			'icon-boxes',
			'image-grid',
			'image-slider',
			'gallery',
			'number-counter',
			'price-table',
			'process',
			'progress-bars',
			'search-form',
			'team-members',
			'testimonials',
			'testimonials-carousel'
		];

		if ( class_exists( 'WooCommerce' ) ) {
			$custom_widgets[] = 'products';
		}

		$widgets = apply_filters( 'sf_elementor_registered_widgets', $custom_widgets );
		
		foreach ( $widgets as $widget ) {
			require_once SF_FRAMEWORK_DIR . 'includes/elementor/widgets/' . $widget . '.php';
		}

		$theme_widgets = apply_filters( 'sf_elementor_widgets', [] );
		
		foreach ( $theme_widgets as $widget ) {
			if ( file_exists( $widget ) ) {
				require_once $widget;
			}
		}
	}

	/**
	 * Load gallery items
	 */
	public function load_gallery_items() {
		if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'sf_gallery_nonce' ) ) {
            exit;
        }
		$page_number 	= isset( $_POST['page_number'] ) ? $_POST['page_number'] : 1;
		$settings 		= isset( $_POST['settings'] ) ? $_POST['settings'] : [];
		$items_per_page = ! empty( $settings['items_per_page'] ) ? $settings['items_per_page'] : 5;
		$items 			= ! empty( $settings['items'] ) ? $settings['items'] : [];
		$offset 		= ( $page_number - 1 ) * $items_per_page;
		$images 		= array_slice( $items, $offset, $items_per_page );

		sf_display_gallery_items( $images, $settings );

		exit;
	}
}